/*
 * Parsing Quake Logs
 *
 * Jorgen Lundman, (c) Cranberry Source 1997.
 *
 * lundman@cranberry.co.uk (Accident)
 *
 * This should have been written in perl, or similar, but I can't remember
 * perl and I left the book in NZ.
 *
 * We take no responsibility for anything etc.
 * If you want to use/modify this code, you need to keep our name in its 
 * credits.
 *
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <stdlib.h>


#define MAX_PLAYERS 32


static char buf[1024];

struct player {
  char *name;

  int id;

  int suicide;
  int killed;
  int was_killed;

  int killed_list[MAX_PLAYERS];
  int was_killed_list[MAX_PLAYERS];

  int tele;
  int was_tele;

  int tele_list[MAX_PLAYERS];
  int was_tele_list[MAX_PLAYERS];

  int start_time;
  int total_time;

  struct player *next;

} *player_head = NULL;


static int player_num = 0;

static last_i = 0;


char *mystrcpy(char *s)
{
  char *r;

  r = (char *) malloc(strlen(s) + 1);

  if (!r) {
    perror("Malloc2");
    exit(1);
  }

  strcpy(r, s);

  return r;
}



struct player *find_player(char *name)
{
  struct player *list;

  for (list = player_head; list; list = list->next) {
    if (!strcmp(name, list->name)) return list;
  }

  return NULL;
}


struct player *find_player_num(int id)
{
  struct player *list;

  for (list = player_head; list; list = list->next) {
    if (id == list->id) return list;
  }

  return NULL;
}



void add_player(char *name, int time)
{
  struct player *newd;


  if ((newd = find_player(name))) {
    newd->start_time = time;
    return;
  }

  if (player_num >= MAX_PLAYERS) {
    printf("Max players of %d reached. Increase MAX_PLAYERS and recompile\n",
	   MAX_PLAYERS);
    exit(1);
  }




  newd = (struct player *) malloc(sizeof(struct player));

  if (!newd) {
    perror("malloc");
    exit(1);
  }

  memset(newd, 0, sizeof(*newd));
  newd->name = mystrcpy(name);
  newd->id = player_num++;
  newd->start_time = time;

  newd->next = player_head;
  player_head = newd;


}


void print_player(struct player *list)
{
  int i;
  struct player *other;


  if (!list->killed && !list->was_killed) return;


  printf("*** %s ***\n", list->name);

  printf("Total Playing Time: %5d\nKills / Min:       %2.2f\nDeaths / Min:      %2.2f\n\n",
	 list->total_time, (float) list->killed * 60.0 / list->total_time ,
	 (float)list->was_killed * 60.0 / list->total_time);

  printf("Suicides: %3d    Killed: %3d            Deaths: %3d\n\n", 
	 list->suicide, list->killed, list->was_killed);

  printf("Player Name:     Killed Percentage:     Deaths-'by' Percentage:\n");


  for (i = 0; i < player_num; i++) {

    if (i == list->id) {
      continue;
    }

    other = find_player_num(i);
    if (!other) continue;

#ifdef CRANBERRY
    if ((unsigned char)other->name[0] == 0xD8) { /* hack for XiX */
      strcpy(other->name, "XiX");
    }
#endif

    printf("%-16s%3d%% (%3d)             %3d%% (%3d)\n", other->name,
	   list->killed ? list->killed_list[i] * 100 / list->killed : 0,
	   list->killed_list[i],
	   list->was_killed ? list->was_killed_list[i] * 100 / 
	   list->was_killed : 0,
	   list->was_killed_list[i]);

  }


  if (!list->tele && !list->was_tele) {
    printf("\n\n\n\n");
    return;
  }


  printf("\n                 TeleFrgd: %3d          WasTeleFrgd: %3d\n", 
	 list->tele, list->was_tele);

  printf("Player Name:     TeleFrgd Percentage:   WasTeleFrgd Percentage:\n");


  for (i = 0; i < player_num; i++) {

    if (i == list->id) {
      continue;
    }

    other = find_player_num(i);
    if (!other) continue;

#ifdef CRANBERRY
    if ((unsigned char)other->name[0] == 0xD8) { /* hack for XiX */
      strcpy(other->name, "XiX");
    }
#endif

    printf("%-16s%3d%% (%3d)             %3d%% (%3d)\n", other->name,
	   list->tele ? list->tele_list[i] * 100 / list->tele : 0,
	   list->tele_list[i],
	   list->was_tele ? list->was_tele_list[i] * 100 / list->was_tele : 0,
	   list->was_tele_list[i]);

  }



  printf("\n\n\n\n");

}




void list_players()
{
  struct player *list;

  for (list = player_head; list; list=list->next) {

    print_player(list);

  }
}


void killed_player(char *x, char *y)
{
  struct player *xx, *yy;

  xx = find_player(x);
  yy = find_player(y);

  if (!xx || !yy) {
    printf("Ahh? couldn't find a player '%s' '%s' %s\n", x, y, buf);
    return;
  }

  xx->killed++;
  yy->was_killed++;

  xx->killed_list[yy->id]++;
  yy->was_killed_list[xx->id]++;

#if 0
  printf("%s killed %s: %d %d %d %d\n",
	 xx->name, yy->name, xx->killed, xx->was_killed,
	 xx->killed_list[yy->id], yy->was_killed_list[xx->id]);
#endif
}


void tele_player(char *x, char *y)
{
  struct player *xx, *yy;

  xx = find_player(x);
  yy = find_player(y);

  if (!xx || !yy) {
    printf("Ahh? couldn't find a player\n");
    return;
  }

  xx->tele++;
  yy->was_tele++;

  xx->tele_list[yy->id]++;
  yy->was_tele_list[xx->id]++;

}


void suicide(char *name)
{
  struct player *xx;

  xx = find_player(name);
  if (!xx) return;

  xx->suicide++;
}


void end_level(int time)
{
  struct player *list;

  for (list = player_head; list; list = list->next) {

    if (!list->start_time) continue; /* Not playing now */

    list->total_time += time - list->start_time;
    list->start_time = 0;
  }

}


void left_game(char *x, int time)
{
  struct player *xx;

  if (!(xx = find_player(x))) return;

  if (!xx->start_time) {
    printf("This shouldn't happen\n");
    return;
  }

  xx->total_time += time - xx->start_time;
  xx->start_time = 0;
}



int recognise_string(char *buf)
{
  /* Does it start with dddd.d ? */
  int i,j;
  char *s, firstname[100], secondname[100];
  

  /*  if (*buf == ' ') return 0; /* Skip line if it starts with space */

  if (sscanf(buf, "%d.%d", &i, &j) != 2) {
    if (strstr(buf, "Starting map")) {
      
      /* Whole game, add everyone up. */
      if (!last_i) return 0;
      
      end_level(last_i);
    }
    return 0; /* not a game line */
  }

  if ((s = strchr(buf, '\n'))) *s = (char)0;
  if ((s = strchr(buf, '\r'))) *s = (char)0;



  /* Skip string until first space */
  s = buf;
  while(*s == ' ') s++;
  s = (char *)strchr(s, ' ');

  if (!s) {
    printf("What?\n");
    return 0; /* should never happen */
  }

  

  if (strstr(buf, "entered the game")) {
    /* entered the game */
    
    sscanf(s, "%s", firstname);

    add_player(firstname, i);


  } else if (strstr(buf, "rides a rocket from") ||            /* rocket */
	     strstr(buf, "takes the shaft from") ||           /* lightning*/
	     strstr(buf, "eats pineapple pie from") ||        /* grenade */
	     strstr(buf, "chews on boomstick from") ||    /* shotgun */
	     strstr(buf, "eats 2 loads of buckshot from") ||  /* double shot*/
	     strstr(buf, "gets punctured by") ||              /* axe */
	     strstr(buf, "gets nailed by") ||                 /* nail gun */
	     strstr(buf, "is ax-murdered by") ||              /* supr nail*/
	     strstr(buf, "accepts a discharge from")) {
    
    /* XXX killed by YYY */
    sscanf(s, "%s", firstname);

    s = strrchr(buf, ' ');
    if (!s) return 0;
    
    sscanf(s, "%s", secondname);
    
    killed_player(secondname, firstname);


  } else if (strstr(buf, "should stand further back from explosions") ||
	     strstr(buf, "tries to put the pin back in") ||
	     strstr(buf, "discharges into the water") ||
	     strstr(buf, "fell to his death") ||
	     strstr(buf, "was spiked") ||
	     strstr(buf, "was squished") ||
	     strstr(buf, " died") ||
	     strstr(buf, "turned into hot slag") ||
  	     strstr(buf, "gulped a load of slime") ||
	     strstr(buf, "sleeps with the fishes") ||
	     strstr(buf, "tried to leave") ||
	     strstr(buf, "ate a lavaball") ||
	     strstr(buf, "blew up") ||
	     strstr(buf, "becomes bored with life")) {

    sscanf(s, "%s", firstname);
    suicide(firstname);


  } else if (strstr(buf, "was telefragged by")) {

    /* XXX tele by YYY */
    sscanf(s, "%s", firstname);

    s = strrchr(buf, ' ');
    if (!s) return 0;
    
    sscanf(s, "%s", secondname);
    
    tele_player(secondname, firstname);


  } else if (strstr(buf, "left the game with")) {

    sscanf(s, "%s", firstname);

    left_game(firstname, i);


  } else return 0;

  last_i = i;

  return 1;
  
}





void parse_file(FILE *fd)
{
  
  /* While we can read a line, parse it */
  while(fgets(buf, 1024, fd)) {

    
    recognise_string(buf) ;



  }

  end_level(last_i);


}



void print_stats()
{

  list_players();

}


int main(int argc, char **argv)
{
  /* Parse all files in argv list, if none, do stdin... */
  int i;
  FILE *fd;

  printf("Quake Console Log Parser.\n(c) Cranberry Source.\nlundman@cranberry.co.uk.\n");


  for (i = 1; i < argc; i++) {

    fd = fopen(argv[i], "r");

    if (!fd) {
      fprintf(stderr, "Couldn't open file %s (skipping): ", argv[i]);
      perror("");
      continue;
    }

    parse_file(fd);
    fclose(fd);

  }

  if (argc == 1) { /* no file, stdin */

    printf("Reading from stdin...\n");
    parse_file(stdin);

  }

  printf("\n");

  print_stats(); /* This one... */


  return 0;
}
