#!/usr/local/bin/perl
##
## statssum.pl
## Perl script to summarize QstatList statistical information.
##
## David G. Hesprich (Dark Grue)
## darkgrue@iname.com
## Last Revision: February 22, 1999
##
## QstatList is Copyright (c) 1999 David G. Hesprich (Dark Grue).
##
## This program is free software; you can redistribute it and/or modify it
## under the terms of the GNU General Public License as published by the
## Free Software Foundation; either version 2 of the License, or (at your
## option) any later version.
##
## This program is distributed in the hope that it will be useful, but
## WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
## General Public License for more details.
##
## You should have received a copy of the GNU General Public License along
## with this program; if not, write to the Free Software Foundation, Inc.,
## 675 Mass Ave, Cambridge, MA 02139, USA.
##
## ============================================================================
##
## README
##
## This assumes that the input log file is in ascending-by-date order.
## Once the input stream date stamp has met or exceeded the current day of
## the month, the summary data is written out to $LIST_STATS_NEW, and any
## subsequent back-dated entries are discarded. Lines with dates in the present
## or future are written directly out to $LIST_STATS_NEW.
##
## At the end of the process, $LIST_STATS is moved to $LIST_STATS_OLD, and
## $LIST_STATS_NEW is moved to $LIST_STATS.
##
## Month summary lines are in the format mm/01/yyyy 00:00:01 (first second of
## the first day of the month).
## Day summary lines are in the format mm/dd/yyyy 00:00:01 (first second of the
## day).
##

## Location of QstatList base install (qstatlist and qstatlist.conf).
##
$QSTATLIST_BASEDIR = '/usr/local/etc/qstatlist';


## Includes
##
use Getopt::Std;
use File::Copy;

require("$QSTATLIST_BASEDIR/qstatlist.conf");


## Define Variables
## 
$LIST_STATS_NEW = $TMPDIR . '/cullstats.qstatlist';
$LIST_STATS_OLD = $LIST_STATS . '.bak';

$DEBUG = 0;


## ============================================================================
## ============================================================================
## 
## SUBROUTINES
##

# Subroutine: Write out accumulated summary statistics.
sub write_summary {
  print "  Writing summary to file...\n" if ($DEBUG);			# debug info
  foreach my $month (sort({ $a <=> $b } keys(%smonth))) {
    print(sfh2 sprintf("%02d/%02d/%04d %02d:%02d:%02d", $month, 1, $nowyear, 0, 0, 1));
    foreach (sort(keys(%{$smonth{$month}}))) {
      # <servertype> <servers>, <actservers>, <maxplayers>, <players>
      print(sfh2 " $_ ");
      print(sfh2 int($smonth{$month}{$_}{'servers'} / $smonth{$month}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($smonth{$month}{$_}{'actservers'} / $smonth{$month}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($smonth{$month}{$_}{'players'} / $smonth{$month}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($smonth{$month}{$_}{'actplayers'} / $smonth{$month}{$_}{'new'}));
    }
    print(sfh2 "\n");
  }

  foreach my $day (sort({ $a <=> $b } keys(%sday))) {
    print(sfh2 sprintf("%02d/%02d/%04d %02d:%02d:%02d", $nowmonth, $day, $nowyear, 0, 0, 1));
    foreach (sort(keys(%{$sday{$day}}))) {
      # <servertype> <servers>, <actservers>, <maxplayers>, <players>
      print(sfh2 " $_ ");
      print(sfh2 int($sday{$day}{$_}{'servers'} / $sday{$day}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($sday{$day}{$_}{'actservers'} / $sday{$day}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($sday{$day}{$_}{'players'} / $sday{$day}{$_}{'new'}));
      print(sfh2 ", ");
      print(sfh2 int($sday{$day}{$_}{'actplayers'} / $sday{$day}{$_}{'new'}));
    }
    print(sfh2 "\n");
  }
  print "  Done.\n" if ($DEBUG);					# debug info
}


## ============================================================================
## ============================================================================
##
## MAIN
##

# Process command-line options.
unless (getopts('d')) {
  print <<End_of_Here;

Usage: statssum.pl [switches]
  -d			set debugging flags (positive integer)
End_of_Here

  exit;
}
$DEBUG = $opt_d if (defined($opt_d));

# Get and format the current time as a baseline reference.
local($nowseconds, $nowminutes, $nowhour, $nowday, $nowmonth, $nowyear) = localtime();
$nowmonth++;
$nowyear += 1900;

my($month, $day, $year, $hour, $minutes, $seconds);

# Look 82800 seconds (= 23 hours) into the past.
my($pastseconds, $pastminutes, $pasthour, $pastday, $pastmonth, $pastyear) = localtime($Time - 82800);
my($pastindex) = sprintf("%04d%02d%02d%02d%02d%02d", $pastyear + 1900, $pastmonth + 1, $pastday, $pasthour, $pastminutes, $pastseconds);

print "\n<-- Stats summarization starting. -->\n" if ($DEBUG);		# debug info
print "\nGetting $LIST_STATS for read operation...\n" if ($DEBUG);	# debug info
open(sfh, "<$LIST_STATS") || die("Can't read from file '$LIST_STATS': $!\n");
flock(sfh, $LOCK_SH);
print "Got exclusive lock.\n" if ($DEBUG);				# debug info

print "\nGetting $LIST_STATS_NEW for write operation...\n" if ($DEBUG);	# debug info
open(sfh2, ">$LIST_STATS_NEW") || die("Can't append to file '$LIST_STATS_NEW': $!\n");
flock(sfh2, $LOCK_EX);
print "Got exclusive lock.\n" if ($DEBUG);				# debug info

print "\nReading from file...\n" if ($DEBUG);				# debug info
while (<sfh>) {
  # Skip blank lines.
  next if (/^\n$/);

  # Remove trailing <CR> character.
  chop();

  # Make sure we don't get tainted info!
  next unless (($month, $day, $year, $hour, $minutes, $seconds) = /^(\d{2})\/(\d{2})\/(\d{4})\s(\d{2}):(\d{2}):(\d{2})\s/);

  # Since statistics graphing does not roll over, do only the current year.
  next if ($year != $nowyear);

  my($index) = join('', ($year, $month, $day, $hour, $minutes, $seconds));

  # Trim leading zero.
  $month =~ s/^0//;
  $day =~ s/^0//;

  # This assumes that the input log file is in ascending-by-date order.
  #   Consequently, once all the data that can be summarized has been read, the
  #   data which follows is immediately written into the output file.
#  if ((($month == $nowmonth) && ($day >= $nowday)) || ($month > $nowmonth)) {
  if ($index > $pastindex) {
    if (!$passthrough) {
      print "  Paused after $linecount lines.\n" if ($DEBUG);		# debug info
      write_summary();
      print "  Continuing from file...\n" if ($DEBUG);			# debug info
      $passthrough = 1;
    }

    print(sfh2 "$_\n");
  }

  $linecount++;

  while (/(?:($ST_PAT)\s(\d*),\s(\d*),\s(\d*),\s(\d*)\s*)/g) {
    # Average over the month.
    if ($month < $nowmonth) {
      $smonth{$month}{$1}{'new'}++;
      $smonth{$month}{$1}{'servers'} += $2;
      $smonth{$month}{$1}{'actservers'} += $3;
      $smonth{$month}{$1}{'players'} += $4;
      $smonth{$month}{$1}{'actplayers'} += $5;

      next;
    }

    # Average over the day.
    if ($day < $nowday) {
      $sday{$day}{$1}{'new'}++;
      $sday{$day}{$1}{'servers'} += $2;
      $sday{$day}{$1}{'actservers'} += $3;
      $sday{$day}{$1}{'players'} += $4;
      $sday{$day}{$1}{'actplayers'} += $5;

      next;
    }
  }
}

# File ended before summarizable data did.
if (!$passthrough) {
  write_summary();
  $passthrough = 1;
}

if ($DEBUG) {								# debug info
  print "  EOF after $linecount lines.\n";				# debug info
  print "Done.\n";							# debug info
}									# debug info

close(sfh);
flock(sfh, $LOCK_UN);
print "\nUnlocked input statistics log.\n" if ($DEBUG);			# debug info

close(sfh2);
flock(sfh2, $LOCK_UN);
print "\nUnlocked output statistics log.\n" if ($DEBUG);		# debug info
									# debug info
print "\n<-- Stats summarization finished. -->\n" if ($DEBUG);		# debug info

print "\nCopying $LIST_STATS to $LIST_STATS_OLD..." if ($DEBUG);	# debug info
copy("$LIST_STATS", "$LIST_STATS_OLD");
print "Done.\n" if ($DEBUG);						# debug info

print "\nMoving $LIST_STATS_NEW to $LIST_STATS..." if ($DEBUG);		# debug info
move("$LIST_STATS_NEW", "$LIST_STATS");
print "Done.\n" if ($DEBUG);						# debug info

print "\nFinished.\n" if ($DEBUG);					# debug info
exit(0);
