/*
 *  GoodBot - autonomous client side Quake2 robot
 *  Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include<stdio.h>
#include<stdlib.h>

#ifdef HAVE_STRING_H
#include<string.h>
#endif

#ifdef HAVE_STRINGS_H
#include<strings.h>
#endif

#ifdef HAVE_LIMITS_H
#include<limits.h>
#endif

#include"net.h"
#include"misc.h"
#include"buffer.h"
#include"msgdefs.h"
#include"checksum.h"
#include"protocol.h"

static void ping_start(protocol *state)
{
	state->pinglast = gettime();
}

static void ping_end(protocol *state)
{
	if(state->pinglast == 0.0) return;

	state->shared->ping = gettime() - state->pinglast;

	state->pinglast = 0.0;
}

static void filter_send_console_command(protocol *state,char *cmd)
{
	if(state->level == IN_GAME)
		protocol_send_console_command(state,cmd);
}

protocol *protocol_init(char *name,unsigned long rate,
			char *host,unsigned short port,shared_data *shared)
{
	protocol *state;
	unsigned c;

	state = xcalloc(1,sizeof(protocol));

	state->shared = shared;
	state->shared->scc_arg = (void*)state;
	state->shared->scc
		 = (void (*)(void*,char*))filter_send_console_command;

	state->host = xstrdup(host);
	state->addr = NULL;
	state->port = port;
	state->name = xstrdup(name);
	state->rate = rate;

	state->sock = sock_init(state->host,state->port,&(state->addr));

	state->need_ack = NULL;

	state->pinglast = 0.0;
	state->shared->ping = 0.0;

	state->need_ack_last = 0.0;

	state->level = PRE_CHALLENGE;

	state->challenge = NULL;
	state->qport = rand() % 65536;

	state->seq = 1;
	state->last_seq = 0;

	state->frame_num = 0xffffffff;

	for(c=0;c<16;c++)
		state->oldframes[c]=(edict*)calloc(MAX_EDICTS,sizeof(edict));

	return state;
}

void protocol_free(protocol *state)
{
	unsigned c;

	for(c=0;c<16;c++) free(state->oldframes[c]);

	sock_free(state->sock);

	free(state->challenge);
	free(state->host);
	free(state->addr);
	free(state->name);

	free(state);
}

void protocol_connect(protocol *state)
{
	while(state->level < POST_SYN) protocol_update(state);
}

void protocol_disconnect(protocol *state)
{
	if(state->level < POST_GAME) state->level = POST_GAME;

	while(state->level != GAME_OVER) protocol_update(state);
}

static void send_getchallenge(protocol *state)
{
	buffer *buf;

	printf("Connecting to %s:%hu...\n",state->host,state->port);

	buf = newbuffer();

	putLEu32(buf,0xffffffff);
	putnstr(buf,"getchallenge\n");

	sock_send(state->sock,buf);

	freebuffer(buf);

	ping_start(state);
}

static void send_connect(protocol *state)
{
	buffer *buf;
	char tmp[sizeof(long) * CHAR_BIT + 2];

	buf = newbuffer();

	putLEu32(buf,0xffffffff);
	putnstr(buf,"connect ");

	sprintf(tmp,"%lu",(unsigned long)PROTOCOL_VERSION);
	putnstr(buf,tmp);
	putnstr(buf," ");

	sprintf(tmp,"%hu",state->qport);
	putnstr(buf,tmp);
	putnstr(buf," ");

	putnstr(buf,state->challenge);
	putnstr(buf," ");

	putnstr(buf,"\"\\fov\\90\\rate\\");
	sprintf(tmp,"%lu",state->rate);
	putnstr(buf,tmp);
	putnstr(buf,"\\msg\\0\\skin\\cyborg/oni911\\name\\");
	putnstr(buf,state->name);
	putnstr(buf,"\\hand\\2\"\n");

	sock_send(state->sock,buf);

	freebuffer(buf);

	ping_start(state);
}

static void parse_connectionless(protocol *state,buffer *buf)
{
	unsigned long oldpos;
	char *cmd_str;

	oldpos = bufgetgpos(buf);
	cmd_str = getstr(buf);

	if(!strncmp(cmd_str,"challenge ",10))
	{
		printf("%s:%hu: challenge\n",state->addr,state->port);

		free(state->challenge);
		bufsetgpos(buf,oldpos + 10);
		state->challenge = getstr(buf);

		state->level = PRE_CONNECT;
	}
	else if(!strncmp(cmd_str,"client_connect",14))
	{
		printf("%s:%hu: client_connect\n",state->addr,state->port);
		state->level = POST_SYN;
	}
	else if(!strncmp(cmd_str,"print",5))
	{
		printf("%s:%hu: %s",state->addr,state->port,cmd_str);
		state->level = PRE_CHALLENGE;
	}
	else printf("%s:%hu: unknown packet\n",state->addr,state->port);

	free(cmd_str);

	ping_end(state);
}

static void send_reliable(protocol *state,buffer *buf)
{
	packet *need_ack,**next;

	need_ack = xmalloc(sizeof(packet));

	need_ack->seq = 0;
	need_ack->buf = buf;
	need_ack->next = NULL;

	next = &(state->need_ack);

	while(*next) next = &((*next)->next);

	(*next) = need_ack;
}

static vector cap_angles(vector ang)
{
	while(ang.x > Q_PI) ang.x -= 2.0 * Q_PI;
	while(ang.y > Q_PI) ang.y -= 2.0 * Q_PI;
	while(ang.z > Q_PI) ang.z -= 2.0 * Q_PI;

	while(ang.x < -Q_PI) ang.x += 2.0 * Q_PI;
	while(ang.y < -Q_PI) ang.y += 2.0 * Q_PI;
	while(ang.z < -Q_PI) ang.z += 2.0 * Q_PI;

	return ang;
}

void protocol_send_player_update(protocol *state,usercmd_t update)
{
	buffer *buf;
	usercmd_t *last,*uct,zero;
	unsigned long checksum_off;
	unsigned char mask;
	int c;

	if(state->level != IN_GAME) return;

	update.angles = vec_mul(update.angles,Q_PI/M_PI);
	update.angles = vec_sub(update.angles,state->delta_angles);
	update.angles = cap_angles(update.angles);

	memset(&zero,0,sizeof(zero));

	state->last_updates[0] = state->last_updates[1];
	state->last_updates[1] = state->last_updates[2];
	state->last_updates[2] = update;

	buf = newbuffer();

	putLEu32(buf,state->seq);
	putLEu32(buf,state->last_seq);
	putLEu16(buf,state->qport);
	putu8(buf,CLIENT_PLAYER_UPDATE);

	checksum_off = bufgetppos(buf);
	putu8(buf,0);

	putLEu32(buf,state->frame_num);

	uct = state->last_updates;

	for(c=0;c<3;c++)
	{
		last = (c) ? (uct + c - 1) : &zero;

		mask = 0;

		if((short)uct[c].angles.x != (short)last->angles.x) mask|=0x01;
		if((short)uct[c].angles.y != (short)last->angles.y) mask|=0x02;
		if((short)uct[c].angles.z != (short)last->angles.z) mask|=0x04;

		if((short)uct[c].move.x != (short)last->move.x) mask|=0x08;
		if((short)uct[c].move.y != (short)last->move.y) mask|=0x10;
		if((short)uct[c].move.z != (short)last->move.z) mask|=0x20;

		if(uct[c].buttons != last->buttons) mask |= 0x40;

		putu8(buf,mask);

		if(mask & 0x01) putLEs16(buf,uct[c].angles.x);
		if(mask & 0x02) putLEs16(buf,uct[c].angles.y);
		if(mask & 0x04) putLEs16(buf,uct[c].angles.z);

		if(mask & 0x08) putLEs16(buf,uct[c].move.x);
		if(mask & 0x10) putLEs16(buf,uct[c].move.y);
		if(mask & 0x20) putLEs16(buf,uct[c].move.z);

		if(mask & 0x40) putu8(buf,uct[c].buttons);

		putu8(buf,0xff);
		putu8(buf,0xff);
	}

	bufsetppos(buf,checksum_off);
	putu8(buf,checksum(getdata(buf) + checksum_off + 1,
		getsize(buf) - checksum_off - 1,state->seq));

	sock_send(state->sock,buf);

	freebuffer(buf);

	state->seq++;
}

void protocol_send_user_info(protocol *state,char *str)
{
	buffer *buf;

	buf = newbuffer();
	putLEu32(buf,0);
	putLEu32(buf,0);
	putLEu16(buf,state->qport);
	putu8(buf,CLIENT_USER_INFO);
	putstr(buf,str);

	send_reliable(state,buf);
}

static void proc_stufftext_command(protocol *state,char *cmd)
{
	strip_head_space(&cmd);
	if(!strncmp(cmd,"reconnect\n",10)) state->level = POST_SYN;
}

static void proc_user_console_command(protocol *state,char *cmd)
{
	strip_head_space(&cmd);
	if(!strncmp(cmd,"disconnect\n",11)) state->level = PRE_GAME_OVER;
}

void protocol_send_console_command(protocol *state,char *str)
{
	buffer *buf;

	buf = newbuffer();
	putLEu32(buf,0);
	putLEu32(buf,0);
	putLEu16(buf,state->qport);
	putu8(buf,CLIENT_CONSOLE_COMMAND);
	putstr(buf,str);

	send_reliable(state,buf);

	proc_user_console_command(state,str);
}

static void decode_console_command(protocol *state,char *str)
{
	char *multi = xstrdup(str);
	char *cmd,*tmp;

	tmp = multi;

	while(tmp)
	{
		cmd = tmp;
		tmp = strchr(tmp,';');
		if(tmp)
		{
			*tmp = '\0';
			tmp++;
		}

		strip_head_space(&cmd);

		if(!strncmp(cmd,"cmd ",4))
		{
			cmd += 4;
			protocol_send_console_command(state,cmd);
			if(!strncmp(cmd,"begin",5)) state->level = IN_GAME;
		}
		else proc_stufftext_command(state,cmd);
	}

	free(multi);
}

static void proc_ack(protocol *state,unsigned long seq)
{
	packet *next;

	if(state->need_ack == NULL) return;

	if( ((state->need_ack->seq | HIDDEN_BIT) == (seq | HIDDEN_BIT))
	 || ((seq & HIDDEN_BIT) && (seq > (state->need_ack->seq|HIDDEN_BIT))) )
	{
		freebuffer(state->need_ack->buf);
		next = state->need_ack->next;
		free(state->need_ack);
		state->need_ack = next;

		ping_end(state);
	}
}

static float readangle(buffer *buf)
{
	return (2.0 * M_PI * (float)gets8(buf)) / 256.0;
}

static float readcoord(buffer *buf)
{
	return 0.125 * (float)getLEs16(buf);
}

static vector readvector(buffer *buf)
{
	vector v;

	v.x = readcoord(buf);
	v.y = readcoord(buf);
	v.z = readcoord(buf);

	return v;
}

static void parse_bad(protocol *state,buffer *buf)
{
	(void)state;
	(void)buf;

	fprintf(stderr,"CL_ParseServerMessage: Illegible server message\n");
	exit(7);
}
static void parse_muzzle_flash(protocol *state,buffer *buf)
{
	(void)state;
	(void)getLEs16(buf);
	(void)getu8(buf);
}
static void parse_muzzle_flash2(protocol *state,buffer *buf)
{
	(void)state;
	(void)getLEs16(buf);
	(void)getu8(buf);
}
static void parse_temp_entity(protocol *state,buffer *buf)
{
	(void)state;

	switch(getu8(buf))
	{
		case TE_EXPLOSION1:
		case TE_EXPLOSION2:
		case TE_ROCKET_EXPLOSION:
		case TE_GRENADE_EXPLOSION:
		case TE_ROCKET_EXPLOSION_WATER:
		case TE_GRENADE_EXPLOSION_WATER:
		case TE_BFG_EXPLOSION:
		case TE_BFG_BIGEXPLOSION:
		case TE_BOSSTPORT:
			(void)readvector(buf);
			break;

		case TE_GUNSHOT:
		case TE_BLOOD:
		case TE_BLASTER:
		case TE_SHOTGUN:
		case TE_SPARKS:
		case TE_SCREEN_SPARKS:
		case TE_SHIELD_SPARKS:
		case TE_BULLET_SPARKS:
		case TE_GREENBLOOD:
			(void)readvector(buf);
			(void)getu8(buf);
			break;

		case TE_RAILTRAIL:
		case TE_BUBBLETRAIL:
		case TE_BFG_LASER:
		case TE_PLASMATRAIL:
			(void)readvector(buf);
			(void)readvector(buf);
			break;

		case TE_SPLASH:
			(void)getu8(buf);
			(void)readvector(buf);
			(void)getu8(buf);
			(void)getu8(buf);
			break;

		case TE_LASER_SPARKS:
		case TE_WELDING_SPARKS:
			(void)getu8(buf);
			(void)readvector(buf);
			(void)getu8(buf);
			(void)getu8(buf);
			break;

		case TE_PARASITE_ATTACK:
		case TE_MEDIC_CABLE_ATTACK:
			(void)getLEs16(buf);
			(void)readvector(buf);
			(void)readvector(buf);
			break;

		case TE_GRAPPLE_CABLE:
			(void)getLEs16(buf);
			(void)readvector(buf);
			(void)readvector(buf);
			(void)readvector(buf);
			break;

		default:
			fprintf(stderr,"CL_ParseTEnt: bad type\n");
			exit(7);
			break;
	}
}
static void parse_layout(protocol *state,buffer *buf)
{
	(void)state;

	free(getstr(buf));
}
static void parse_inventory(protocol *state,buffer *buf)
{
	unsigned c;

	for(c=0;c<MAX_ITEMS;c++) (void)getLEs16(buf);
}
static void parse_nop(protocol *state,buffer *buf)
{
	(void)state;
	(void)buf;
}
static void parse_disconnect(protocol *state,buffer *buf)
{
	(void)buf;

	state->level = GAME_OVER;
	printf("Server disconnected\n");
}
static void parse_reconnect(protocol *state,buffer *buf)
{
	(void)buf;

	printf("Server: reconnect\n");
	state->level = PRE_CHALLENGE;
}
static void parse_sound(protocol *state,buffer *buf)
{
	unsigned char mask;

	(void)state;

	mask = getu8(buf);
	(void)getu8(buf);
	if(mask & 0x01) (void)getu8(buf);
	if(mask & 0x02) (void)getu8(buf);
	if(mask & 0x10) (void)getu8(buf);
	if(mask & 0x08) (void)getLEs16(buf);
	if(mask & 0x04) (void)readvector(buf);
}
static void parse_print(protocol *state,buffer *buf)
{
	char *str;

	(void)state;

	(void)getu8(buf);

	str = getstr(buf);

	printf("%s",str);
	if((strlen(str)) && (str[strlen(str) - 1] != '\n')) printf("\n");

	if(stristr(str,"no bots")) protocol_disconnect(state);

	free(str);
}
static void parse_stuff_text(protocol *state,buffer *buf)
{
	char *str;

	str = getstr(buf);
	decode_console_command(state,str);
	free(str);
}
static void parse_server_data(protocol *state,buffer *buf)
{
	unsigned long version;
	char *str;

	version = getLEu32(buf);

	if(version != (unsigned long)PROTOCOL_VERSION)
	{
		fprintf(stderr,"Supported Protocol version %lu\n"
			"Server Protocol version %lu\n",
			(unsigned long)PROTOCOL_VERSION,version);
		exit(7);
	}

	(void)getLEu32(buf);
	(void)getu8(buf);
	free(getstr(buf));

	(void)getLEu16(buf);
	str = getstr(buf);
	printf("%s\n",str);
	free(str);
}

static void set_map_name(protocol *state,char *mapname)
{
	state->shared->smn(state->shared->smn_arg,mapname);
}

static void parse_config_string(protocol *state,buffer *buf)
{
	unsigned short index;
	char *str;

	index = getLEu16(buf);
	if(index > MAX_CONFIGSTRINGS)
	{
		fprintf(stderr,"configstring > MAX_CONFIGSTRINGS\n");
		exit(7);
	}

	str = getstr(buf);
	free(state->shared->configs[index]);
	state->shared->configs[index] = str;

	if(index == (CS_MODELS+1)) set_map_name(state,str);
}
static void parse_spawn_baseline(protocol *state,buffer *buf)
{
	unsigned long mask,effects,renderfx;
	unsigned entity,sound,event,solid;
	unsigned modelindex2,modelindex3,modelindex4;
	vector old_origin;
	edict *ent;

	mask = getu8(buf);
	if(mask & 0x00000080) mask |= (getu8(buf) << 8);
	if(mask & 0x00008000) mask |= (getu8(buf) << 16);
	if(mask & 0x00800000) mask |= (getu8(buf) << 24);
	entity = (mask & 0x00000100) ? getLEu16(buf) : getu8(buf);
	if(entity >= MAX_EDICTS)
	{
		fprintf(stderr,"CL_ParseSpawnBaseLine:"
				" bad number: %u",entity);
		exit(7);
	}

	ent = state->baseline + entity;
	ent->valid = 1;

	if(mask & 0x00000800) ent->modelindex = getu8(buf);
	if(mask & 0x00100000) modelindex2 = getu8(buf);
	if(mask & 0x00200000) modelindex3 = getu8(buf);
	if(mask & 0x00400000) modelindex4 = getu8(buf);
	if(mask & 0x00000010) ent->mframe = getu8(buf);
	if(mask & 0x00020000) ent->mframe = getLEu16(buf);
	if(mask & 0x00010000)
	{
		ent->skinnum = (mask&0x02000000)?getLEu32(buf):getu8(buf);
	}
	else
	{
		if(mask & 0x02000000) ent->skinnum = getLEu16(buf);
	}
	if(mask & 0x00004000)
	{
		effects = (mask & 0x00080000) ? getLEu32(buf) : getu8(buf);
	}
	else
	{
		if(mask & 0x00080000) effects = getLEu16(buf);
	}
	if(mask & 0x00001000)
	{
		renderfx = (mask & 0x00040000) ? getLEu32(buf) : getu8(buf);
	}
	else
	{
		if(mask & 0x00040000) renderfx = getLEu16(buf);
	}
	if(mask & 0x00000001) ent->origin.x = readcoord(buf);
	if(mask & 0x00000002) ent->origin.y = readcoord(buf);
	if(mask & 0x00000200) ent->origin.z = readcoord(buf);
	if(mask & 0x00000400) ent->angles.x = readangle(buf);
	if(mask & 0x00000004) ent->angles.y = readangle(buf);
	if(mask & 0x00000008) ent->angles.z = readangle(buf);
	if(mask & 0x01000000) old_origin = readvector(buf);
	if(mask & 0x04000000) sound = getu8(buf);
	event = (mask & 0x00000020) ? getu8(buf) : 0;
	if(mask & 0x08000000) solid = getLEu16(buf);

	ent->velocity.x = 0.0;
	ent->velocity.y = 0.0;
	ent->velocity.z = 0.0;

	ent->last = gettime();
}
static void parse_center_print(protocol *state,buffer *buf)
{
	parse_print(state,buf);
}
static void parse_download(protocol *state,buffer *buf)
{
	(void)getLEs16(buf);
	(void)getu8(buf);
}
static void parse_player_info(protocol *state,buffer *buf)
{
	unsigned long mask,mask2;
	unsigned pm_type,teleport_time,pm_flags,gravity;
	unsigned gunframe,fov,rdflags,c;
	vector viewoffset,viewangles;
	vector kick_angles,gunoffset,gunangles;

	mask = getLEu16(buf);
	if(mask & 0x0001) pm_type = getu8(buf);
	if(mask & 0x0002) state->shared->player_info.origin = readvector(buf);
	if(mask & 0x0004)
	{
		state->shared->player_info.velocity.x = (float)getLEs16(buf);
		state->shared->player_info.velocity.y = (float)getLEs16(buf);
		state->shared->player_info.velocity.z = (float)getLEs16(buf);
	}
	if(mask & 0x0008) teleport_time = getu8(buf);
	if(mask & 0x0010) pm_flags = getu8(buf);
	if(mask & 0x0020) gravity = getLEs16(buf);
	if(mask & 0x0040)
	{
		state->delta_angles.x = (float)getLEs16(buf);
		state->delta_angles.y = (float)getLEs16(buf);
		state->delta_angles.z = (float)getLEs16(buf);
	}
	if(mask & 0x0080)
	{
		viewoffset.x = 0.25 * (float)gets8(buf);
		viewoffset.y = 0.25 * (float)gets8(buf);
		viewoffset.z = 0.25 * (float)gets8(buf);
	}
	if(mask & 0x0100)
	{
		viewangles.x = (float)getLEs16(buf);
		viewangles.y = (float)getLEs16(buf);
		viewangles.z = (float)getLEs16(buf);

		viewangles = vec_mul(viewangles,M_PI/Q_PI);
		state->shared->player_info.angles = viewangles;
	}
	if(mask & 0x0200)
	{
		kick_angles.x = 0.25 * (float)gets8(buf);
		kick_angles.y = 0.25 * (float)gets8(buf);
		kick_angles.z = 0.25 * (float)gets8(buf);
	}
	if(mask & 0x1000) state->shared->player_info.gun_index = getu8(buf);
	if(mask & 0x2000)
	{
		gunframe = getu8(buf);
		gunoffset.x = 0.25 * (float)gets8(buf);
		gunoffset.y = 0.25 * (float)gets8(buf);
		gunoffset.z = 0.25 * (float)gets8(buf);
		gunangles.x = 0.25 * (float)gets8(buf);
		gunangles.y = 0.25 * (float)gets8(buf);
		gunangles.z = 0.25 * (float)gets8(buf);
	}
	if(mask & 0x0400)
	{
		(void)getu8(buf);
		(void)getu8(buf);
		(void)getu8(buf);
		(void)getu8(buf);
	}
	if(mask & 0x0800) fov = (float)getu8(buf);
	if(mask & 0x4000) rdflags = getu8(buf);

	mask2 = getLEu32(buf);

	for(c=0;c<32;c++)
	{
		if(mask2 & (1 << c))
			state->shared->player_info.stats[c] = getLEs16(buf);
	}
}

static void init_edicts(protocol *state)
{
	unsigned c;

	if(state->delta_frame == 0xffffffff)
	{
		memcpy(state->shared->edicts,state->baseline,
					sizeof(edict)*MAX_EDICTS);
		return;
	}

	for(c=0;c<16;c++)
	{
		if(state->oldframe_nums[c] != state->delta_frame) continue;

		memcpy(state->shared->edicts,state->oldframes[c],
					sizeof(edict)*MAX_EDICTS);
		return;
	}

	fprintf(stderr,"Server requested delta frame we do not have\n");
	exit(7);
}

static void save_frame(protocol *state)
{
	unsigned c;

	for(c=0;c<15;c++)
	{
		memcpy(state->oldframes[c],state->oldframes[c+1],
					sizeof(edict)*MAX_EDICTS);
		state->oldframe_nums[c] = state->oldframe_nums[c+1];
	}

	memcpy(state->oldframes[15],state->shared->edicts,
				sizeof(edict)*MAX_EDICTS);
	state->oldframe_nums[15] = state->frame_num;
}

static void parse_packet_entities(protocol *state,buffer *buf)
{
	unsigned long mask,effects,renderfx;
	unsigned entity,sound,event,solid,modelindex2,modelindex3,modelindex4;
	vector old_origin;
	edict last[MAX_EDICTS];
	edict *old,*ent;
	double dif;

	memcpy(last,state->shared->edicts,sizeof(edict)*MAX_EDICTS);

	init_edicts(state);

	for(;;)
	{
		mask = getu8(buf);
		if(mask & 0x00000080) mask |= (getu8(buf) << 8);
		if(mask & 0x00008000) mask |= (getu8(buf) << 16);
		if(mask & 0x00800000) mask |= (getu8(buf) << 24);
		entity = (mask & 0x00000100) ? getLEu16(buf) : getu8(buf);
		if(entity >= MAX_EDICTS)
		{
			fprintf(stderr,"CL_ParsePacketEntities:"
					" bad number: %u",entity);
			exit(7);
		}
		if(entity == 0) break;

		ent = state->shared->edicts + entity;
		old = last + entity;

		ent->valid = (mask & 0x00000040) ? 0 : 1;
		if(mask & 0x00000800) ent->modelindex = getu8(buf);
		if(mask & 0x00100000) modelindex2 = getu8(buf);
		if(mask & 0x00200000) modelindex3 = getu8(buf);
		if(mask & 0x00400000) modelindex4 = getu8(buf);
		if(mask & 0x00000010) ent->mframe = getu8(buf);
		if(mask & 0x00020000) ent->mframe = getLEu16(buf);
		if(mask & 0x00010000)
		{
			if(mask&0x02000000) ent->skinnum = getLEu32(buf);
			else ent->skinnum = getu8(buf);
		}
		else
		{
			if(mask & 0x02000000) ent->skinnum = getLEu16(buf);
		}
		if(mask & 0x00004000)
		{
			effects = (mask & 0x00080000)?getLEu32(buf):getu8(buf);
		}
		else
		{
			if(mask & 0x00080000) effects = getLEu16(buf);
		}
		if(mask & 0x00001000)
		{
			renderfx = (mask&0x00040000)?getLEu32(buf):getu8(buf);
		}
		else
		{
			if(mask & 0x00040000) renderfx = getLEu16(buf);
		}

		if(mask & 0x00000001) ent->origin.x = readcoord(buf);
		if(mask & 0x00000002) ent->origin.y = readcoord(buf);
		if(mask & 0x00000200) ent->origin.z = readcoord(buf);

		ent->last = gettime();
		dif = ent->last - old->last;

		ent->velocity = vec_sub(ent->origin,old->origin);
		ent->velocity = vec_mul(ent->velocity,1.0/dif);

		if(mask & 0x00000400) ent->angles.x = readangle(buf);
		if(mask & 0x00000004) ent->angles.y = readangle(buf);
		if(mask & 0x00000008) ent->angles.z = readangle(buf);
		if(mask & 0x01000000) old_origin = readvector(buf);
		if(mask & 0x04000000) sound = getu8(buf);
		event = (mask & 0x00000020) ? getu8(buf) : 0;
		if(mask & 0x08000000) solid = getLEu16(buf);
	}

	save_frame(state);
}
static void parse_delta_packet_entities(protocol *state,buffer *buf)
{
	fprintf(stderr,"received DeltaPacketEntities\n");
	exit(7);
}
static void parse_frame(protocol *state,buffer *buf)
{
	unsigned long uk_b1,count,c;

	state->frame_num = getLEu32(buf);
	state->delta_frame = getLEu32(buf);
	if(PROTOCOL_VERSION != 26) uk_b1 = getu8(buf);
	count = getu8(buf);
	for(c=0;c<count;c++) (void)getu8(buf);
}

static void parse_messages(protocol *state,buffer *buf)
{
	while(bufgetgpos(buf) < getsize(buf))
	{
		switch(getu8(buf))
		{
			case SERVER_BAD:
				parse_bad(state,buf);
				break;

			case SERVER_MUZZLE_FLASH:
				parse_muzzle_flash(state,buf);
				break;

			case SERVER_MUZZLE_FLASH2:
				parse_muzzle_flash2(state,buf);
				break;

			case SERVER_TEMP_ENTITY:
				parse_temp_entity(state,buf);
				break;

			case SERVER_LAYOUT:
				parse_layout(state,buf);
				break;

			case SERVER_INVENTORY:
				parse_inventory(state,buf);
				break;

			case SERVER_NOP:
				parse_nop(state,buf);
				break;

			case SERVER_DISCONNECT:
				parse_disconnect(state,buf);
				break;

			case SERVER_RECONNECT:
				parse_reconnect(state,buf);
				break;

			case SERVER_SOUND:
				parse_sound(state,buf);
				break;

			case SERVER_PRINT:
				parse_print(state,buf);
				break;

			case SERVER_STUFF_TEXT:
				parse_stuff_text(state,buf);
				break;

			case SERVER_SERVER_DATA:
				parse_server_data(state,buf);
				break;

			case SERVER_CONFIG_STRING:
				parse_config_string(state,buf);
				break;

			case SERVER_SPAWN_BASELINE:
				parse_spawn_baseline(state,buf);
				break;

			case SERVER_CENTER_PRINT:
				parse_center_print(state,buf);
				break;

			case SERVER_DOWNLOAD:
				parse_download(state,buf);
				break;

			case SERVER_PLAYER_INFO:
				parse_player_info(state,buf);
				break;

			case SERVER_PACKET_ENTITIES:
				parse_packet_entities(state,buf);
				break;

			case SERVER_DELTA_PACKET_ENTITIES:
				parse_delta_packet_entities(state,buf);
				break;

			case SERVER_FRAME:
				parse_frame(state,buf);
				break;

			default:
				parse_bad(state,buf);
				break;
		}
	}
}

static void send_ack(protocol *state)
{
	buffer *buf;

	buf = newbuffer();

	putLEu32(buf,state->seq);
	putLEu32(buf,state->last_seq);
	putLEu16(buf,state->qport);

	sock_send(state->sock,buf);

	freebuffer(buf);

	state->seq++;
}

static void parse_connected(protocol *state,buffer *buf,unsigned long seq)
{
	state->last_seq = seq;

	if((seq & HIDDEN_BIT) || (getsize(buf) == 8)) send_ack(state);

	proc_ack(state,getLEu32(buf));

	parse_messages(state,buf);
}

static void parse_packet(protocol *state,buffer *buf)
{
	unsigned long seq;

	seq = getLEu32(buf);

	if(seq == 0xffffffff) parse_connectionless(state,buf);
	else parse_connected(state,buf,seq);
}

static void proc_need_ack(protocol *state)
{
	double now;

	if(state->need_ack == NULL) return;

	now = gettime();
	if((now - state->need_ack_last) < state->shared->ping) return;
	state->need_ack_last = now;

	if(state->need_ack->seq == 0) state->need_ack->seq = state->seq;

	bufsetppos(state->need_ack->buf,0);
	putLEu32(state->need_ack->buf,HIDDEN_BIT | state->seq);
	putLEu32(state->need_ack->buf,state->last_seq);

	sock_send(state->sock,state->need_ack->buf);

	state->seq++;
	ping_start(state);
}

static void proc_packet(protocol *state)
{
	buffer *buf;

	buf = sock_recv(state->sock);
	if(!buf) return;

	parse_packet(state,buf);
	freebuffer(buf);
}

static void clear_data(protocol *state)
{
	unsigned c;

	state->delta_frame = 0xffffffff;

	for(c=0;c<16;c++) state->oldframe_nums[c] = 0xffffffff;

	memset(state->baseline,0,sizeof(edict)*MAX_EDICTS);
	memset(state->shared->edicts,0,sizeof(edict)*MAX_EDICTS);

	state->delta_angles = vec_mk(0.0,0.0,0.0);
}

void protocol_update(protocol *state)
{
	switch(state->level)
	{
		case PRE_CHALLENGE:
			send_getchallenge(state);
			state->level = POST_CHALLENGE;
			break;

		case POST_CHALLENGE:
			state->level = PRE_CHALLENGE;
		break;

		case PRE_CONNECT:
			send_connect(state);
			state->level = POST_CONNECT;
			break;

		case POST_CONNECT:
			state->level = PRE_CONNECT;
			break;

		case POST_SYN:
			clear_data(state);
			protocol_send_console_command(state,"new\n");
			state->level = PRE_GAME;
			break;

		case PRE_GAME:
			break;

		case IN_GAME:
			break;

		case POST_GAME:
			protocol_send_console_command(state,"disconnect\n");
			state->level = PRE_GAME_OVER;
			break;

		case PRE_GAME_OVER:
			if(!state->need_ack) state->level = GAME_OVER;
			break;

		case GAME_OVER:
			break;

		default:
			fprintf(stderr,"internal error: unknown state\n");
			exit(7);
	}

	proc_need_ack(state);
	proc_packet(state);
}

int protocol_getstate(protocol *state)
{
	return (state->level - IN_GAME);
}

