/*
 *  GoodBot - autonomous client side Quake2 robot
 *  Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include<stdio.h>
#include<stdlib.h>
#include<stdarg.h>

#ifdef HAVE_STRING_H
#include<string.h>
#endif

#ifdef HAVE_STRINGS_H
#include<strings.h>
#endif

#include<signal.h>
#include<errno.h>
#include<ctype.h>

#ifdef HAVE_UNISTD_H
#include<unistd.h>
#endif

#ifdef HAVE_SYS_TIME_H
#include<sys/time.h>
#endif

#include<sys/types.h>

#ifdef HAVE_SYS_SELECT_H
#include<sys/select.h>
#endif

#include"misc.h"

void *xmalloc(unsigned long size)
{
	void *mem;

	mem = malloc(size);
	if(!mem)
	{
		perror("malloc");
		exit(1);
	}

	return mem;
}

void *xrealloc(void *ptr,unsigned long size)
{
	void *mem;

	mem = realloc(ptr,size);
	if(!mem)
	{
		perror("realloc");
		exit(1);
	}

	return mem;
}

void *xcalloc(unsigned long nmemb,unsigned long size)
{
	void *mem;

	mem = calloc(nmemb,size);
	if(!mem)
	{
		perror("calloc");
		exit(1);
	}

	return mem;
}

static char fd_getchar(void)
{
	int err;
	char c;

retry_read:
	err = read(0,&c,1);

	if(err == 0) c = '\0';
	else if(err != 1)
	{
		if(errno == EINTR) goto retry_read;
		perror("read");
		exit(8);
	}

	return c;
}

char *getline(void)
{
	char *line=NULL;
	unsigned long len=0;
	struct timeval timeout;
	fd_set fds;
	int err;

retry_select:
	FD_ZERO(&fds);
	FD_SET(0,&fds);

	timeout.tv_sec = 0;
	timeout.tv_usec = 0;

	err = select(1,&fds,NULL,NULL,&timeout);
	if(err == 0) return NULL;
	else if(err != 1)
	{
		if(errno == EINTR) goto retry_select;
		perror("select");
		exit(8);
	}

	do
	{
		len++;
		line = xrealloc(line,len + 1);

		line[len - 1] = fd_getchar();
		if(line[len - 1] == '\0')
		{
			free(line);
			return NULL;
		}
	}
	while(line[len - 1] != '\n');

	line[len] = '\0';

	return line;
}

char *xstrdup(char *str)
{
	char *nstr;
	unsigned long len;

	if(!str) return NULL;

	len = strlen(str) + 1;
	nstr = (char*)xmalloc(len);
	memcpy(nstr,str,len);

	return nstr;
}

char *va_strcat(char *first,...)
{
	va_list ap;
	char *str = NULL;
	char *next = first;
	unsigned long l,len = 0;

	va_start(ap,first);

	while(next)
	{
		l = strlen(next);

		str = (char*)xrealloc(str,len + l + 1);
		memcpy(str + len,next,l + 1);

		len += l;

		next = va_arg(ap,char*);
	}

	va_end(ap);

	return str;
}

char **delim_to_array(char *str,char tok)
{
	unsigned long count = 0;
	char **strs = NULL;
	char *nstr,*next;

	nstr = xstrdup(str);
	next = nstr;

	count++;
	strs = xrealloc(strs,count * sizeof(next));
	strs[count-1] = next;

	if((next) && (tok == '\0'))
	{
		count++;
		strs = xrealloc(strs,count * sizeof(next));
		strs[count-1] = NULL;
		return strs;
	}

	while(next)
	{
		next = strchr(next,tok);
		if(next)
		{
			(*next) = '\0';
			next++;
		}

		count++;
		strs = xrealloc(strs,count * sizeof(next));
		strs[count-1] = next;
	}

	return strs;
}

static int strsncasecmp(char *s1,char *s2)
{
	unsigned long c;
	int d;

	for(c=0;s1[c] && s2[c];c++)
	{
		d = tolower(s1[c]) - tolower(s2[c]);
		if(d) return d;
	}

	return 0;
}

char *stristr(char *haystack,char *needle)
{
	unsigned long len;
	char *tmp;

	len = strlen(needle);
	if(len > strlen(haystack)) return NULL;

	for(tmp=haystack;*(tmp+len-1);tmp++)
	{
		if(!strsncasecmp(tmp,needle)) return tmp;
	}

	return NULL;
}

void strip_head_space(char **str)
{
	while(isspace(**str)) (*str)++;
}

void strip_tail_space(char *str)
{
	char *tmp;

	for(tmp = str + strlen(str);tmp > str;tmp--)
	{
		if((*tmp) == '\0') continue;
		if(!isspace(*tmp)) break;
		(*tmp) = '\0';
	}
}

void strlower(char *str)
{
	for(;*str;str++) *str = tolower(*str);
}

double gettime(void)
{
	struct timeval tv;

	if(gettimeofday(&tv,NULL))
	{
		perror("gettimeofday");
		exit(5);
	}

	return (double)tv.tv_sec + (double)tv.tv_usec / 1000000.0;
}

static void xsignal(int signum,RETSIGTYPE (*handler)(int))
{
	if(signal(signum,handler) == SIG_ERR)
	{
		perror("signal");
		exit(5);
	}
}

void regsigs(RETSIGTYPE (*handler)(int))
{
	if(handler == NULL) handler = SIG_DFL;

	xsignal(SIGINT,handler);
	xsignal(SIGTERM,handler);
}

