/*
 *  GoodBot - autonomous client side Quake2 robot
 *  Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include<stdio.h>
#include<stdlib.h>

#ifdef HAVE_STRING_H
#include<string.h>
#endif

#ifdef HAVE_STRINGS_H
#include<strings.h>
#endif

#include"misc.h"
#include"buffer.h"

#define ALIGN	(64*1024)

static void reverse(void *data,unsigned long size)
{
	unsigned char tmp,*p = (unsigned char*)data;
	unsigned long c;

	for(c=0;c<(size/2);c++)
	{
		tmp = p[c];
		p[c] = p[size - c - 1];
		p[size - c - 1] = tmp;
	}
}

buffer *newbuffer(void)
{
	buffer *buf;

	buf = (buffer*)xcalloc(1,sizeof(buffer));

	return buf;
}

void freebuffer(buffer *buf)
{
	if(!buf) return;

	free(buf->data);
	free(buf);
}

unsigned long getsize(buffer *buf)
{
	return buf->size;
}

unsigned char *getdata(buffer *buf)
{
	return buf->data;
}

void bufgseek(buffer *buf,long off)
{
	buf->getp += off;

	if((buf->getp > buf->size) || (buf->getp < 0))
	{
		fprintf(stderr,
			"buffer: attempt to gseek past the end of data\n");
		exit(3);
	}
}

void bufpseek(buffer *buf,long off)
{
	buf->putp += off;

	if((buf->putp > buf->size) || (buf->putp < 0))
	{
		fprintf(stderr,
			"buffer: attempt to seek past the end of data\n");
		exit(3);
	}
}

unsigned long bufgetgpos(buffer *buf)
{
	return buf->getp;
}

void bufsetgpos(buffer *buf,unsigned long off)
{
	buf->getp = off;

	if(buf->getp > buf->size)
	{
		fprintf(stderr,
			"buffer: attempt to setgpos past the end of data\n");
		exit(3);
	}
}

unsigned long bufgetppos(buffer *buf)
{
	return buf->putp;
}

void bufsetppos(buffer *buf,unsigned long off)
{
	buf->putp = off;

	if(buf->putp > buf->size)
	{
		fprintf(stderr,
			"buffer: attempt to setppos past the end of data\n");
		exit(3);
	}
}

void getmem(buffer *buf,void *mem,unsigned long count)
{
	if((buf->getp + count) > buf->size)
	{
		fprintf(stderr,
			"buffer: attempt to getmem past the end of data\n");
		exit(3);
	}

	memmove(mem,buf->data + buf->getp,count);
	buf->getp += count;
}

void putmem(buffer *buf,void *mem,unsigned long count)
{
	buf->putp += count;

	if(buf->putp > buf->maxs)
	{
		buf->maxs = buf->putp;
		buf->maxs += ALIGN - (buf->maxs % ALIGN);
		buf->data = xrealloc(buf->data,buf->maxs);
	}

	memmove(buf->data + buf->putp - count,mem,count);

	if(buf->putp > buf->size) buf->size = buf->putp;
}

void reset(buffer *buf)
{
	buf->getp = 0;
	buf->putp = 0;
	buf->size = 0;
	buf->maxs = 0;
	free(buf->data);
	buf->data = NULL;
}

char *getstr(buffer *buf)
{
	unsigned long len,nlen;
	char *s,*ns;

	s = (char*)(buf->data + buf->getp);
	len = buf->getp;

	while((buf->getp < buf->size) && gets8(buf));
	len = buf->getp - len;

	nlen = len + ((s[len - 1]) ? 1 : 0);

	ns = xmalloc(nlen);
	memcpy(ns,s,len);

	ns[nlen - 1] = '\0';

	return ns;
}

void putstr(buffer *buf,char *s)
{
	putmem(buf,s,strlen(s)+1);
}

void putnstr(buffer *buf,char *s)
{
	putmem(buf,s,strlen(s));
}

unsigned char getu8(buffer *buf)
{
	unsigned char r = 0;
	getmem(buf,&r,1);
	return r;
}

void putu8(buffer *buf,unsigned char v)
{
	putmem(buf,&v,1);
}

char gets8(buffer *buf)
{
	unsigned char u,s;
	char r;

	u = getu8(buf);
	if(u & 0x80)
	{
		s = 0xff;
		s = ~s;
		u |= s;
	}

	r = *(char*)&u;

	return r;
}

void puts8(buffer *buf,char v)
{
	unsigned char u;

	u = *(unsigned char*)&v;

	putu8(buf,u);
}

unsigned short getLEu16(buffer *buf)
{
	unsigned char b[2] = {0,0};
	unsigned short r;

	getmem(buf,b,2);

	r  = (unsigned short)b[0];
	r |= (unsigned short)b[1] << 8;

	return r;
}

void putLEu16(buffer *buf,unsigned short v)
{
	unsigned char b[2];

	b[0] =  v & 0xff;
	b[1] = (v & 0xff00) >> 8;

	putmem(buf,b,2);
}

short getLEs16(buffer *buf)
{
	unsigned short u,s;
	short r;

	u = getLEu16(buf);
	if(u & 0x8000)
	{
		s = 0xffff;
		s = ~s;
		u |= s;
	}

	r = *(short*)&u;

	return r;
}

void putLEs16(buffer *buf,short v)
{
	unsigned short u;

	u = *(unsigned short*)&v;

	putLEu16(buf,u);
}

unsigned long getLEu32(buffer *buf)
{
	unsigned char b[4] = {0,0,0,0};
	unsigned long r;

	getmem(buf,b,4);

	r  = (unsigned long)b[0];
	r |= (unsigned long)b[1] << 8;
	r |= (unsigned long)b[2] << 16;
	r |= (unsigned long)b[3] << 24;

	return r;
}

void putLEu32(buffer *buf,unsigned long v)
{
	unsigned char b[4];

	b[0] =  v & 0xff;
	b[1] = (v & 0xff00) >>  8;
	b[2] = (v & 0xff0000) >> 16;
	b[3] = (v & 0xff000000) >> 24;

	putmem(buf,b,4);
}

long getLEs32(buffer *buf)
{
	unsigned long u,s;
	long r;

	u = getLEu32(buf);
	if(u & 0x80000000)
	{
		s = 0xffffffff;
		s = ~s;
		u |= s;
	}

	r = *(long*)&u;

	return r;
}

void putLEs32(buffer *buf,long v)
{
	unsigned long u;

	u = *(unsigned long*)&v;

	putLEu32(buf,u);
}

float getLEf32(buffer *buf)
{
	float f;

	getmem(buf,(void*)&f,4);

#ifdef WORDS_BIGENDIAN
	reverse(&f,4);
#endif

	return f;
}

void putLEf32(buffer *buf,float f)
{
#ifdef WORDS_BIGENDIAN
	reverse(&f,4);
#endif
	putmem(buf,(void*)&f,4);
}

