/*
 *  GoodBot - autonomous client side Quake2 robot
 *  Copyright (C) 1998 Jens Vaasjo <jvaasjo@iname.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
*   This program is distributed in the hope that it will be useful,
*   but WITHOUT ANY WARRANTY; without even the implied warranty of
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*   GNU General Public License for more details.
*
*   You should have received a copy of the GNU General Public License
*   along with this program; if not, write to the Free Software
*   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

#include<stdio.h>
#include<stdlib.h>

#ifdef HAVE_STRING_H
#include<string.h>
#endif

#ifdef HAVE_STRINGS_H
#include<strings.h>
#endif

#include"findfile.h"
#include"buffer.h"
#include"vector.h"
#include"misc.h"
#include"bsp.h"

#define BSP_MAGIC		"IBSP"
#define BSP_VERSION		38

#define CONTENTS_SOLID		1
#define CONTENTS_WINDOW		2
#define CONTENTS_COLLISION	(CONTENTS_SOLID | CONTENTS_WINDOW)

typedef struct
{
	unsigned long offset;
	unsigned long length;
} lump_t;

static lump_t readlump(buffer *buf)
{
	lump_t lump;

	lump.offset = getLEu32(buf);
	lump.length = getLEu32(buf);

	return lump;
}

static vector buf_read_vector(buffer *buf)
{
	vector v;

	v.x = getLEf32(buf);
	v.y = getLEf32(buf);
	v.z = getLEf32(buf);

	return v;
}

static plane_t *loadplanes(buffer *buf,lump_t lump)
{
	plane_t *planes;
	unsigned count,c;

	bufsetgpos(buf,lump.offset);
	count = lump.length/20;

	planes = (plane_t*)xmalloc(count*sizeof(plane_t));

	for(c=0;c<count;c++)
	{
		planes[c].normal = buf_read_vector(buf);
		planes[c].dist = getLEf32(buf);
		bufgseek(buf,4);
	}

	return planes;
}

static node_t *loadnodes(buffer *buf,lump_t lump)
{
	node_t *nodes;
	unsigned count,c;

	bufsetgpos(buf,lump.offset);
	count = lump.length/28;

	nodes = (node_t*)xmalloc(count*sizeof(node_t));

	for(c=0;c<count;c++)
	{
		nodes[c].plane = getLEu32(buf);
		nodes[c].c1 = getLEs32(buf);
		nodes[c].c2 = getLEs32(buf);
		bufgseek(buf,16);
	}

	return nodes;
}

static leaf_t *loadleafs(buffer *buf,lump_t lump)
{
	leaf_t *leafs;
	unsigned count,c;

	bufsetgpos(buf,lump.offset);
	count = lump.length/28;

	leafs = (leaf_t*)xmalloc(count*sizeof(leaf_t));

	for(c=0;c<count;c++)
	{
		leafs[c].contents = getLEu32(buf);
		bufgseek(buf,24);
	}

	return leafs;
}

static bsp *loadbspdata(buffer *buf)
{
	lump_t planes,nodes,leafs;
	bsp *map;

	bufgseek(buf,8);
	planes = readlump(buf);

	bufgseek(buf,16);
	nodes = readlump(buf);

	bufgseek(buf,24);
	leafs = readlump(buf);

	map = xmalloc(sizeof(bsp));

	map->planes = loadplanes(buf,planes);
	map->nodes = loadnodes(buf,nodes);
	map->leafs = loadleafs(buf,leafs);

	return map;
}

static bsp *buf_loadbsp(buffer *buf,char *name)
{
	char magic[] = BSP_MAGIC;
	unsigned long version;

	if(!buf) return NULL;

	getmem(buf,magic,strlen(BSP_MAGIC));
	if(memcmp(magic,BSP_MAGIC,strlen(BSP_MAGIC)))
	{
		fprintf(stderr,"%s: bad magic\n",name);
		return NULL;
	}

	version = getLEu32(buf);
	if(version != BSP_VERSION)
	{
		fprintf(stderr,"%s: unsupported version: %lu\n",name,version);
		return NULL;
	}

	return loadbspdata(buf);
}

bsp *bsp_load(char *name,char **path)
{
	buffer *buf;
	bsp *map;

	buf = findfile(path,name);
	map = buf_loadbsp(buf,name);
	freebuffer(buf);

	return map;
}

void bsp_free(bsp *map)
{
	if(!map) return;

	free(map->leafs);
	free(map->nodes);
	free(map->planes);
	free(map);
}

static float distance_to_plane(vector *v,plane_t *p)
{
	return p->normal.x * v->x + p->normal.y * v->y
			+ p->normal.z * v->z - p->dist;
}

static int notvisible(bsp *map,vector p1,vector p2,long i)
{
	float d1,d2;
	long i1,i2;
	vector p;

	if(i < 0)
	{
		i = -(i+1);
		if(map->leafs[i].contents & CONTENTS_COLLISION) return 1;
		return 0;
	}

	d1 = distance_to_plane(&p1,map->planes + map->nodes[i].plane);
	d2 = distance_to_plane(&p2,map->planes + map->nodes[i].plane);

	i1 = (d1 > 0.0) ? map->nodes[i].c1 : map->nodes[i].c2;
	i2 = (d2 > 0.0) ? map->nodes[i].c1 : map->nodes[i].c2;

	if(i1 == i2) return notvisible(map,p1,p2,i1);

	p.x = (d1 * p2.x - d2 * p1.x) / (d1 - d2);
	p.y = (d1 * p2.y - d2 * p1.y) / (d1 - d2);
	p.z = (d1 * p2.z - d2 * p1.z) / (d1 - d2);

	return notvisible(map,p1,p,i1) || notvisible(map,p,p2,i2);
}

int bsp_isvisible(bsp *map,vector p1,vector p2)
{
	if(!map) return 1;

	return !notvisible(map,p1,p2,0);
}

