/* wads.c -- miscellaneous functions used in WAD utilities */
#include "wads.h"

void UsageError (char *message,...)
{
  if (message!=NULL) {
    va_list ap;
    va_start(ap, message);
    vfprintf (stderr,message,ap);
    va_end(ap);
    fprintf (stderr," !\n\n");
  }
  exit(1);
}

int wc (char *str)
/* returns 1 if str contains a ? or a * */
{
  char  *p;
  for (p=str;*p!='\0';p++)
    if ((*p=='*')||(*p=='?'))
      return (1);
  return (0);
}

int wcmatch (char *pattern, char *str)
/* "wildcard match"; pattern can contain '*'s and '?'s */
{
  int i,k;
  if (strlen(pattern) > strlen(str))
	k = strlen (pattern);
  else
	k = strlen (str);
  for (i=0;i<k;i++) {
	if (i>=strlen(pattern))
	  return(0);
	if ((i>=strlen(str))&&(pattern[i]!='*'))
	  return (0);
	if (pattern[i]=='*')
	  return (1);
	if ((pattern[i]!='?')&&(pattern[i]!=str[i]))
	  return (0);
  };
  return (1);
}

/* the following two functions are used to get around the fact that all
   the names in WAD files are 8 characters long, uppercase, and not
   always null-terminated. */

/* Converts a null-terminated string into an all-upper-case character
   array of length 'length';
   if str is shorter than 'length', it's padded with '\0's at the end;
   it str is longer  than 'length', the end is truncated;
   if str is 'length' characters long or longer, the result is not
      null-terminated;
   the return value is the adress of the destination string (name) */
char *str2name (char *str, char *name, int length)
{
  int i;
  char c;
  for (i=0;((i<length)&&(str[i]!='.')&&(str[i]!='\0'));i++) {
    name[i]=toupper(str[i]);
  }
  for (;i<length;i++)
    name[i]='\0';
  return (name);
}

/* Converts a character array of length 'length' into a standard
   null-terminated string of length 'length'+1.
   Str must be declared to be at least 'length'+1 characters long
   the return value is the adress of the destination string (str) */
char *name2str (char *name, char *str, int length)
{
  memcpy (str, name, length);
  str[length]='\0';
  return (str);
}

/* Tests whether 'name' can be the name of a WAD entry that starts a map.
   Returns 1 if name is of the form 'ExMy' or 'MAPxy', where xs and y
   are digits; returns 0 otherwise.  */
int mapname (char *name)
{
  if ((name[0]=='E')&&(name[2]=='M')&&(isdigit(name[1]))&&(isdigit(name[3])))
    return(1); /* a Doom 1 map */
  if ((name[0]=='M')&&(name[1]=='A')&&(name[2]=='P')&&(isdigit(name[3]))&&(isdigit(name[4])))
    return(1); /* a Doom 2 map */
  return (0);  /* obviously, not a pre-1.9 map */
}

/* Tests whether name can be the name of a map data entry */
int map_data (char *name)
{
  char s[9];
  name2str(name,s,8);
  if (!strcmp(s,"THINGS"))
    return (1);
  if (!strcmp(s,"VERTEXES"))
    return (1);
  if (!strcmp(s,"LINEDEFS"))
    return (1);
  if (!strcmp(s,"SIDEDEFS"))
    return (1);
  if (!strcmp(s,"SECTORS"))
    return (1);
  if (!strcmp(s,"SSECTORS"))
    return (1);
  if (!strcmp(s,"SEGS"))
    return (1);
  if (!strcmp(s,"NODES"))
    return (1);
  if (!strcmp(s,"BLOCKMAP"))
    return (1);
  if (!strcmp(s,"REJECT"))
    return (1);
  return (0);
}

int wadfile (FILE *fp)
{
  Header header;
  long pos;
  char s[4];

  if (fp==NULL)
    return(0);
  fseek(fp,0,SEEK_END);
  pos=ftell(fp);
  fseek(fp,0,SEEK_SET);
  if(pos<sizeof(header))
    return (0);
  fread(&header,sizeof(header),1,fp);
  fseek(fp,0,SEEK_SET);
  if (strcmp(name2str(header.type+1,s,3),"WAD"))
    return(0);
  return(1);
}

WAD *new_wad (char *fname)
/* opens a file (overwrites if it exixts) and allocates a directory for it */
{
  WAD *p;
  Header header;

  p=malloc (sizeof(WAD));
  p->dirsize=INIT_ALLOC;
  p->num_entries=0;
  if ((p->dirptr=malloc(INIT_ALLOC*sizeof(DirEntry)))==NULL) {
    free (p);
    return (NULL);
  }
  if ((p->fp=fopen(fname,"wb"))==NULL) {
    free(p->dirptr);
    free(p);
    return(NULL);
  }
  fwrite(&header,sizeof(header),1,p->fp); /* fix this later */
  return(p);
}

long fcopynbytes (FILE* from, FILE* to, long n)
{
  void *p;
  long i;
  long bufsize;

  if (n<=0)
    return(n);
  bufsize=(n>MAX_FCOPY_BUFSIZE) ? FCOPY_BUFSIZE : n;
  p=malloc (bufsize);
  for (i=0;i<n/bufsize;i++) {
    fread(p, bufsize, 1, from);
    fwrite(p, bufsize, 1, to);
  };
  fread (p, n%bufsize, 1, from);
  fwrite(p, n%bufsize, 1, to);
  free(p);
  return(n);
}

void copy_file_to_wad (WAD *dest, FILE* src, char *entry_name)
{
  DirEntry entry;

  if ((dest==NULL)||(src==NULL))
    return;
  str2name(entry_name,entry.name,8);
  fseek(src,0,SEEK_END);
  entry.length=ftell(src);
  entry.start=0;
  copy_entry_to_wad(dest,src,entry);
}

/* Copies the data pointed to by direntry from infile to outfile,
   preserving the original position of infile. */
void copy_entry_to_file (FILE *dest, FILE *src, DirEntry entry)
{
  long savpos;

  if ((dest==NULL)||(src==NULL)||(entry.length==0))
    return;
  savpos=ftell(src);
  fseek(src,entry.start,SEEK_SET);
  fcopynbytes(src,dest,entry.length);
  fseek(src,savpos,SEEK_SET);
}

void copy_entry_to_wad (WAD *dest, FILE* src, DirEntry entry)
{
  DirEntry tmp;

  if (dest==NULL)
    return;
  if (src==NULL)
    entry.length=0;
  tmp=entry;
  tmp.start=ftell(dest->fp);
  if (entry.length!=0)
    copy_entry_to_file(dest->fp, src, entry);
  if (dest->num_entries>=dest->dirsize)
    dest->dirptr=realloc(dest->dirptr,(dest->dirsize+=REALLOC_INCR)*sizeof(DirEntry));
  dest->dirptr[dest->num_entries++]=tmp;
}

void close_wad (WAD *wad, int iwad)
{
  Header header;

  if (wad==NULL)
    return;
  memcpy(header.type,"PWAD",4);
  if (iwad)
    header.type[0]='I';
  header.num_entries=wad->num_entries;
  header.dir_start=ftell(wad->fp);
  fwrite(wad->dirptr,sizeof(DirEntry),wad->num_entries,wad->fp);
  fseek(wad->fp,0,SEEK_SET);
  fwrite(&header,sizeof(header),1,wad->fp);
  fclose(wad->fp);
  free(wad->dirptr);
  free(wad);
}