/*
	mylib.c and mylib.h are two files I wrote to make my life easier.
They provide error checking for functions that are likely to generate errors,
so I don't have to worry about it each time I call fread or malloc.  The way
it is setup is as follows.
	For every error-prone function I use, mylib.c provides a similarly
named "jacket", which calls the standard function, checks its return value,
and reports any errors.  In most cases, this also causes the program to
terminate.  The only exception to this rule is when fopen cannot open a file.
If no errors are detected, the result is simpley returned to the calling
function.
	Mylib.h contains the prototypes of the functions in mylib.c.  For each
standard functions for which mylib.c provides a "jacket", mylib.h also
contains a #define that replaces any call to the standard function by a
call to the corresponding jacket.  Thus, the program files look as if they
were using the standard fuctions, except that each on them #includes wads.h,
which in turn #includes mylib.c.
	Since read/write errors are among the most frequent (especially during
debugging), I wanted them to be reported as gracefully as possible.  One thing
I found particularly useful was to report the name of the file in which the
error occurred.  However, as far as I know there's no wasy way to get that
information from the file handle.  Thus, I decided to comlicate the matters
a bit by using my own FILE structure (which includes a pointer to a regular
FILE structure).  All of my functions actually work with structures of type
MILE (a clumsy abbreviation for MY FILE).  To make the change transparent to
the programs, I re'#defined FILE to mean MILE in mylib.h.
	This #definition has a couple of nasty side effects.  First, my
functions do not work with stdin, stdout, and stderr, as these are always
declared to be of type FILE*.  Second, functions like gets and fprintf (for
which I haven't written a jacket) do not work with file handles declared as
FILE* (which now means MILE*).
	In my case it didn't matter (I just dodn't use those functions),
but I figured it might be useful to include a switch that would let me use
the error-checking memory allocation functions, but not mess with the
standard file IO functions.  Thus, the redefinition of FILE and certain
file IO functions takes place only if MYFILEIO_YES is defined.  This macro
must be defined BEFORE #including "mylib.h", since otherwise it will have
no effect on the directives inside mylib.h.
	Another switch I thought might be useful is MYLIB_YES.  It also must
be defined before the #include.  If it is not, the entire file mylib.h has
no effect, except to #include some standard headers like <stdio.h> and
<ctype.h>.  Thus, if you don't like this business of renaming standard
functions and are willing to give up the error checking it provides, all you
need to do is go tp WADS.H and comment out the line

#define MYLIB_YES

(it's not necessary to undefine MYFILEIO_YES -- MYLIB_YES takes precedence)
*/

#ifndef MYLIB_H
#define MYLIB_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <stdarg.h>
#include <ctype.h>

#ifdef MYLIB_YES

#define DEBUG
#define PROG_ERROR prog_error (__FILE__, __LINE__)

typedef struct {
  FILE *fp;
  char *name;
} MILE;

/* the following #define's do not apply inside mylib.c */
#ifndef mylib_c
  #define malloc  memalloc
  #define free    memfree
  #define realloc memrealloc
#endif

#ifdef MYFILEIO_YES /* MYFILEIO_YES must not be defined in mylib.c */
  #define FILE   MILE
  #define fopen   fileopen
  #define fclose  fileclose
  #define fread   fileread
  #define fwrite  filewrite
  #define fseek   fileseek
  #undef  feof
  #define feof    fileeof
  #define ftell   filetell
  #define fgets   filegets
#endif

#define strdup strdp

#define size_t_short /* define this if size_t is 2 bytes */
#define MAX_FILES 10 /* how many files can be open simultaneously */

char *strdp (const char *s);
void *memalloc (size_t size);
void *memrealloc (void *block, size_t size);
void memfree (void *block);
MILE *fileopen(const char *name, const char *mode);
void fileclose(MILE* handle);
size_t fileread (void *ptr, size_t size, size_t n, MILE* handle);
size_t filewrite(void *ptr, size_t size, size_t n, MILE* handle);
int fileseek (MILE* stream, long offset, int whence);
int fileeof (MILE* handle);
long int filetell (MILE* handle);
char *filegets(char *s, int n, MILE *stream);


void prog_error (char *file, long line);
void sys_error (char *format_string, ...);

#endif /* #ifdef MYLIB_YES */
#endif /* #ifndef MYLIB_H */