//-------------------------------------------------------------------------
/*
Copyright (C) 2016 EDuke32 developers and contributors

This file is part of EDuke32.

EDuke32 is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License version 2
as published by the Free Software Foundation.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/
//-------------------------------------------------------------------------

#include "compat.h"

#include "duke3d.h"
#include "renderlayer.h" // for win_gethwnd()
#include <atomic>

#define DQSIZE 128

int32_t g_numEnvSoundsPlaying, g_highestSoundIdx = 0;

static int32_t MusicIsWaveform = 0;
static char *MusicPtr = NULL;
static int32_t MusicVoice = -1;
static int32_t MusicPaused = 0;
static int32_t SoundPaused = 0;

std::atomic<uint32_t> dnum;
uint32_t dq[DQSIZE];

void S_SoundStartup(void)
{
    void *initdata = NULL;

#ifdef MIXERTYPEWIN
    initdata = (void *) win_gethwnd(); // used for DirectSound
#endif

    initprintf("Initializing sound... ");

    if (FX_Init(ud.config.NumVoices, ud.config.NumChannels, ud.config.MixRate, initdata) != FX_Ok)
    {
        initprintf("failed! %s\n", FX_ErrorString(FX_Error));
        return;
    }

    initprintf("%d voices, %d channels, %d-bit %d Hz\n", ud.config.NumVoices, ud.config.NumChannels,
        ud.config.NumBits, ud.config.MixRate);

    for (bssize_t i=0; i<=g_highestSoundIdx; ++i)
    {
        for (bssize_t j = 0; j<MAXSOUNDINSTANCES; ++j)
        {
            g_sounds[i].num                    = 0;
            g_sounds[i].instances[j].voice     = 0;
            g_sounds[i].instances[j].spriteNum = -1;
            g_sounds[i].instances[j].sndist    = UINT16_MAX;
            g_sounds[i].instances[j].clock     = 0;
        }

        g_soundlocks[i] = 199;
    }

    S_PrecacheSounds();

    FX_SetVolume(ud.config.MasterVolume);
    S_MusicVolume(MASTER_VOLUME(ud.config.MusicVolume));

    FX_SetReverseStereo(ud.config.ReverseStereo);
    FX_SetCallBack(S_Callback);
    FX_SetPrintf(initprintf);
}

void S_SoundShutdown(void)
{
    if (MusicVoice >= 0)
        S_MusicShutdown();

    if (FX_Shutdown() != FX_Ok)
    {
        Bsprintf(tempbuf, "S_SoundShutdown(): error: %s", FX_ErrorString(FX_Error));
        G_GameExit(tempbuf);
    }
}

void S_MusicStartup(void)
{
    initprintf("Initializing music...\n");

    if (MUSIC_Init(0, 0) == MUSIC_Ok || MUSIC_Init(1, 0) == MUSIC_Ok)
    {
        MUSIC_SetVolume(MASTER_VOLUME(ud.config.MusicVolume));
        return;
    }

    initprintf("S_MusicStartup(): failed initializing\n");
}

void S_MusicShutdown(void)
{
    S_StopMusic();

    if (MUSIC_Shutdown() != MUSIC_Ok)
        initprintf("%s\n", MUSIC_ErrorString(MUSIC_ErrorCode));
}

void S_PauseMusic(int32_t onf)
{
    if (MusicPaused == onf || (MusicIsWaveform && MusicVoice < 0))
        return;

    MusicPaused = onf;

    if (MusicIsWaveform)
    {
        FX_PauseVoice(MusicVoice, onf);
        return;
    }

    if (onf)
        MUSIC_Pause();
    else
        MUSIC_Continue();
}

void S_PauseSounds(int32_t onf)
{
    if (SoundPaused == onf)
        return;

    SoundPaused = onf;

    for (bssize_t i=0; i<=g_highestSoundIdx; ++i)
    {
        for (bssize_t j = 0; j<MAXSOUNDINSTANCES; ++j)
        {
            if (g_sounds[i].instances[j].voice > 0)
                FX_PauseVoice(g_sounds[i].instances[j].voice, onf);
        }
    }
}



void S_MusicVolume(int32_t volume)
{
    if (MusicIsWaveform && MusicVoice >= 0)
        FX_SetPan(MusicVoice, volume, volume, volume);

    MUSIC_SetVolume(volume);
}

void S_RestartMusic(void)
{
    if (ud.recstat != 2 && g_player[myconnectindex].ps->gm&MODE_GAME)
    {
        S_PlayLevelMusicOrNothing(g_musicIndex);
    }
    else if (G_GetLogoFlags() & LOGO_PLAYMUSIC)
    {
        S_PlaySpecialMusicOrNothing(MUS_INTRO);
    }
}

void S_MenuSound(void)
{
#ifndef EDUKE32_STANDALONE
    static int32_t SoundNum;
    int32_t const menusnds[] = {
        LASERTRIP_EXPLODE, DUKE_GRUNT,       DUKE_LAND_HURT,   CHAINGUN_FIRE, SQUISHED,      KICK_HIT,
        PISTOL_RICOCHET,   PISTOL_BODYHIT,   PISTOL_FIRE,      SHOTGUN_FIRE,  BOS1_WALK,     RPG_EXPLODE,
        PIPEBOMB_BOUNCE,   PIPEBOMB_EXPLODE, NITEVISION_ONOFF, RPG_SHOOT,     SELECT_WEAPON,
    };
    int32_t s = VM_OnEventWithReturn(EVENT_OPENMENUSOUND, g_player[screenpeek].ps->i, screenpeek, menusnds[SoundNum++ % ARRAY_SIZE(menusnds)]);
#else
    int32_t s = VM_OnEventWithReturn(EVENT_OPENMENUSOUND, g_player[screenpeek].ps->i, screenpeek, -1);
#endif
    if (s != -1)
        S_PlaySound(s);
}

static int32_t S_PlayMusic(const char *fn)
{
    if (!ud.config.MusicToggle)
        return 0;

    if (fn == NULL)
        return 1;

    int32_t fp = S_OpenAudio(fn, 0, 1);
    if (EDUKE32_PREDICT_FALSE(fp < 0))
    {
        OSD_Printf(OSD_ERROR "S_PlayMusic(): error: can't open \"%s\" for playback!\n",fn);
        return 2;
    }

    int32_t MusicLen = kfilelength(fp);

    if (EDUKE32_PREDICT_FALSE(MusicLen < 4))
    {
        OSD_Printf(OSD_ERROR "S_PlayMusic(): error: empty music file \"%s\"\n", fn);
        kclose(fp);
        return 3;
    }

    char * MyMusicPtr = (char *)Xaligned_alloc(16, MusicLen);
    int MyMusicSize = kread(fp, MyMusicPtr, MusicLen);

    if (EDUKE32_PREDICT_FALSE(MyMusicSize != MusicLen))
    {
        OSD_Printf(OSD_ERROR "S_PlayMusic(): error: read %d bytes from \"%s\", expected %d\n",
                   MyMusicSize, fn, MusicLen);
        kclose(fp);
        ALIGNED_FREE_AND_NULL(MyMusicPtr);
        return 4;
    }

    kclose(fp);

    if (!Bmemcmp(MyMusicPtr, "MThd", 4))
    {
        int32_t retval = MUSIC_PlaySong(MyMusicPtr, MUSIC_LoopSong);

        if (retval != MUSIC_Ok)
        {
            ALIGNED_FREE_AND_NULL(MyMusicPtr);
            return 5;
        }

        if (MusicIsWaveform && MusicVoice >= 0)
        {
            FX_StopSound(MusicVoice);
            MusicVoice = -1;
        }

        MusicIsWaveform = 0;
        ALIGNED_FREE_AND_NULL(MusicPtr);
        MusicPtr    = MyMusicPtr;
        g_musicSize = MyMusicSize;
    }
    else
    {
        int32_t const mvol = MASTER_VOLUME(ud.config.MusicVolume);
        int MyMusicVoice = FX_Play(MyMusicPtr, MusicLen, 0, 0,
                                       0, mvol, mvol, mvol,
                                       FX_MUSIC_PRIORITY, MUSIC_ID);

        if (MyMusicVoice <= FX_Ok)
        {
            ALIGNED_FREE_AND_NULL(MyMusicPtr);
            return 5;
        }

        if (MusicIsWaveform && MusicVoice >= 0)
            FX_StopSound(MusicVoice);

        MUSIC_StopSong();

        MusicVoice      = MyMusicVoice;
        MusicIsWaveform = 1;
        ALIGNED_FREE_AND_NULL(MusicPtr);
        MusicPtr    = MyMusicPtr;
        g_musicSize = MyMusicSize;
    }

    return 0;
}

static void S_SetMusicIndex(unsigned int m)
{
    g_musicIndex = m;
    ud.music_episode = m / MAXLEVELS;
    ud.music_level   = m % MAXLEVELS;
}

int S_TryPlayLevelMusic(unsigned int m)
{
    ud.returnvar[0] = m / MAXLEVELS;
    ud.returnvar[1] = m % MAXLEVELS;
    int retval = VM_OnEvent(EVENT_PLAYLEVELMUSICSLOT, g_player[myconnectindex].ps->i, myconnectindex);

    if (retval < 0)
        return 0;

    char const * musicfn = g_mapInfo[m].musicfn;
    if (musicfn != NULL)
    {
        if (!S_PlayMusic(musicfn))
        {
            S_SetMusicIndex(m);
            return 0;
        }
    }

    return 1;
}

void S_PlayLevelMusicOrNothing(unsigned int m)
{
    if (S_TryPlayLevelMusic(m))
    {
        S_StopMusic();
        S_SetMusicIndex(m);
    }
}

int S_TryPlaySpecialMusic(unsigned int m)
{
    char const * musicfn = g_mapInfo[m].musicfn;
    if (musicfn != NULL)
    {
        if (!S_PlayMusic(musicfn))
        {
            S_SetMusicIndex(m);
            return 0;
        }
    }

    return 1;
}

void S_PlaySpecialMusicOrNothing(unsigned int m)
{
    if (S_TryPlaySpecialMusic(m))
    {
        S_StopMusic();
        S_SetMusicIndex(m);
    }
}

void S_ContinueLevelMusic(void)
{
    VM_OnEvent(EVENT_CONTINUELEVELMUSICSLOT, g_player[myconnectindex].ps->i, myconnectindex);
}

int32_t S_GetMusicPosition(void)
{
    int32_t position = 0;

    if (MusicIsWaveform)
        FX_GetPosition(MusicVoice, &position);

    return position;
}

void S_SetMusicPosition(int32_t position)
{
    if (MusicIsWaveform)
        FX_SetPosition(MusicVoice, position);
}

void S_StopMusic(void)
{
    MusicPaused = 0;

    if (MusicIsWaveform && MusicVoice >= 0)
    {
        FX_StopSound(MusicVoice);
        MusicVoice = -1;
        MusicIsWaveform = 0;
    }

    MUSIC_StopSong();

    ALIGNED_FREE_AND_NULL(MusicPtr);
    g_musicSize = 0;
}

void S_Cleanup(void)
{
    static uint32_t ldnum;
    uint32_t const odnum = dnum;

    while (ldnum < odnum)
    {
        uint32_t num = dq[ldnum++ & (DQSIZE - 1)];

        // negative index is RTS playback
        if ((int32_t)num < 0)
        {
            if (rts_lumplockbyte[-(int32_t)num] >= 200)
                rts_lumplockbyte[-(int32_t)num]--;
            continue;
        }

        // num + (MAXSOUNDS*MAXSOUNDINSTANCES) is a sound played globally
        // for which there was no open slot to keep track of the voice
        if (num >= (MAXSOUNDS*MAXSOUNDINSTANCES))
        {
            g_soundlocks[num-(MAXSOUNDS*MAXSOUNDINSTANCES)]--;
            continue;
        }

        int32_t j = num & (MAXSOUNDINSTANCES-1);

        num = (num - j) / MAXSOUNDINSTANCES;

        sound_t & snd = g_sounds[num];

        int const spriteNum = snd.instances[j].spriteNum;

        if (EDUKE32_PREDICT_FALSE(snd.num > MAXSOUNDINSTANCES))
            OSD_Printf(OSD_ERROR "S_Cleanup(): num exceeds MAXSOUNDINSTANCES! g_sounds[%d].num %d wtf?\n", num, snd.num);

        if (snd.num > 0)
            snd.num--;

        // MUSICANDSFX uses t_data[0] to control restarting the sound
        // CLEAR_SOUND_T0
        if (spriteNum != -1 && S_IsAmbientSFX(spriteNum) && sector[SECT(spriteNum)].lotag < 3)  // ST_2_UNDERWATER
            actor[spriteNum].t_data[0] = 0;

        snd.instances[j].spriteNum = -1;
        snd.instances[j].voice     = 0;
        snd.instances[j].sndist    = UINT16_MAX;
        snd.instances[j].clock     = 0;

        g_soundlocks[num]--;
    }
}

// returns number of bytes read
int32_t S_LoadSound(uint32_t num)
{
    if (num > (unsigned)g_highestSoundIdx || EDUKE32_PREDICT_FALSE(g_sounds[num].filename == NULL))
        return 0;

    int32_t fp = S_OpenAudio(g_sounds[num].filename, g_loadFromGroupOnly, 0);

    if (EDUKE32_PREDICT_FALSE(fp == -1))
    {
        OSD_Printf(OSDTEXT_RED "Sound %s(#%d) not found!\n",g_sounds[num].filename,num);
        return 0;
    }

    int32_t l = kfilelength(fp);
    g_soundlocks[num] = 200;
    g_sounds[num].soundsiz = l;
    cacheAllocateBlock((intptr_t *)&g_sounds[num].ptr, l, (char *)&g_soundlocks[num]);
    l = kread(fp, g_sounds[num].ptr, l);
    kclose(fp);

    return l;
}

void S_PrecacheSounds(void)
{
    for (int32_t i = 0, j = 0; i <= g_highestSoundIdx; ++i)
        if (g_sounds[i].ptr == 0)
        {
            j++;
            if ((j&7) == 0)
                G_HandleAsync();

            S_LoadSound(i);
        }
}

static inline int32_t S_GetPitch(int32_t num)
{
    int32_t const range = klabs(g_sounds[num].pe - g_sounds[num].ps);
    return (range == 0) ? g_sounds[num].ps : min(g_sounds[num].ps, g_sounds[num].pe) + rand() % range;
}

static int32_t S_TakeSlot(int32_t num)
{
    S_Cleanup();

    uint16_t dist = 0, clock = 0;
    int32_t i = 0, j = 0;

    while (j < MAXSOUNDINSTANCES && g_sounds[num].instances[j].voice > 0)
    {
        if (g_sounds[num].instances[j].sndist > dist ||
            (g_sounds[num].instances[j].sndist == dist && g_sounds[num].instances[j].clock > clock))
        {
            clock = g_sounds[num].instances[j].clock;
            dist = g_sounds[num].instances[j].sndist;
            i = j;
        }

        j++;
    }

    if (j != MAXSOUNDINSTANCES)
        return j;

    if (FX_SoundActive(g_sounds[num].instances[i].voice))
        FX_StopSound(g_sounds[num].instances[i].voice);

    dq[dnum & (DQSIZE - 1)] = (num * MAXSOUNDINSTANCES) + i;
    dnum++;
    S_Cleanup();

    return i;
}

static int32_t S_GetSlot(int32_t num)
{
    int32_t j = 0;

    while (j < MAXSOUNDINSTANCES && g_sounds[num].instances[j].voice > 0)
        j++;

    return j == MAXSOUNDINSTANCES ? S_TakeSlot(num) : j;
}

static inline int32_t S_GetAngle(int32_t camang, const vec3_t *cam, const vec3_t *pos)
{
    return (2048 + camang - getangle(cam->x-pos->x, cam->y-pos->y))&2047;
}

static int32_t S_CalcDistAndAng(int32_t i, int32_t num, int32_t camsect, int32_t camang,
                                const vec3_t *cam, const vec3_t *pos,
                                int32_t *sndistptr, int32_t *sndangptr)
{
    int32_t sndang = 0, sndist = 0;
    int32_t explosion = 0;

    if (PN(i) == APLAYER && P_Get(i) == screenpeek)
        goto sound_further_processing;

    sndang = S_GetAngle(camang, cam, pos);
    sndist = FindDistance3D(cam->x-pos->x, cam->y-pos->y, (cam->z-pos->z));

#ifdef SPLITSCREEN_MOD_HACKS
    if (g_fakeMultiMode==2)
    {
        // HACK for splitscreen mod: take the min of sound distances
        // to 1st and 2nd player.

        if (PN(i) == APLAYER && P_Get(i) == 1)
        {
            sndist = sndang = 0;
            goto sound_further_processing;
        }

        {
            const vec3_t *cam2 = &g_player[1].ps->pos;
            int32_t sndist2 = FindDistance3D(cam2->x-pos->x, cam2->y-pos->y, (cam2->z-pos->z));

            if (sndist2 < sndist)
            {
                cam = cam2;
                camsect = g_player[1].ps->cursectnum;
                camang = g_player[1].ps->ang;

                sndist = sndist2;
                sndang = S_GetAngle(camang, cam, pos);
            }
        }
    }
#endif

    if ((g_sounds[num].m & SF_GLOBAL) == 0 && S_IsAmbientSFX(i) && (sector[SECT(i)].lotag&0xff) < 9)  // ST_9_SLIDING_ST_DOOR
        sndist = divscale14(sndist, SHT(i)+1);

sound_further_processing:
    sndist += g_sounds[num].vo;
    if (sndist < 0)
        sndist = 0;

    if (camsect > -1 && sndist && PN(i) != MUSICANDSFX &&
            !cansee(cam->x,cam->y,cam->z-(24<<8),camsect, SX(i),SY(i),SZ(i)-(24<<8),SECT(i)))
        sndist += sndist>>5;

#ifndef EDUKE32_STANDALONE
    switch (DYNAMICSOUNDMAP(num))
    {
    case PIPEBOMB_EXPLODE__STATIC:
    case LASERTRIP_EXPLODE__STATIC:
    case RPG_EXPLODE__STATIC:
        explosion = 1;
        if (sndist > 6144)
            sndist = 6144;
        break;
    }
#endif

    if ((g_sounds[num].m & SF_GLOBAL) || sndist < ((255-LOUDESTVOLUME)<<6))
        sndist = ((255-LOUDESTVOLUME)<<6);

    *sndistptr = sndist;
    *sndangptr = sndang;

    return explosion;
}

int32_t S_PlaySound3D(int32_t num, int32_t i, const vec3_t *pos)
{
    int32_t j = VM_OnEventWithReturn(EVENT_SOUND, i, screenpeek, num);

    if ((j == -1 && num != -1) || !ud.config.SoundToggle) // check that the user returned -1, but only if -1 wasn't playing already (in which case, warn)
        return -1;

    int const sndNum = j;
    sound_t &snd = g_sounds[sndNum];

    if (EDUKE32_PREDICT_FALSE((unsigned) sndNum > (unsigned) g_highestSoundIdx || snd.filename == NULL || snd.ptr == NULL))
    {
        OSD_Printf("WARNING: invalid sound #%d\n", num);
        return -1;
    }

    const DukePlayer_t *const pPlayer = g_player[myconnectindex].ps;

    if (((snd.m & SF_ADULT) && ud.lockout) || (unsigned)i >= MAXSPRITES || (pPlayer->gm & MODE_MENU) || !FX_VoiceAvailable(snd.pr)
        || (pPlayer->timebeforeexit > 0 && pPlayer->timebeforeexit <= GAMETICSPERSEC * 3))
        return -1;

    // Duke talk
    if (snd.m & SF_TALK)
    {
        if ((g_netServer || ud.multimode > 1) && PN(i) == APLAYER && P_Get(i) != screenpeek) // other player sound
        {
            if ((ud.config.VoiceToggle & 4) != 4)
                return -1;
        }
        else if ((ud.config.VoiceToggle & 1) != 1)
            return -1;

        // don't play if any Duke talk sounds are already playing
        for (j = 0; j <= g_highestSoundIdx; ++j)
            if ((g_sounds[j].m & SF_TALK) && g_sounds[j].num > 0)
                return -1;
    }
    else if (snd.m & SF_DTAG)  // Duke-Tag sound
    {
        int const voice = S_PlaySound(sndNum);

        if (voice <= FX_Ok)
            return -1;

        j = 0;
        while (j < MAXSOUNDINSTANCES && snd.instances[j].voice != voice)
            j++;

#ifdef DEBUGGINGAIDS
        if (EDUKE32_PREDICT_FALSE(j >= MAXSOUNDINSTANCES))
        {
            OSD_Printf(OSD_ERROR "%s %d: WTF?\n", __FILE__, __LINE__);
            return -1;
        }
#endif

        snd.instances[j].spriteNum = i;

        return voice;
    }

    int32_t sndist, sndang;
    int const explosionp = S_CalcDistAndAng(i, sndNum, CAMERA(sect), fix16_to_int(CAMERA(q16ang)), &CAMERA(pos), pos, &sndist, &sndang);
    int       pitch      = S_GetPitch(sndNum);

    DukePlayer_t const *const pOther = g_player[screenpeek].ps;

#ifdef SPLITSCREEN_MOD_HACKS
    if (g_fakeMultiMode==2)
    {
        // splitscreen HACK
        if (g_player[1].ps->i == i)
            pOther = g_player[1].ps;
}
#endif

    if (pOther->sound_pitch)
        pitch += pOther->sound_pitch;

    if (explosionp)
    {
        if (pOther->cursectnum > -1 && sector[pOther->cursectnum].lotag == ST_2_UNDERWATER)
            pitch -= 1024;
    }
    else
    {
        if (sndist > 32767 && PN(i) != MUSICANDSFX && (snd.m & (SF_LOOP|SF_MSFX)) == 0)
            return -1;

        if (pOther->cursectnum > -1 && sector[pOther->cursectnum].lotag == ST_2_UNDERWATER
            && (snd.m & SF_TALK) == 0)
            pitch = -768;
    }

    if (snd.num > 0 && PN(i) != MUSICANDSFX)
        S_StopEnvSound(sndNum, i);

    if (++g_soundlocks[sndNum] < 200)
        g_soundlocks[sndNum] = 200;

    int const sndSlot = S_GetSlot(sndNum);

    if (sndSlot >= MAXSOUNDINSTANCES)
    {
        g_soundlocks[sndNum]--;
        return -1;
    }

    int const repeatp = (snd.m & SF_LOOP);

    if (repeatp && (snd.m & SF_ONEINST_INTERNAL) && snd.num > 0)
    {
        g_soundlocks[sndNum]--;
        return -1;
    }

    int voice;
    int const ambsfxp = S_IsAmbientSFX(i);

    if (repeatp && !ambsfxp)
    {
        // XXX: why is 'right' 0?
        voice = FX_Play(snd.ptr, snd.soundsiz, 0, -1, pitch, FX_VOLUME(sndist >> 6), FX_VOLUME(sndist >> 6), 0, snd.pr,
                        (sndNum * MAXSOUNDINSTANCES) + sndSlot);
    }
    else
    {
        // Ambient MUSICANDSFX always start playing using the 3D routines!
        voice = FX_Play3D(snd.ptr, snd.soundsiz, repeatp ? FX_LOOP : FX_ONESHOT, pitch, sndang >> 4, FX_VOLUME(sndist >> 6), snd.pr,
            (sndNum * MAXSOUNDINSTANCES) + sndSlot);
    }

    if (voice <= FX_Ok)
    {
        g_soundlocks[sndNum]--;
        return -1;
    }

    snd.num++;
    snd.instances[sndSlot].spriteNum = i;
    snd.instances[sndSlot].voice     = voice;
    snd.instances[sndSlot].sndist    = sndist >> 6;
    snd.instances[sndSlot].clock     = 0;

    return voice;
}

int32_t S_PlaySound(int32_t num)
{
    int32_t sndnum = VM_OnEventWithReturn(EVENT_SOUND, g_player[screenpeek].ps->i, screenpeek, num);

    if ((sndnum == -1 && num != -1) || !ud.config.SoundToggle) // check that the user returned -1, but only if -1 wasn't playing already (in which case, warn)
        return -1;

    num = sndnum;

    sound_t & snd = g_sounds[num];

    if (EDUKE32_PREDICT_FALSE((unsigned)num > (unsigned)g_highestSoundIdx || snd.filename == NULL || snd.ptr == NULL))
    {
        OSD_Printf("WARNING: invalid sound #%d\n",num);
        return -1;
    }

    if ((!(ud.config.VoiceToggle & 1) && (snd.m & SF_TALK)) || ((snd.m & SF_ADULT) && ud.lockout) || !FX_VoiceAvailable(snd.pr))
        return -1;

    int32_t pitch = S_GetPitch(num);

    if (++g_soundlocks[num] < 200)
        g_soundlocks[num] = 200;

    sndnum = S_GetSlot(num);

    if (sndnum >= MAXSOUNDINSTANCES)
    {
        g_soundlocks[num]--;
        return -1;
    }

    int32_t voice;

    if (snd.m & SF_LOOP)
        voice = FX_Play(snd.ptr, snd.soundsiz, 0, -1, pitch, FX_VOLUME(LOUDESTVOLUME), FX_VOLUME(LOUDESTVOLUME), FX_VOLUME(LOUDESTVOLUME),
                        snd.soundsiz, (num * MAXSOUNDINSTANCES) + sndnum);
    else
        voice = FX_Play3D(snd.ptr, snd.soundsiz, FX_ONESHOT, pitch, 0, FX_VOLUME(255 - LOUDESTVOLUME), snd.pr, (num * MAXSOUNDINSTANCES) + sndnum);

    if (voice <= FX_Ok)
    {
        g_soundlocks[num]--;
        return -1;
    }

    snd.num++;
    snd.instances[sndnum].spriteNum = -1;
    snd.instances[sndnum].voice = voice;
    snd.instances[sndnum].sndist = 255-LOUDESTVOLUME;
    snd.instances[sndnum].clock = 0;

    return voice;
}

int32_t A_PlaySound(uint32_t num, int32_t i)
{
    if (EDUKE32_PREDICT_FALSE((unsigned)num > (unsigned)g_highestSoundIdx)) return -1;

    return (unsigned)i >= MAXSPRITES ? S_PlaySound(num) :
        S_PlaySound3D(num, i, (vec3_t *)&sprite[i]);
}

void S_StopEnvSound(int32_t num, int32_t i)
{
    if (EDUKE32_PREDICT_FALSE((unsigned)num > (unsigned)g_highestSoundIdx) || g_sounds[num].num <= 0)
        return;

    int32_t j;

    do
    {
        for (j=0; j<MAXSOUNDINSTANCES; ++j)
        {
            if ((i == -1 && g_sounds[num].instances[j].voice > FX_Ok) || (i != -1 && g_sounds[num].instances[j].spriteNum == i))
            {
#ifdef DEBUGGINGAIDS
                if (EDUKE32_PREDICT_FALSE(i >= 0 && g_sounds[num].instances[j].voice <= FX_Ok))
                    initprintf(OSD_ERROR "S_StopEnvSound(): bad voice %d for sound ID %d index %d!\n", g_sounds[num].instances[j].voice, num, j);
                else
#endif
                if (g_sounds[num].instances[j].voice > FX_Ok)
                {
                    FX_StopSound(g_sounds[num].instances[j].voice);
                    S_Cleanup();
                    break;
                }
            }
        }
    }
    while (j < MAXSOUNDINSTANCES);
}

// Do not remove this or make it inline.
void S_StopAllSounds(void)
{
    FX_StopAllSounds();
}

void S_ChangeSoundPitch(int32_t num, int32_t i, int32_t pitchoffset)
{
    if ((unsigned)num > (unsigned)g_highestSoundIdx || g_sounds[num].num <= 0)
        return;

    for (int j=0; j<MAXSOUNDINSTANCES; ++j)
    {
        int const voice = g_sounds[num].instances[j].voice;

        if ((i == -1 && voice > FX_Ok) || (i != -1 && g_sounds[num].instances[j].spriteNum == i))
        {
            if (EDUKE32_PREDICT_FALSE(i >= 0 && voice <= FX_Ok))
                initprintf(OSD_ERROR "S_ChangeSoundPitch(): bad voice %d for sound ID %d index %d!\n", voice, num, j);
            else if (voice > FX_Ok && FX_SoundActive(voice))
                FX_SetPitch(voice, pitchoffset);
            break;
        }
    }
}

void S_Update(void)
{
    S_Cleanup();

    if ((g_player[myconnectindex].ps->gm & (MODE_GAME|MODE_DEMO)) == 0)
        return;

    g_numEnvSoundsPlaying = 0;

    const vec3_t *c;
    int32_t ca,cs;

    if (ud.camerasprite == -1)
    {
        c = &CAMERA(pos);
        cs = CAMERA(sect);
        ca = fix16_to_int(CAMERA(q16ang));
    }
    else
    {
        c = (vec3_t *)&sprite[ud.camerasprite];
        cs = sprite[ud.camerasprite].sectnum;
        ca = sprite[ud.camerasprite].ang;
    }

    int       sndnum  = 0;
    int const highest = g_highestSoundIdx;

    do
    {
        if (g_sounds[sndnum].num == 0)
            continue;

        for (bssize_t k=0; k<MAXSOUNDINSTANCES; ++k)
        {
            sndinst_t & sndinst = g_sounds[sndnum].instances[k];
            int const spriteNum = sndinst.spriteNum;

            if ((unsigned)spriteNum >= MAXSPRITES || sndinst.voice <= FX_Ok || !FX_SoundActive(sndinst.voice))
                continue;

            int32_t sndist, sndang;

            S_CalcDistAndAng(spriteNum, sndnum, cs, ca, c, (const vec3_t *)&sprite[spriteNum], &sndist, &sndang);

            if (S_IsAmbientSFX(spriteNum))
                g_numEnvSoundsPlaying++;

            // AMBIENT_SOUND
            FX_Pan3D(sndinst.voice, sndang>>4, FX_VOLUME(sndist>>6));
            sndinst.sndist = sndist>>6;
            sndinst.clock++;
        }
    }
    while (++sndnum <= highest);
}

void S_Callback(uint32_t num)
{
    if ((int32_t)num == MUSIC_ID)
        return;

    dq[dnum & (DQSIZE - 1)] = num;
    dnum++;
}

void S_ClearSoundLocks(void)
{
#ifdef CACHING_DOESNT_SUCK
    int32_t i;
    int32_t const msp = g_highestSoundIdx;

    for (native_t i = 0; i < 11; ++i)
        if (rts_lumplockbyte[i] >= 200)
            rts_lumplockbyte[i] = 199;

    int32_t const msp = g_highestSoundIdx;

    for (native_t i = 0; i <= msp; ++i)
        if (g_soundlocks[i] >= 200)
            g_soundlocks[i] = 199;
#endif
}

int32_t A_CheckSoundPlaying(int32_t i, int32_t num)
{
    if (EDUKE32_PREDICT_FALSE((unsigned)num > (unsigned)g_highestSoundIdx)) return 0;

    if (g_sounds[num].num > 0 && i >= 0)
    {
        for (bssize_t j=0; j<MAXSOUNDINSTANCES; ++j)
            if (g_sounds[num].instances[j].spriteNum == i)
                return 1;
    }

    return (i == -1) ? g_sounds[num].num : 0;
}

// Check if actor <i> is playing any sound.
int32_t A_CheckAnySoundPlaying(int32_t i)
{
    int32_t const msp = g_highestSoundIdx;

    for (native_t j=0; j<=msp; ++j)
    {
        for (native_t k=0; k<MAXSOUNDINSTANCES; ++k)
            if (g_sounds[j].instances[k].spriteNum == i)
                return 1;
    }

    return 0;
}

int32_t S_CheckSoundPlaying(int32_t i, int32_t num)
{
    if (EDUKE32_PREDICT_FALSE((unsigned)num > (unsigned)g_highestSoundIdx)) return 0;
    return (i == -1) ? (g_soundlocks[num] > 200) : g_sounds[num].num;
}
