// Windows DIB/DirectDraw interface layer
// for the Build Engine
// by Jonathon Fowler (jonof@edgenetwk.com)
//
// This is all very ugly.
//
// Written for DirectX 6.0

#define DIRECTINPUT_VERSION 0x0500
#define DIRECTDRAW_VERSION  0x0600

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <ddraw.h>
#include <dinput.h>

#include "dxdidf.h"	// comment this out if c_dfDI* is being reported as multiply defined
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>

#include "winlayer.h"
#include "pragmas.h"
#include "build.h"
#include "a.h"
#include "osd.h"
#include "compat.h"

extern char moustat;

// undefine to restrict windowed resolutions to conventional sizes
#define ANY_WINDOWED_SIZE


// Windows crud
       HINSTANCE hInstance = 0;
       HWND hWindow = 0;
#define WINDOW_STYLE (WS_POPUP|WS_CAPTION|WS_SYSMENU|WS_MINIMIZEBOX)
extern long app_main(long argc, char *argv[]);
static BOOL window_class_registered = FALSE;
int   _buildargc = 1;
char *_buildargv[30] = { "buildapp" };

// DirectInput objects
static HMODULE               hDInputDLL    = NULL;
static LPDIRECTINPUTA        lpDI          = NULL;
static LPDIRECTINPUTDEVICE2A lpDIDKeyboard = NULL;
static LPDIRECTINPUTDEVICE2A lpDIDMouse    = NULL;
static BOOL                  bDInputInited = FALSE;
#define INPUT_BUFFER_SIZE	32

// DirectDraw objects
static HMODULE              hDDrawDLL      = NULL;
static LPDIRECTDRAW         lpDD           = NULL;
static LPDIRECTDRAWSURFACE  lpDDSPrimary   = NULL;
static LPDIRECTDRAWSURFACE  lpDDSBack      = NULL;
static LPDIRECTDRAWSURFACE  lpDDSOffscreen = NULL;
static LPDIRECTDRAWPALETTE  lpDDPalette = NULL;
static BOOL                 bDDrawInited = FALSE;

// DIB stuff
static HDC      hDC         = NULL;
static HDC      hDCSection  = NULL;
static HBITMAP  hDIBSection = NULL;
static HPALETTE hPalette    = NULL;
static VOID    *lpPixels    = NULL;

#define NUM_SYS_COLOURS	25
static int syscolouridx[NUM_SYS_COLOURS] = {
	COLOR_SCROLLBAR,		// 1
	COLOR_BACKGROUND,
	COLOR_ACTIVECAPTION,
	COLOR_INACTIVECAPTION,
	COLOR_MENU,
	COLOR_WINDOW,
	COLOR_WINDOWFRAME,
	COLOR_MENUTEXT,
	COLOR_WINDOWTEXT,
	COLOR_CAPTIONTEXT,		// 10
	COLOR_ACTIVEBORDER,
	COLOR_INACTIVEBORDER,
	COLOR_APPWORKSPACE,
	COLOR_HIGHLIGHT,
	COLOR_HIGHLIGHTTEXT,
	COLOR_BTNFACE,
	COLOR_BTNSHADOW,
	COLOR_GRAYTEXT,
	COLOR_BTNTEXT,
	COLOR_INACTIVECAPTIONTEXT,	// 20
	COLOR_BTNHIGHLIGHT,
	COLOR_3DDKSHADOW,
	COLOR_3DLIGHT,
	COLOR_INFOTEXT,
	COLOR_INFOBK			// 25
};
static DWORD syscolours[NUM_SYS_COLOURS];
static char system_colours_saved = 0, bw_colours_set = 0;

static const LPTSTR GetWindowsErrorMsg(DWORD code);
static const char * GetDDrawError(HRESULT code);
static const char * GetDInputError(HRESULT code);
static void ShowErrorBox(const char *m);
static void ShowDDrawErrorBox(const char *m, HRESULT r);
static void ShowDInputErrorBox(const char *m, HRESULT r);
static BOOL CheckWinVersion(void);
static LRESULT CALLBACK WndProcCallback(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);
static BOOL InitDirectDraw(void);
static void UninitDirectDraw(void);
static void ReleaseSurfaces(void);
static BOOL InitDirectInput(void);
static void UninitDirectInput(void);
static void AcquireInputDevices(char acquire);
static void ProcessInputDevices(void);
static void UninitDIB(void);
static BOOL RegisterWindowClass(void);
static BOOL CreateAppWindow(int width, int height, int fs);
static void DestroyAppWindow(void);
static void SaveSystemColours(void);
static void SetBWSystemColours(void);
static void RestoreSystemColours(void);


// timer
static int64 timerfreq=0;
static long timerlastsample=0;
static int timerticspersec=0;
static void (*usertimercallback)(void) = NULL;


// video
static long desktopxdim, desktopydim, desktoppal;
long xres=-1, yres=-1, fullscreen=0, bytesperline=0, imageSize=0;
long frameplace=0, lockcount=0;
static PALETTEENTRY curPalette[256];
static int windowposx, windowposy;
char modechange=1, repaintneeded=0;
char offscreenrendering=0;


// input and events
char inputdevices=0;
char quitevent=0, appactive=1;
short mousex=0, mousey=0, mouseb=0,mouseba[2], mousegrab=1;

static char taskswitching=1;

char keystatus[256], keyfifo[KEYFIFOSIZ], keyfifoplc, keyfifoend;
static unsigned long lastKeyDown = 0;
static unsigned long lastKeyTime = 0;

// I don't see any pressing need to store the key-up events yet
#define SetKey(key,state) { \
	keystatus[key] = state; \
		if (state) { \
	keyfifo[keyfifoend] = key; \
	keyfifo[(keyfifoend+1)&(KEYFIFOSIZ-1)] = state; \
	keyfifoend = ((keyfifoend+2)&(KEYFIFOSIZ-1)); \
		} \
}

#define NUM_INPUTS 2
static HANDLE inputevt[NUM_INPUTS] = {0,0};

void (*keypresscallback)(long,long) = 0;
void (*mousepresscallback)(long,long) = 0;




//-------------------------------------------------------------------------------------------------
//  MAIN CRAP
//=================================================================================================


//
// allowtaskswitching() -- captures/releases alt+tab hotkeys
void allowtaskswitching(int onf)
{
	if (onf == taskswitching) return;

	if (onf) {
		UnregisterHotKey(0,0);
		UnregisterHotKey(0,1);
	} else {
		RegisterHotKey(0,0,MOD_ALT,VK_TAB);
		RegisterHotKey(0,1,MOD_ALT|MOD_SHIFT,VK_TAB);
	}

	taskswitching = onf;
}


//
// SignalHandler() -- called when we've sprung a leak
//
static void SignalHandler(int signum)
{
	switch (signum) {
		case SIGSEGV:
			printOSD("Fatal Signal caught: SIGSEGV. Bailing out.\n");
			uninitsystem();
			if (stdout) fclose(stdout);
			break;
		default:
			break;
	}
}


//
// WinMain() -- main Windows entry point
//
int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
	int r;
	char *argp;
	FILE *fp;
	HDC hdc;

	hInstance = hInst;

	if (CheckWinVersion() || hPrevInst) {
		MessageBox(0, "This application must be run under Windows 95/98/Me or Windows 2000/XP or better.",
			apptitle, MB_OK|MB_ICONSTOP);
		return -1;
	}

	hdc = GetDC(NULL);
	r = GetDeviceCaps(hdc, BITSPIXEL);
	ReleaseDC(NULL, hdc);
	if (r < 8) {
		MessageBox(0, "This application requires a desktop colour depth of 256-colours or more.",
			apptitle, MB_OK|MB_ICONSTOP);
		return -1;
	}

	// carve up the commandline into more recognizable pieces
	argp = lpCmdLine;
	while (*argp && (_buildargc < 30)) {
		while (((*argp <= 32) || (*argp > 126)) && (*argp != 0)) argp++;
		if (*argp == 0) break;

		_buildargv[_buildargc++] = argp;

		while ((*argp > 32) && (*argp <= 126)) argp++;
		if (*argp == 0) break;

		*(argp++) = 0;
	}

	// pipe standard outputs to files
	fp = freopen("stdout.txt", "w", stdout);
	if (!fp) {
		fp = fopen("stdout.txt", "w");
	}
	if (fp) setvbuf(fp, 0, _IONBF, 0);
	*stdout = *fp;

	// install signal handlers
	signal(SIGSEGV, SignalHandler);

	if (RegisterWindowClass()) return -1;

#ifdef DISCLAIMER
	MessageBox(0,
		DISCLAIMER,
		"Notice",
		MB_OK|MB_ICONINFORMATION);
#endif

	atexit(uninitsystem);

	r = app_main(_buildargc, _buildargv);

	fclose(stdout);

	return r;
}


//
// initsystem() -- init systems
//
int initsystem(void)
{
	HDC hdc;
	int i;

	printOSD("Initializing Windows DirectX/GDI system interface\n");

	hdc = GetDC(NULL);

	// get the desktop dimensions before anything changes them
	desktopxdim = GetDeviceCaps(hdc, HORZRES);
	desktopydim = GetDeviceCaps(hdc, VERTRES);
	desktoppal  = ((GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE) == RC_PALETTE);

	ReleaseDC(NULL, hdc);

	if (desktoppal)
		// save the system colours
		SaveSystemColours();

	memset(curPalette, 0, sizeof(PALETTEENTRY) * 256);

	atexit(uninitsystem);

	frameplace=0;
	lockcount=0;

	// try and start DirectDraw
	if (InitDirectDraw())
		printOSD("DirectDraw initialization failed. Fullscreen modes will be unavailable.\n");

	return 0;
}


//
// uninitsystem() -- uninit systems
//
void uninitsystem(void)
{
	DestroyAppWindow();

	uninitinput();
	uninittimer();

	allowtaskswitching(1);
}


//
// handleevents() -- process the Windows message queue
//   returns !0 if there was an important event worth checking (like quitting)
//
int handleevents(void)
{
	int rv=0;
	MSG msg;

	//if (frameplace && fullscreen) printf("Offscreen buffer is locked!\n");

	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
		if (msg.message == WM_QUIT)
			quitevent = 1;
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	ProcessInputDevices();

	if (!appactive || quitevent) rv = -1;

	sampletimer();

	if (repaintneeded) {
		showframe(0);
		repaintneeded=0;
	}

	return rv;
}







//-------------------------------------------------------------------------------------------------
//  INPUT (MOUSE/KEYBOARD/JOYSTICK?)
//=================================================================================================

//
// initinput() -- init input system
//
int initinput(void)
{
	moustat=0;
	memset(keystatus, 0, sizeof(keystatus));
	keyfifoplc = keyfifoend = 0;

	inputdevices = 0;

	if (InitDirectInput())
		return -1;

	AcquireInputDevices(1);

	return 0;
}


//
// uninitinput() -- uninit input system
//
void uninitinput(void)
{
	uninitmouse();
	UninitDirectInput();
}


//
// setkeypresscallback() -- sets a callback which gets notified when keys are pressed
//
void setkeypresscallback(void (*callback)(long, long))
{
	keypresscallback = callback;
}

//
// setmousepresscallback() -- sets a callback which gets notified when a mouse button is pressed
//
void setmousepresscallback(void (*callback)(long, long))
{
	mousepresscallback = callback;
}


//
// initmouse() -- init mouse input
//
int initmouse(void)
{
	if (moustat) return 0;

	printOSD("Initializing mouse\n");

	// grab input
	moustat=1;
	grabmouse(1);

	return 0;
}


//
// uninitmouse() -- uninit mouse input
//
void uninitmouse(void)
{
	if (!moustat) return;

	grabmouse(0);
	moustat=0;
}


//
// grabmouse() -- show/hide mouse cursor
//
void grabmouse(char a)
{
	if (!moustat) return;

	mousegrab = a;

	AcquireInputDevices(a);

	mousex = 0;
	mousey = 0;
	mouseb = 0;
}


//
// readmousexy() -- return mouse motion information
//
void readmousexy(short *x, short *y)
{
	*x = mousex;
	*y = mousey;
	mousex = 0;
	mousey = 0;
}


//
// readmousebstatus() -- return mouse button information
//
void readmousebstatus(short *b)
{
	*b = mouseb;
}







//-------------------------------------------------------------------------------------------------
//  TIMER
//=================================================================================================

//  This timer stuff is all Ken's idea.

//
// installusertimercallback() -- set up a callback function to be called when the timer is fired
//
void (*installusertimercallback(void (*callback)(void)))(void)
{
	void (*oldtimercallback)(void);

	oldtimercallback = usertimercallback;
	usertimercallback = callback;

	return oldtimercallback;
}


//
// inittimer() -- initialize timer
//
int inittimer(int tickspersecond)
{
	int64 t;
	
	if (timerfreq) return 0;	// already installed

	printOSD("Initializing timer\n");

	QueryPerformanceFrequency((LARGE_INTEGER*)&timerfreq);
	timerticspersec = tickspersecond;
	QueryPerformanceCounter((LARGE_INTEGER*)&t);
	timerlastsample = (long)(t*timerticspersec / timerfreq);

	usertimercallback = NULL;

	return 0;
}

//
// uninittimer() -- shut down timer
//
void uninittimer(void)
{
	if (!timerfreq) return;

	timerfreq=0;
}

//
// sampletimer() -- update totalclock
//
void sampletimer(void)
{
	int64 i;
	long n;
	
	if (!timerfreq) return;

	QueryPerformanceCounter((LARGE_INTEGER*)&i);
	n = (long)(i*timerticspersec / timerfreq) - timerlastsample;
	if (n>0) {
		totalclock += n;
		timerlastsample += n;
	}

	if (usertimercallback) for (; n>0; n--) usertimercallback();
}


//
// getticks() -- returns the windows ticks count
//
unsigned long getticks(void)
{
	int64 i;
	QueryPerformanceCounter((LARGE_INTEGER*)&i);
	return (unsigned long)((i*0xffffffffll/timerfreq)&0xffffffffu);
}




//-------------------------------------------------------------------------------------------------
//  VIDEO
//=================================================================================================

//
// checkvideomode() -- makes sure the video mode passed is legal
//
int checkvideomode(long *x, long *y, int fs)
{
	int i, nearest=-1, dx, dy, odx=9999, ody=9999;

	getvalidmodes();

	// fix up the passed resolution values to be multiples of 8
	// and at least 320x200 or at most MAXXDIMxMAXYDIM
	if (*x < 320) *x = 320;
	if (*y < 200) *y = 200;
	if (*x > MAXXDIM) *x = MAXXDIM;
	if (*y > MAXYDIM) *y = MAXYDIM;
	*x &= 0xfffffff8l;

#ifdef ANY_WINDOWED_SIZE
	if (fs) {
#endif
		for (i=0; i<validmodecnt; i++) {
			if (validmodefs[i] != fs) continue;
			dx = klabs(validmodexdim[i] - *x);
			dy = klabs(validmodeydim[i] - *y);
			if (!(dx | dy)) { 	// perfect match
				nearest = i;
				break;
			}
			if ((dx <= odx) && (dy <= ody)) {
				nearest = i;
				odx = dx; ody = dy;
			}
		}

		if (nearest < 0) {
			// no mode that will match (eg. if no fullscreen modes)
			return -1;
		}
		*x = validmodexdim[nearest];
		*y = validmodeydim[nearest];
#ifdef ANY_WINDOWED_SIZE
	}
#endif

	return 0;
}


//
// setvideomode() -- set the video mode
//
int setvideomode(int x, int y, int fs)
{
	char t[384];
	HRESULT result;
	DDSURFACEDESC ddsd;

	if ((fs == fullscreen) && (x == xres) && (x == yres)) return 0;

	AcquireInputDevices(0);

	printOSD("Setting video mode %dx%d (%s)\n", x,y, ((fs) ? "fullscreen" : "windowed"));
	if (CreateAppWindow(x, y, fs)) return -1;

	sprintf(t, "%s (%dx%d %s)", apptitle, x, y, ((fs) ? "fullscreen" : "windowed"));
	SetWindowText(hWindow, t);

	AcquireInputDevices(1);
	modechange=1;

	return 0;
}


//
// getvalidmodes() -- figure out what video modes are available
//
#define ADDMODE(x,y,f) { \
	validmodexdim[validmodecnt]=x; \
	validmodeydim[validmodecnt]=y; \
	validmodefs[validmodecnt]=f; \
	validmodecnt++; \
	printOSD("Adding mode %dx%d (%s)\n", x, y, (f)?"fullscreen":"windowed"); }

#define CHECK(w,h) if ((w <= maxx) && (h <= maxy))

// mode enumerator
static HRESULT WINAPI getvalidmodes_enum(DDSURFACEDESC *ddsd, VOID *udata)
{
	unsigned maxx = MAXXDIM, maxy = MAXYDIM;

	if (ddsd->ddpfPixelFormat.dwRGBBitCount == 8) {
		CHECK(ddsd->dwWidth, ddsd->dwHeight) ADDMODE(ddsd->dwWidth, ddsd->dwHeight, 1);
	}

	return(DDENUMRET_OK);
}

void getvalidmodes(void)
{
	static int modeschecked=0;
	int i, maxx=0, maxy=0;
	HRESULT result;

	if (modeschecked) return;

	validmodecnt=0;

	if (bDDrawInited) {
		// if DirectDraw initialization didn't fail enumerate fullscreen modes

		result = IDirectDraw_EnumDisplayModes(lpDD, 0, NULL, 0, getvalidmodes_enum);
		if (result != DD_OK) {
			printOSD("Unable to enumerate fullscreen modes. Using default list.\n");
			ADDMODE(1280,1024,1)
			ADDMODE(1280,960,1)
			ADDMODE(1152,864,1)
			ADDMODE(1024,768,1)
			ADDMODE(800,600,1)
			ADDMODE(640,480,1)
			ADDMODE(640,400,1)
			ADDMODE(512,384,1)
			ADDMODE(480,360,1)
			ADDMODE(400,300,1)
			ADDMODE(320,240,1)
			ADDMODE(320,200,1)
		}
	}

	// windowed modes cant be bigger than the current desktop resolution
	maxx = desktopxdim-1;
	maxy = desktopydim-1;

	// add windowed modes next
	CHECK(320,200) ADDMODE(320,200,0)
	CHECK(320,240) ADDMODE(320,240,0)
	CHECK(400,300) ADDMODE(400,300,0)
	CHECK(480,360) ADDMODE(480,360,0)
	CHECK(512,384) ADDMODE(512,384,0)
	CHECK(640,400) ADDMODE(640,400,0)
	CHECK(640,480) ADDMODE(640,480,0)
	CHECK(800,600) ADDMODE(800,600,0)
	CHECK(1024,768) ADDMODE(1024,768,0)
	CHECK(1152,864) ADDMODE(1152,864,0)
	CHECK(1280,960) ADDMODE(1280,960,0)
	CHECK(1280,1024) ADDMODE(1280,1024,0)

	modeschecked=1;
}

#undef CHECK
#undef ADDMODE


//
// begindrawing() -- locks the framebuffer for drawing
//
void begindrawing(void)
{
	HRESULT result;
	DDSURFACEDESC ddsd;
	long i,j;

	if (lockcount++ > 0)
		return;		// already locked

	if (offscreenrendering) return;
	
	if (!fullscreen) {
		frameplace = (long)lpPixels;
		if (xres != bytesperline || modechange) {
			bytesperline = xres|4;
			imageSize = bytesperline*yres;
			setvlinebpl(bytesperline);

			j = 0;
			for(i=0;i<=ydim;i++) ylookup[i] = j, j += bytesperline;
			modechange=0;
		}
	} else {
		memset(&ddsd, 0, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);

		result = IDirectDrawSurface_Lock(lpDDSOffscreen, NULL, &ddsd, DDLOCK_NOSYSLOCK | DDLOCK_WAIT, NULL);
		if (result == DDERR_SURFACELOST) {
			IDirectDrawSurface_Restore(lpDDSOffscreen);
			result = IDirectDrawSurface_Lock(lpDDSOffscreen, NULL, &ddsd, DDLOCK_NOSYSLOCK | DDLOCK_WAIT, NULL);
		}
		if (result != DD_OK) {
			printOSD("Failed locking offscreen surface: %s\n", GetDDrawError(result));
			frameplace = 0;
			return;
		}

		frameplace = (long)ddsd.lpSurface;

		if (ddsd.lPitch != bytesperline || modechange) {
			bytesperline = ddsd.lPitch;
			imageSize = bytesperline*yres;
			setvlinebpl(bytesperline);

			j = 0;
			for(i=0;i<=ydim;i++) ylookup[i] = j, j += bytesperline;
			modechange=0;
		}
	}
}


//
// enddrawing() -- unlocks the framebuffer
//
void enddrawing(void)
{
	HRESULT result;

	if (!frameplace) return;
	if (lockcount > 1) { lockcount--; return; }
	if (!offscreenrendering) frameplace = 0;
	if (lockcount == 0) return;
	lockcount = 0;

	if (offscreenrendering) return;

	if (fullscreen) {
		result = IDirectDrawSurface_Unlock(lpDDSOffscreen, NULL);
		if (result != DD_OK) {
			printOSD("Failed unlocking offscreen surface: %s\n", GetDDrawError(result));
			return;
		}
	}
}


//
// showframe() -- update the display
//
void showframe(int w)
{
	HRESULT result;
	RECT r;

	if (offscreenrendering) return;

	if (lockcount) {
		printf("Frame still locked %d times when showframe() called.\n", lockcount);
		while (lockcount) enddrawing();
	}

	if (!fullscreen) {
		BitBlt(hDC, 0, 0, xres, yres, hDCSection, 0, 0, SRCCOPY);
	} else {
		if (!w)
			if ((result = IDirectDrawSurface_GetBltStatus(lpDDSBack, DDGBS_CANBLT)) == DDERR_WASSTILLDRAWING)
				return;

		r.left = 0;
		r.top = 0;
		r.right = xres;
		r.bottom = yres;
		result = IDirectDrawSurface_BltFast(lpDDSBack, 0, 0, lpDDSOffscreen, &r, DDBLTFAST_NOCOLORKEY| w?DDBLTFAST_WAIT:0);
		if (result == DDERR_SURFACELOST) {
			IDirectDrawSurface_Restore(lpDDSPrimary);
			result = IDirectDrawSurface_BltFast(lpDDSBack, 0, 0, lpDDSOffscreen, &r, DDBLTFAST_NOCOLORKEY| w?DDBLTFAST_WAIT:0);
		}

		if (result != DD_OK) {
			if (result == DDERR_SURFACELOST && !appactive)
				return;	// not in a position to restore display anyway
			if (result != DDERR_WASSTILLDRAWING)
				printOSD("IDirectDrawSurface_BltFast(): %s\n", GetDDrawError(result));
		} else {
			// wait for the blit to complete
			if (!w)
				if ((result = IDirectDrawSurface_GetFlipStatus(lpDDSPrimary, DDGFS_CANFLIP)) == DDERR_WASSTILLDRAWING)
					return;
			
			result = IDirectDrawSurface_Flip(lpDDSPrimary, NULL, w?DDFLIP_WAIT:0);
			if (result == DDERR_SURFACELOST) {
				IDirectDrawSurface_Restore(lpDDSPrimary);
				result = IDirectDrawSurface_Flip(lpDDSPrimary, NULL, w?DDFLIP_WAIT:0);
			}
			if (result != DD_OK) {
				if (result == DDERR_SURFACELOST && !appactive)
					return;	// not in a position to restore display anyway
				if (result != DDERR_WASSTILLDRAWING)
					printOSD("IDirectDrawSurface_Flip(): %s\n", GetDDrawError(result));
			}
		}
	}
}


//
// setpalette() -- set palette values
//
int setpalette(int start, int num, char *dapal)
{
	int i, n;
	HRESULT result;
	RGBQUAD *rgb;
	HPALETTE hPalPrev;

	struct logpal {
		WORD palVersion;
		WORD palNumEntries;
		PALETTEENTRY palPalEntry[256];
	} *lpal;

	for (i=start, n=num; n>0; i++, n--) {
		curPalette[i].peBlue = dapal[0] << 2;
		curPalette[i].peGreen = dapal[1] << 2;
		curPalette[i].peRed = dapal[2] << 2;
		curPalette[i].peFlags = PC_RESERVED | PC_NOCOLLAPSE;
		dapal += 4;
	}

	if (!fullscreen) {
		if (num > 0) {
			rgb = (RGBQUAD *)alloca(sizeof(RGBQUAD)*num);
			for (i=start, n=0; n<num; i++, n++) {
				rgb[n].rgbBlue = curPalette[i].peBlue;
				rgb[n].rgbGreen = curPalette[i].peGreen;
				rgb[n].rgbRed = curPalette[i].peRed;
				rgb[n].rgbReserved = 0;
			}

			SetDIBColorTable(hDCSection, start, num, rgb);
		}

		if (!desktoppal) return 0;	// only if an 8bit desktop do we do what follows

		lpal = (struct logpal *)alloca(sizeof(struct logpal));

		memcpy(lpal->palPalEntry, curPalette, sizeof(PALETTEENTRY)*256);

		// set 0 and 255 to black and white
		lpal->palVersion = 0x300;
		lpal->palNumEntries = 256;
		lpal->palPalEntry[0].peBlue = 0;
		lpal->palPalEntry[0].peGreen = 0;
		lpal->palPalEntry[0].peRed = 0;
		lpal->palPalEntry[0].peFlags = 0;
		lpal->palPalEntry[255].peBlue = 255;
		lpal->palPalEntry[255].peGreen = 255;
		lpal->palPalEntry[255].peRed = 255;
		lpal->palPalEntry[255].peFlags = 0;

		if (SetSystemPaletteUse(hDC, SYSPAL_NOSTATIC) == SYSPAL_ERROR) {
			printOSD("Problem setting system palette use.\n");
			return -1;
		}

		if (hPalette) {
			if (num == 0) { start = 0; num = 256; }		// refreshing the palette only
			SetPaletteEntries(hPalette, start, num, lpal->palPalEntry+start);
		} else {
			hPalette = CreatePalette((LOGPALETTE *)lpal);
			if (!hPalette) {
				printOSD("Problem creating palette.\n");
				return -1;
			}
		}

		if (SelectPalette(hDC, hPalette, FALSE) == NULL) {
			printOSD("Problem selecting palette.\n");
			return -1;
		}

		if (RealizePalette(hDC) == GDI_ERROR) {
			printOSD("Failure realizing palette.\n");
			return -1;
		}

		SetBWSystemColours();
	} else {
		if (!lpDDPalette) return -1;
		result = IDirectDrawPalette_SetEntries(lpDDPalette, 0, start, num, &curPalette[start]);
		if (result != DD_OK) {
			printOSD("Palette set failed: %s\n", GetDDrawError(result));
			return -1;
		}
	}

	return 0;
}

//
// getpalette() -- get palette values
//
int getpalette(int start, int num, char *dapal)
{
	int i;

	for (i=num; i>0; i--, start++) {
		dapal[0] = curPalette[start].peBlue >> 2;
		dapal[1] = curPalette[start].peGreen >> 2;
		dapal[2] = curPalette[start].peRed >> 2;
		dapal += 4;
	}

	return 0;
}






//-------------------------------------------------------------------------------------------------
//  MOSTLY STATIC INTERNAL WINDOWS THINGS
//=================================================================================================

//
// ShowErrorBox() -- shows an error message box
//
static void ShowErrorBox(const char *m)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetWindowsErrorMsg(GetLastError()));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// ShowDDrawErrorBox() -- shows an error message box for a DirectDraw error
//
static void ShowDDrawErrorBox(const char *m, HRESULT r)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetDDrawError(r));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// ShowDInputErrorBox() -- shows an error message box for a DirectInput error
//
static void ShowDInputErrorBox(const char *m, HRESULT r)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetDInputError(r));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// CheckWinVersion() -- check to see what version of Windows we happen to be running under
//
static BOOL CheckWinVersion(void)
{
	OSVERSIONINFO osv;

	osv.dwOSVersionInfoSize = sizeof(osv);
	if (!GetVersionEx(&osv)) return TRUE;

	// haha, yeah, like it will work on Win32s
	if (osv.dwPlatformId == VER_PLATFORM_WIN32s) return TRUE;

	// we don't like NT 3.51
	if (osv.dwMajorVersion < 4) return TRUE;

	// nor do we like NT 4
	if (osv.dwPlatformId == VER_PLATFORM_WIN32_NT &&
	    osv.dwMajorVersion == 4) return TRUE;

	return FALSE;
}


//
// WndProcCallback() -- the Windows window callback
//
static LRESULT CALLBACK WndProcCallback(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	RECT rect;
	POINT pt;
	HRESULT result;

	switch (uMsg) {
		case WM_ACTIVATE:
			appactive = (LOWORD(wParam) != WA_INACTIVE) && !(HIWORD(wParam));

			if (desktoppal) {
				if (appactive) {
					setpalette(0,0,0);
					SetBWSystemColours();
//					printOSD("Resetting palette.\n");
				} else {
					RestoreSystemColours();
//					printOSD("Resetting system colours.\n");
				}
			}

			if (bDInputInited) AcquireInputDevices(appactive);
			break;

		case WM_PALETTECHANGED:
			// someone stole the palette so try and steal it back
			if (appactive && wParam != hWindow) setpalette(0,0,0);
			break;

		case WM_DISPLAYCHANGE:
			// desktop settings changed so adjust our world-view accordingly
			desktopxdim = LOWORD(lParam);
			desktopydim = HIWORD(lParam);
			desktoppal  = (wParam <= 8);
			getvalidmodes();
			break;

		case WM_PAINT:
			repaintneeded=1;
			break;

		case WM_ERASEBKGND:
			return TRUE;

		case WM_MOVE:
			windowposx = LOWORD(lParam);
			windowposy = HIWORD(lParam);
			return 0;

		case WM_CLOSE:
			quitevent = 1;
			return 0;

		case WM_SYSCOMMAND:
			// don't let the monitor fall asleep or let the screensaver activate
			if (wParam == SC_SCREENSAVE || wParam == SC_MONITORPOWER) return 0;

			// Since DirectInput won't let me set an exclusive-foreground
			// keyboard for some unknown reason I just have to tell Windows to
			// rack off with its keyboard accelerators.
			if (wParam == SC_KEYMENU || wParam == SC_HOTKEY) return 0;
			break;

		case WM_KEYDOWN:
		case WM_KEYUP:
			// pause sucks. I read that apparently it doesn't work the same everwhere
			// with DirectInput but it does with Windows messages. Oh well.
			if (wParam == VK_PAUSE && (lParam & 0x80000000l)) {
				SetKey(0x59, 1);

				if (keypresscallback)
					keypresscallback(0x59, 1);
			}
			break;
			
		case WM_HOTKEY:
			return 0;

		case WM_ENTERMENULOOP:
		case WM_ENTERSIZEMOVE:
			AcquireInputDevices(0);
			return 0;
		case WM_EXITMENULOOP:
		case WM_EXITSIZEMOVE:
			AcquireInputDevices(1);
			return 0;

		case WM_DESTROY:
			hWindow = 0;
			PostQuitMessage(0);
			return 0;

		default:
			break;
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


//
// RegisterWindowClass() -- register the window class
//
static BOOL RegisterWindowClass(void)
{
	WNDCLASSEX wcx;

	if (window_class_registered) return FALSE;

	printOSD("Registering window class\n");

	wcx.cbSize	= sizeof(wcx);
	wcx.style	= 0;
	wcx.lpfnWndProc	= WndProcCallback;
	wcx.cbClsExtra	= 0;
	wcx.cbWndExtra	= 0;
	wcx.hInstance	= hInstance;
	wcx.hIcon	= LoadImage(hInstance, MAKEINTRESOURCE(100), IMAGE_ICON,
				GetSystemMetrics(SM_CXICON), GetSystemMetrics(SM_CYICON), LR_DEFAULTCOLOR);
	wcx.hCursor	= LoadCursor(hInstance, IDC_ARROW);
	wcx.hbrBackground = (HBRUSH)COLOR_GRAYTEXT;
	wcx.lpszMenuName = NULL;
	wcx.lpszClassName = "buildapp";
	wcx.hIconSm	= LoadImage(hInstance, MAKEINTRESOURCE(100), IMAGE_ICON,
				GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON), LR_DEFAULTCOLOR);
	if (!RegisterClassEx(&wcx)) {
		ShowErrorBox("Failed to register window class");
		return TRUE;
	}

	window_class_registered = TRUE;

	return FALSE;
}


//
// InitDirectDraw() -- get DirectDraw started
//

// device enumerator
static BOOL WINAPI InitDirectDraw_enum(GUID *lpGUID, LPSTR lpDesc, LPSTR lpName, LPVOID lpContext)
{
	printOSD("    * %s\n", lpDesc);
	return 1;
}

static BOOL InitDirectDraw(void)
{
	HRESULT result;
	HRESULT (WINAPI *aDirectDrawCreate)(GUID *, LPDIRECTDRAW *, IUnknown *);
	HRESULT (WINAPI *aDirectDrawEnumerate)(LPDDENUMCALLBACK, LPVOID);

	if (bDDrawInited) return FALSE;

	printOSD("Initializing DirectDraw...\n");

	// load up the DirectDraw DLL
	if (!hDDrawDLL) {
		printOSD("  - Loading DDRAW.DLL\n");
		hDDrawDLL = LoadLibrary("DDRAW.DLL");
		if (!hDDrawDLL) {
			ShowErrorBox("Error loading DDRAW.DLL");
			return TRUE;
		}
	}

	// get the pointer to DirectDrawEnumerate
	aDirectDrawEnumerate = (void *)GetProcAddress(hDDrawDLL, "DirectDrawEnumerateA");
	if (!aDirectDrawEnumerate) {
		ShowErrorBox("Error fetching DirectDrawEnumerate()");
		UninitDirectDraw();
		return TRUE;
	}

	// enumerate the devices to make us look fancy
	printOSD("  - Enumerating display devices\n");
	aDirectDrawEnumerate(InitDirectDraw_enum, NULL);

	// get the pointer to DirectDrawCreate
	aDirectDrawCreate = (void *)GetProcAddress(hDDrawDLL, "DirectDrawCreate");
	if (!aDirectDrawCreate) {
		ShowErrorBox("Error fetching DirectDrawCreate()");
		UninitDirectDraw();
		return TRUE;
	}

	// create a new DirectDraw object
	printOSD("  - Creating DirectDraw object\n");
	result = aDirectDrawCreate(NULL, &lpDD, NULL);
	if (result != DD_OK) {
		ShowDDrawErrorBox("DirectDrawCreate() failed", result);
		UninitDirectDraw();
		return TRUE;
	}

	bDDrawInited = TRUE;

	return FALSE;
}


//
// UninitDirectDraw() -- clean up DirectDraw
//
static void UninitDirectDraw(void)
{
	if (bDDrawInited) printOSD("Uninitializing DirectDraw...\n");

	AcquireInputDevices(0);
	ReleaseSurfaces();

	if (lpDD) {
		printOSD("  - Releasing DirectDraw object\n");
		IDirectDraw_Release(lpDD);
		lpDD = NULL;
	}

	if (hDDrawDLL) {
		printOSD("  - Unloading DDRAW.DLL\n");
		FreeLibrary(hDDrawDLL);
		hDDrawDLL = NULL;
	}

	bDDrawInited = FALSE;
}


//
// ReleaseSurfaces() -- release the front and back buffers
//
static void ReleaseSurfaces(void)
{
	if (lpDDPalette) {
		printOSD("  - Releasing palette\n");
		IDirectDrawPalette_Release(lpDDPalette);
		lpDDPalette = NULL;
	}

	if (lpDDSBack) {
		printOSD("  - Releasing back-buffer surface\n");
		IDirectDrawSurface_Release(lpDDSBack);
		lpDDSBack = NULL;
	}
	
	if (lpDDSPrimary) {
		printOSD("  - Releasing primary surface\n");
		IDirectDrawSurface_Release(lpDDSPrimary);
		lpDDSPrimary = NULL;
	}

	if (lpDDSOffscreen) {
		printOSD("  - Releasing offscreen surface\n");
		IDirectDrawSurface_Release(lpDDSOffscreen);
		lpDDSOffscreen = NULL;
	}
}


//
// InitDirectInput() -- get DirectInput started
//

// device enumerator
static BOOL CALLBACK InitDirectInput_enum(LPCDIDEVICEINSTANCE lpddi, LPVOID pvRef)
{
	const char *d;
	
	switch (lpddi->dwDevType&0xff) {
		case DIDEVTYPE_KEYBOARD: inputdevices |= 1; d = "KEYBOARD"; break;
		case DIDEVTYPE_MOUSE:    inputdevices |= 2; d = "MOUSE"; break;
		case DIDEVTYPE_JOYSTICK: inputdevices |= 4; d = "JOYSTICK"; break;
		default: d = "OTHER";
	}
	
	printOSD("    * %s: %s\n", d, lpddi->tszProductName);

	return DIENUM_CONTINUE;
}

static BOOL InitDirectInput(void)
{
	HRESULT result;
	HRESULT (WINAPI *aDirectInputCreateA)(HINSTANCE, DWORD, LPDIRECTINPUTA *, LPUNKNOWN);
	DIPROPDWORD dipdw;
	LPDIRECTINPUTDEVICEA dev;

	if (bDInputInited) return FALSE;

	printOSD("Initializing DirectInput...\n");

	// load up the DirectInput DLL
	if (!hDInputDLL) {
		printOSD("  - Loading DINPUT.DLL\n");
		hDInputDLL = LoadLibrary("DINPUT.DLL");
		if (!hDInputDLL) {
			ShowErrorBox("Error loading DINPUT.DLL");
			return TRUE;
		}
	}

	// get the pointer to DirectInputCreate
	aDirectInputCreateA = (void *)GetProcAddress(hDInputDLL, "DirectInputCreateA");
	if (!aDirectInputCreateA) {
		ShowErrorBox("Error fetching DirectInputCreateA()");
		UninitDirectInput();
		return TRUE;
	}

	// create a new DirectInput object
	printOSD("  - Creating DirectInput object\n");
	result = aDirectInputCreateA(hInstance, DIRECTINPUT_VERSION, &lpDI, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("DirectInputCreateA() failed", result);
		UninitDirectInput();
		return TRUE;
	}

	// enumerate devices to make us look fancy
	printOSD("  - Enumerating attached input devices\n");
	IDirectInput_EnumDevices(lpDI, 0, InitDirectInput_enum, NULL, DIEDFL_ATTACHEDONLY);

	// ***
	// create the keyboard device
	// ***
	printOSD("  - Creating keyboard device\n");
	result = IDirectInput_CreateDevice(lpDI, &GUID_SysKeyboard, &dev, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed creating keyboard device", result);
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice_QueryInterface(dev, &IID_IDirectInputDevice2, (LPVOID *)&lpDIDKeyboard);
	IDirectInputDevice_Release(dev);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed querying DirectInput2 interface for keyboard device", result);
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice2_SetDataFormat(lpDIDKeyboard, &c_dfDIKeyboard);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting keyboard data format", result);
		UninitDirectInput();
		return TRUE;
	}

	// set up keyboard for buffered input
	memset(&dipdw, 0, sizeof(dipdw));
	dipdw.diph.dwSize = sizeof(DIPROPDWORD);
	dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
	dipdw.diph.dwObj = 0;
	dipdw.diph.dwHow = DIPH_DEVICE;
	dipdw.dwData = INPUT_BUFFER_SIZE;

	result = IDirectInputDevice2_SetProperty(lpDIDKeyboard, DIPROP_BUFFERSIZE, &dipdw.diph);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting keyboard buffering", result);
		UninitDirectInput();
		return TRUE;
	}

	inputevt[0] = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (inputevt[0] == NULL) {
		ShowErrorBox("Couldn't create keyboard event object");
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice2_SetEventNotification(lpDIDKeyboard, inputevt[0]);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting keyboard event object", result);
		UninitDirectInput();
		return TRUE;
	}

	// ***
	// create the mouse device
	// ***
	printOSD("  - Creating mouse device\n");
	result = IDirectInput_CreateDevice(lpDI, &GUID_SysMouse, &dev, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed creating mouse device", result);
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice_QueryInterface(dev, &IID_IDirectInputDevice2A, (LPVOID *)&lpDIDMouse);
	IDirectInputDevice_Release(dev);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed querying DirectInput2 interface for mouse device", result);
		UninitDirectInput();
		return TRUE;
	}
	
	result = IDirectInputDevice2_SetDataFormat(lpDIDMouse, &c_dfDIMouse);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting mouse data format", result);
		UninitDirectInput();
		return TRUE;
	}

	// set up mouse for buffered IO
	memset(&dipdw, 0, sizeof(dipdw));
	dipdw.diph.dwSize = sizeof(DIPROPDWORD);
	dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
	dipdw.diph.dwObj = 0;
	dipdw.diph.dwHow = DIPH_DEVICE;
	dipdw.dwData = INPUT_BUFFER_SIZE;

	result = IDirectInputDevice2_SetProperty(lpDIDMouse, DIPROP_BUFFERSIZE, &dipdw.diph);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting mouse buffering", result);
		UninitDirectInput();
		return TRUE;
	}

	inputevt[1] = CreateEvent(NULL, FALSE, FALSE, NULL);
	if (inputevt[1] == NULL) {
		ShowErrorBox("Couldn't create mouse event object");
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice2_SetEventNotification(lpDIDMouse, inputevt[1]);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting mouse event object", result);
		UninitDirectInput();
		return TRUE;
	}

	bDInputInited = TRUE;
	return FALSE;
}


//
// UninitDirectInput() -- clean up DirectInput
//
static void UninitDirectInput(void)
{
	if (bDInputInited) printOSD("Uninitializing DirectInput...\n");

	AcquireInputDevices(0);

	if (lpDIDKeyboard) IDirectInputDevice2_SetEventNotification(lpDIDKeyboard, NULL);
	if (lpDIDMouse) IDirectInputDevice2_SetEventNotification(lpDIDMouse, NULL);
	if (inputevt[0]) {
		CloseHandle(inputevt[0]);
		inputevt[0] = NULL;
	}
	if (inputevt[1]) {
		CloseHandle(inputevt[1]);
		inputevt[1] = NULL;
	}

	if (lpDIDKeyboard) {
		printOSD("  - Releasing keyboard device\n");
		IDirectInputDevice2_Release(lpDIDKeyboard);
		lpDIDKeyboard = NULL;
	}

	if (lpDIDMouse) {
		printOSD("  - Releasing mouse device\n");
		IDirectInputDevice2_Release(lpDIDMouse);
		lpDIDMouse = NULL;
	}

	if (lpDI) {
		printOSD("  - Releasing DirectInput object\n");
		IDirectInput_Release(lpDI);
		lpDI = NULL;
	}

	if (hDInputDLL) {
		printOSD("  - Unloading DINPUT.DLL\n");
		FreeLibrary(hDInputDLL);
		hDInputDLL = NULL;
	}

	bDInputInited = FALSE;
}


//
// AcquireInputDevices() -- (un)acquires the input devices
//
static void AcquireInputDevices(char acquire)
{
	DWORD flags;
	HRESULT result;

	if (!bDInputInited) return;
	if (!hWindow) return;

	// reset mouse and keypresses
	mousex = 0;
	mousey = 0;
	mouseb = 0;

	memset(keystatus, 0, sizeof(keystatus));
	memset(keyfifo, 0, sizeof(keyfifo));
	keyfifoplc = 0;
	keyfifoend = 0;
	lastKeyDown = 0;

	if (acquire) {
		if (!appactive) return;		// why acquire when inactive?
		if (lpDIDKeyboard) {
			IDirectInputDevice2_Unacquire(lpDIDKeyboard);
			result = IDirectInputDevice2_SetCooperativeLevel(lpDIDKeyboard, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			IDirectInputDevice2_Acquire(lpDIDKeyboard);
			if (result != DD_OK) {
				printOSD("IDirectInputDevice2_SetCooperativeLevel(keyboard): %s\n", GetDInputError(result));
			}
		}

		if (lpDIDMouse) {
			if (fullscreen || mousegrab) flags = DISCL_FOREGROUND|DISCL_EXCLUSIVE;
			else flags = DISCL_FOREGROUND|DISCL_NONEXCLUSIVE;
			IDirectInputDevice2_Unacquire(lpDIDMouse);
			result = IDirectInputDevice2_SetCooperativeLevel(lpDIDMouse, hWindow, flags);
			IDirectInputDevice2_Acquire(lpDIDMouse);
			if (result != DD_OK) {
				printOSD("IDirectInputDevice2_SetCooperativeLevel(mouse): %s\n", GetDInputError(result));
			}
		}
	} else {
		if (lpDIDKeyboard) {
			IDirectInputDevice2_Unacquire(lpDIDKeyboard);
			result = IDirectInputDevice2_SetCooperativeLevel(lpDIDKeyboard, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			if (result != DD_OK) {
				printOSD("IDirectInputDevice2_SetCooperativeLevel(keyboard): %s\n", GetDInputError(result));
			}
		}

		if (lpDIDMouse) {
			IDirectInputDevice2_Unacquire(lpDIDMouse);
			result = IDirectInputDevice2_SetCooperativeLevel(lpDIDMouse, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			if (result != DD_OK) {
				printOSD("IDirectInputDevice2_SetCooperativeLevel(mouse): %s\n", GetDInputError(result));
			}
		}
	}
}


//
// ProcessInputDevices() -- processes the input devices
//
static void ProcessInputDevices(void)
{
	DIMOUSESTATE dims;
	DWORD i;
	HRESULT result;
	
	DIDEVICEOBJECTDATA didod[INPUT_BUFFER_SIZE];
	DWORD dwElements = INPUT_BUFFER_SIZE;
	DWORD ev;
	unsigned long t,u;

	if (lpDIDKeyboard) {
		result = IDirectInputDevice2_Poll(lpDIDKeyboard);
		if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED) {
			IDirectInputDevice2_Acquire(lpDIDKeyboard);
			IDirectInputDevice2_Poll(lpDIDKeyboard);
		}
	}
	if (lpDIDMouse) {
		result = IDirectInputDevice2_Poll(lpDIDMouse);
		if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED) {
			IDirectInputDevice2_Acquire(lpDIDMouse);
			IDirectInputDevice2_Poll(lpDIDMouse);
		}
	}

	// do this here because we only want the wheel to signal once, but hold the state for a moment
	if (mouseba[0]>0) {
		if (mouseba[0]<16) mouseba[0]++;
		else {
			if (mousepresscallback) mousepresscallback(5,0);
			mouseba[0]=0; mouseb &= ~16;
		}
	}
	if (mouseba[1]>0) {
		if (mouseba[1]<16) mouseba[1]++;
		else {
			if (mousepresscallback) mousepresscallback(6,0);
			mouseba[1]=0; mouseb &= ~32;
		}
	}

	// use event objects so that we can quickly get indication of when data is ready
	// to be read and input events processed
	ev = MsgWaitForMultipleObjects(NUM_INPUTS, inputevt, FALSE, 0, 0);
	if ((ev >= WAIT_OBJECT_0) && (ev < (WAIT_OBJECT_0+NUM_INPUTS))) {
		switch (ev - WAIT_OBJECT_0) {
			case 0:		// keyboard
				if (!lpDIDKeyboard) break;
				result = IDirectInputDevice2_GetDeviceData(lpDIDKeyboard, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
				if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED) {
					IDirectInputDevice2_Acquire(lpDIDKeyboard);
					result = IDirectInputDevice2_GetDeviceData(lpDIDKeyboard, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
				}

				if (result == DI_OK) {
					// process the key events
					t = getticks()/(0xffffffffu/1000);
					for (i=0; i<dwElements; i++) {
						if (didod[i].dwOfs == 0xc5) continue;	// fucking pause
						
						// hook in the osd
						if (OSD_HandleKey(didod[i].dwOfs, (didod[i].dwData & 0x80)) != 0) {
							SetKey(didod[i].dwOfs, (didod[i].dwData & 0x80) == 0x80);

							if (keypresscallback)
								keypresscallback(didod[i].dwOfs, (didod[i].dwData & 0x80) == 0x80);
						}

						if (((lastKeyDown & 0x7fffffffl) == didod[i].dwOfs) && !(didod[i].dwData & 0x80))
							lastKeyDown = 0;
						else if (didod[i].dwData & 0x80) {
							lastKeyDown = didod[i].dwOfs;
							lastKeyTime = t;
						}
					}
				}
				break;
				
			case 1:		// mouse
				if (!lpDIDMouse) break;
				result = IDirectInputDevice2_GetDeviceData(lpDIDMouse, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
				if (result == DIERR_INPUTLOST || result == DIERR_NOTACQUIRED) {
					IDirectInputDevice2_Acquire(lpDIDMouse);
					result = IDirectInputDevice2_GetDeviceData(lpDIDMouse, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
				}

				if (result == DI_OK) {
					// process the mouse events
					mousex=0;
					mousey=0;
					for (i=0; i<dwElements; i++) {
						switch (didod[i].dwOfs) {
							case DIMOFS_BUTTON0:
								if (didod[i].dwData & 0x80) mouseb |= 1;
								else mouseb &= ~1;
								if (mousepresscallback)
									mousepresscallback(1, (mouseb&1)==1);
								break;
							case DIMOFS_BUTTON1:
								if (didod[i].dwData & 0x80) mouseb |= 2;
								else mouseb &= ~2;
								if (mousepresscallback)
									mousepresscallback(2, (mouseb&2)==2);
								break;
							case DIMOFS_BUTTON2:
								if (didod[i].dwData & 0x80) mouseb |= 4;
								else mouseb &= ~4;
								if (mousepresscallback)
									mousepresscallback(3, (mouseb&4)==4);
								break;
							case DIMOFS_BUTTON3:
								if (didod[i].dwData & 0x80) mouseb |= 8;
								else mouseb &= ~8;
								if (mousepresscallback)
									mousepresscallback(4, (mouseb&8)==8);
								break;
							case DIMOFS_X:
								mousex = (short)didod[i].dwData; break;
							case DIMOFS_Y:
								mousey = (short)didod[i].dwData; break;
							case DIMOFS_Z:
								if ((int)didod[i].dwData > 0) {		// wheel up
									mouseb |= 32;
									mouseba[1] = 1;
									if (mousepresscallback)
										mousepresscallback(6, 1);
								}
								else if ((int)didod[i].dwData < 0) {	// wheel down
									mouseb |= 16;
									mouseba[0] = 1;
									if (mousepresscallback)
										mousepresscallback(5, 1);
								}
								break;
						}
					}
				}
				break;
		}
	}

	// key repeat
	// this is like this because the period of t is 1000ms
	if (lastKeyDown > 0) {
		t = getticks()/(0xffffffffu/1000);
		u = (1000 + t - lastKeyTime)%1000;
		if ((u >= 250) && !(lastKeyDown&0x80000000l)) {
			if (OSD_HandleKey(lastKeyDown, 1) != 0)
				SetKey(lastKeyDown, 1);
			lastKeyDown |= 0x80000000l;
			lastKeyTime = t;
		} else if ((u >= 30) && (lastKeyDown&0x80000000l)) {
			if (OSD_HandleKey(lastKeyDown&(0x7fffffffl), 1) != 0)
				SetKey(lastKeyDown&(0x7fffffffl), 1);
			lastKeyTime = t;
		}
	}
}


//
// UninitDIB() -- clean up the DIB renderer
//
static void UninitDIB(void)
{
	AcquireInputDevices(0);

	if (desktoppal)
		RestoreSystemColours();

	if (hPalette) {
		DeleteObject(hPalette);
		hPalette = NULL;
	}

	if (hDCSection) {
		DeleteDC(hDCSection);
		hDCSection = NULL;
	}

	if (hDIBSection) {
		DeleteObject(hDIBSection);
		hDIBSection = NULL;
	}

	if (hDC) {
		ReleaseDC(hWindow, hDC);
		hDC = NULL;
	}
}


//
// CreateAppWindow() -- create the application window
//
static BOOL CreateAppWindow(int width, int height, int fs)
{
	TCHAR msg[1024];
	RECT rect;
	int w, h, x, y, i;
	struct binfo {
		BITMAPINFOHEADER header;
		RGBQUAD colours[256];
	} dibsect;

	HRESULT result;
	DDSURFACEDESC ddsd;


	if (!hWindow) {
		hWindow = CreateWindowEx(
			0,
			"buildapp",
			apptitle,
			WINDOW_STYLE,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			320,
			200,
			NULL,
			NULL,
			hInstance,
			0);
		if (!hWindow) {
			ShowErrorBox("Unable to create window");
			return TRUE;
		}
	}

	ReleaseSurfaces();

	if (fullscreen) {
		// restore previous display mode and set to normal cooperative level
		result = IDirectDraw_RestoreDisplayMode(lpDD);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error restoring display mode", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDraw_SetCooperativeLevel(lpDD, hWindow, DDSCL_NORMAL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error setting cooperative level", result);
			UninitDirectDraw();
			return TRUE;
		}
	}

	// resize the window
	if (!fs) {
		rect.left = 0;
		rect.top = 0;
		rect.right = width-1;
		rect.bottom = height-1;
		AdjustWindowRect(&rect, WINDOW_STYLE, FALSE);

		w = (rect.right - rect.left);
		h = (rect.bottom - rect.top);
		x = (desktopxdim - w) / 2;
		y = (desktopydim - h) / 2;
	} else {
		x=y=0;
		w=width;
		h=height;
	}
	SetWindowPos(hWindow, HWND_TOP, x, y, w, h, 0);

	SetFocus(hWindow);
	ShowWindow(hWindow, SW_SHOWNORMAL);
	UpdateWindow(hWindow);
	SetForegroundWindow(hWindow);

	// fullscreen?
	if (!fs) {
		// no, use DIB section

		if (!hDC) {
			hDC = GetDC(hWindow);
			if (!hDC) {
				ShowErrorBox("Error getting device context");
				return TRUE;
			}
		}

		if (hDCSection) {
			DeleteDC(hDCSection);
			hDCSection = NULL;
		}

		// destroy the previous DIB section if it existed
		if (hDIBSection) {
			DeleteObject(hDIBSection);
			hDIBSection = NULL;
		}

		// create the new DIB section
		memset(&dibsect, 0, sizeof(dibsect));
		dibsect.header.biSize = sizeof(dibsect.header);
		dibsect.header.biWidth = width|1;	// Ken did this
		dibsect.header.biHeight = -height;
		dibsect.header.biPlanes = 1;
		dibsect.header.biBitCount = 8;
		dibsect.header.biCompression = BI_RGB;
		dibsect.header.biClrUsed = 256;
		dibsect.header.biClrImportant = 256;
		for (i=0; i<256; i++) {
			dibsect.colours[i].rgbBlue = curPalette[i].peBlue;
			dibsect.colours[i].rgbGreen = curPalette[i].peGreen;
			dibsect.colours[i].rgbRed = curPalette[i].peRed;
		}

		hDIBSection = CreateDIBSection(hDC, (BITMAPINFO *)&dibsect, DIB_RGB_COLORS, &lpPixels, NULL, 0);
		if (!hDIBSection) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;

			ShowErrorBox("Error creating DIB section");
			return TRUE;
		}

		memset(lpPixels, 0, width*height);

		// create a compatible memory DC
		hDCSection = CreateCompatibleDC(hDC);
		if (!hDCSection) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;

			ShowErrorBox("Error creating compatible DC");
			return TRUE;
		}

		// select the DIB section into the memory DC
		if (!SelectObject(hDCSection, hDIBSection)) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;
			DeleteDC(hDCSection);
			hDCSection = NULL;

			ShowErrorBox("Error creating compatible DC");
			return TRUE;
		}

		xres = width;
		yres = height;

		//bytesperline = width;
		frameplace = 0;

		//imageSize = bytesperline*yres;

		fullscreen = fs;
	} else {
		// yes, set up DirectDraw

		// clean up after the DIB renderer if it was being used
		UninitDIB();

		if (!bDDrawInited) {
			DestroyWindow(hWindow);
			hWindow = NULL;
			return TRUE;
		}

		ReleaseSurfaces();	// discard previous mode's surfaces

		// set exclusive cooperative level
		result = IDirectDraw_SetCooperativeLevel(lpDD, hWindow, DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error setting cooperative level", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDraw_RestoreDisplayMode(lpDD);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error restoring display mode", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDraw_SetDisplayMode(lpDD, width, height, 8);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error setting display mode", result);
			UninitDirectDraw();
			return TRUE;
		}

		// now create the DirectDraw surfaces
		ZeroMemory(&ddsd, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
		ddsd.dwBackBufferCount = 2;	// triple-buffer

		printOSD("  - Creating primary surface\n");
		result = IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSPrimary, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating primary surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		ZeroMemory(&ddsd.ddsCaps, sizeof(ddsd.ddsCaps));
		ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;

		printOSD("  - Getting back buffer\n");
		result = IDirectDrawSurface_GetAttachedSurface(lpDDSPrimary, &ddsd.ddsCaps, &lpDDSBack);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure fetching back-buffer surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		ZeroMemory(&ddsd, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
		ddsd.dwWidth = width|1;		// Ken did this
		ddsd.dwHeight = height;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

		printOSD("  - Creating offscreen surface\n");
		result = IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSOffscreen, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating offscreen surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		// attach a palette to the primary surface
		printOSD("  - Creating palette\n");
		result = IDirectDraw_CreatePalette(lpDD, DDPCAPS_8BIT | DDPCAPS_ALLOW256, curPalette, &lpDDPalette, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating palette", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDrawSurface_SetPalette(lpDDSPrimary, lpDDPalette);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure setting palette", result);
			UninitDirectDraw();
			return TRUE;
		}

		xres = width;
		yres = height;

		// bytesperline is set when framebuffer is locked
		frameplace = 0;
		lockcount = 0;

		// imageSize is calculated at lock time

		fullscreen = fs;
	}
	modechange = 1;
	OSD_ResizeDisplay(xres,yres);

	return FALSE;
}


//
// DestroyAppWindow() -- destroys the application window
//
static void DestroyAppWindow(void)
{
	UninitDirectDraw();
	UninitDIB();

	if (hWindow) {
		DestroyWindow(hWindow);
		hWindow = NULL;
	}
}


//
// GetDDrawError() -- stinking huge list of error messages since MS didn't want to include
//   them in the DLL
//
static const char * GetDDrawError(HRESULT code)
{
	switch (code) {
		case DD_OK: return "DD_OK";
		case DDERR_ALREADYINITIALIZED: return "DDERR_ALREADYINITIALIZED";
		case DDERR_BLTFASTCANTCLIP: return "DDERR_BLTFASTCANTCLIP";
		case DDERR_CANNOTATTACHSURFACE: return "DDERR_CANNOTATTACHSURFACE";
		case DDERR_CANNOTDETACHSURFACE: return "DDERR_CANNOTDETACHSURFACE";
		case DDERR_CANTCREATEDC: return "DDERR_CANTCREATEDC";
		case DDERR_CANTDUPLICATE: return "DDERR_CANTDUPLICATE";
		case DDERR_CANTLOCKSURFACE: return "DDERR_CANTLOCKSURFACE";
		case DDERR_CANTPAGELOCK: return "DDERR_CANTPAGELOCK";
		case DDERR_CANTPAGEUNLOCK: return "DDERR_CANTPAGEUNLOCK";
		case DDERR_CLIPPERISUSINGHWND: return "DDERR_CLIPPERISUSINGHWND";
		case DDERR_COLORKEYNOTSET: return "DDERR_COLORKEYNOTSET";
		case DDERR_CURRENTLYNOTAVAIL: return "DDERR_CURRENTLYNOTAVAIL";
		case DDERR_DCALREADYCREATED: return "DDERR_DCALREADYCREATED";
		case DDERR_DEVICEDOESNTOWNSURFACE: return "DDERR_DEVICEDOESNTOWNSURFACE";
		case DDERR_DIRECTDRAWALREADYCREATED: return "DDERR_DIRECTDRAWALREADYCREATED";
		case DDERR_EXCEPTION: return "DDERR_EXCEPTION";
		case DDERR_EXCLUSIVEMODEALREADYSET: return "DDERR_EXCLUSIVEMODEALREADYSET";
		case DDERR_EXPIRED: return "DDERR_EXPIRED";
		case DDERR_GENERIC: return "DDERR_GENERIC";
		case DDERR_HEIGHTALIGN: return "DDERR_HEIGHTALIGN";
		case DDERR_HWNDALREADYSET: return "DDERR_HWNDALREADYSET";
		case DDERR_HWNDSUBCLASSED: return "DDERR_HWNDSUBCLASSED";
		case DDERR_IMPLICITLYCREATED: return "DDERR_IMPLICITLYCREATED";
		case DDERR_INCOMPATIBLEPRIMARY: return "DDERR_INCOMPATIBLEPRIMARY";
		case DDERR_INVALIDCAPS: return "DDERR_INVALIDCAPS";
		case DDERR_INVALIDCLIPLIST: return "DDERR_INVALIDCLIPLIST";
		case DDERR_INVALIDDIRECTDRAWGUID: return "DDERR_INVALIDDIRECTDRAWGUID";
		case DDERR_INVALIDMODE: return "DDERR_INVALIDMODE";
		case DDERR_INVALIDOBJECT: return "DDERR_INVALIDOBJECT";
		case DDERR_INVALIDPARAMS: return "DDERR_INVALIDPARAMS";
		case DDERR_INVALIDPIXELFORMAT: return "DDERR_INVALIDPIXELFORMAT";
		case DDERR_INVALIDPOSITION: return "DDERR_INVALIDPOSITION";
		case DDERR_INVALIDRECT: return "DDERR_INVALIDRECT";
		case DDERR_INVALIDSTREAM: return "DDERR_INVALIDSTREAM";
		case DDERR_INVALIDSURFACETYPE: return "DDERR_INVALIDSURFACETYPE";
		case DDERR_LOCKEDSURFACES: return "DDERR_LOCKEDSURFACES";
		case DDERR_MOREDATA: return "DDERR_MOREDATA";
		case DDERR_NO3D: return "DDERR_NO3D";
		case DDERR_NOALPHAHW: return "DDERR_NOALPHAHW";
		case DDERR_NOBLTHW: return "DDERR_NOBLTHW";
		case DDERR_NOCLIPLIST: return "DDERR_NOCLIPLIST";
		case DDERR_NOCLIPPERATTACHED: return "DDERR_NOCLIPPERATTACHED";
		case DDERR_NOCOLORCONVHW: return "DDERR_NOCOLORCONVHW";
		case DDERR_NOCOLORKEY: return "DDERR_NOCOLORKEY";
		case DDERR_NOCOLORKEYHW: return "DDERR_NOCOLORKEYHW";
		case DDERR_NOCOOPERATIVELEVELSET: return "DDERR_NOCOOPERATIVELEVELSET";
		case DDERR_NODC: return "DDERR_NODC";
		case DDERR_NODDROPSHW: return "DDERR_NODDROPSHW";
		case DDERR_NODIRECTDRAWHW: return "DDERR_NODIRECTDRAWHW";
		case DDERR_NODIRECTDRAWSUPPORT: return "DDERR_NODIRECTDRAWSUPPORT";
		case DDERR_NOEMULATION: return "DDERR_NOEMULATION";
		case DDERR_NOEXCLUSIVEMODE: return "DDERR_NOEXCLUSIVEMODE";
		case DDERR_NOFLIPHW: return "DDERR_NOFLIPHW";
		case DDERR_NOFOCUSWINDOW: return "DDERR_NOFOCUSWINDOW";
		case DDERR_NOGDI: return "DDERR_NOGDI";
		case DDERR_NOHWND: return "DDERR_NOHWND";
		case DDERR_NOMIPMAPHW: return "DDERR_NOMIPMAPHW";
		case DDERR_NOMIRRORHW: return "DDERR_NOMIRRORHW";
		case DDERR_NONONLOCALVIDMEM: return "DDERR_NONONLOCALVIDMEM";
		case DDERR_NOOPTIMIZEHW: return "DDERR_NOOPTIMIZEHW";
		case DDERR_NOOVERLAYDEST: return "DDERR_NOOVERLAYDEST";
		case DDERR_NOOVERLAYHW: return "DDERR_NOOVERLAYHW";
		case DDERR_NOPALETTEATTACHED: return "DDERR_NOPALETTEATTACHED";
		case DDERR_NOPALETTEHW: return "DDERR_NOPALETTEHW";
		case DDERR_NORASTEROPHW: return "DDERR_NORASTEROPHW";
		case DDERR_NOROTATIONHW: return "DDERR_NOROTATIONHW";
		case DDERR_NOSTRETCHHW: return "DDERR_NOSTRETCHHW";
		case DDERR_NOT4BITCOLOR: return "DDERR_NOT4BITCOLOR";
		case DDERR_NOT4BITCOLORINDEX: return "DDERR_NOT4BITCOLORINDEX";
		case DDERR_NOT8BITCOLOR: return "DDERR_NOT8BITCOLOR";
		case DDERR_NOTAOVERLAYSURFACE: return "DDERR_NOTAOVERLAYSURFACE";
		case DDERR_NOTEXTUREHW: return "DDERR_NOTEXTUREHW";
		case DDERR_NOTFLIPPABLE: return "DDERR_NOTFLIPPABLE";
		case DDERR_NOTFOUND: return "DDERR_NOTFOUND";
		case DDERR_NOTINITIALIZED: return "DDERR_NOTINITIALIZED";
		case DDERR_NOTLOADED: return "DDERR_NOTLOADED";
		case DDERR_NOTLOCKED: return "DDERR_NOTLOCKED";
		case DDERR_NOTPAGELOCKED: return "DDERR_NOTPAGELOCKED";
		case DDERR_NOTPALETTIZED: return "DDERR_NOTPALETTIZED";
		case DDERR_NOVSYNCHW: return "DDERR_NOVSYNCHW";
		case DDERR_NOZBUFFERHW: return "DDERR_NOZBUFFERHW";
		case DDERR_NOZOVERLAYHW: return "DDERR_NOZOVERLAYHW";
		case DDERR_OUTOFCAPS: return "DDERR_OUTOFCAPS";
		case DDERR_OUTOFMEMORY: return "DDERR_OUTOFMEMORY";
		case DDERR_OUTOFVIDEOMEMORY: return "DDERR_OUTOFVIDEOMEMORY";
		case DDERR_OVERLAPPINGRECTS: return "DDERR_OVERLAPPINGRECTS";
		case DDERR_OVERLAYCANTCLIP: return "DDERR_OVERLAYCANTCLIP";
		case DDERR_OVERLAYCOLORKEYONLYONEACTIVE: return "DDERR_OVERLAYCOLORKEYONLYONEACTIVE";
		case DDERR_OVERLAYNOTVISIBLE: return "DDERR_OVERLAYNOTVISIBLE";
		case DDERR_PALETTEBUSY: return "DDERR_PALETTEBUSY";
		case DDERR_PRIMARYSURFACEALREADYEXISTS: return "DDERR_PRIMARYSURFACEALREADYEXISTS";
		case DDERR_REGIONTOOSMALL: return "DDERR_REGIONTOOSMALL";
		case DDERR_SURFACEALREADYATTACHED: return "DDERR_SURFACEALREADYATTACHED";
		case DDERR_SURFACEALREADYDEPENDENT: return "DDERR_SURFACEALREADYDEPENDENT";
		case DDERR_SURFACEBUSY: return "DDERR_SURFACEBUSY";
		case DDERR_SURFACEISOBSCURED: return "DDERR_SURFACEISOBSCURED";
		case DDERR_SURFACELOST: return "DDERR_SURFACELOST";
		case DDERR_SURFACENOTATTACHED: return "DDERR_SURFACENOTATTACHED";
		case DDERR_TOOBIGHEIGHT: return "DDERR_TOOBIGHEIGHT";
		case DDERR_TOOBIGSIZE: return "DDERR_TOOBIGSIZE";
		case DDERR_TOOBIGWIDTH: return "DDERR_TOOBIGWIDTH";
		case DDERR_UNSUPPORTED: return "DDERR_UNSUPPORTED";
		case DDERR_UNSUPPORTEDFORMAT: return "DDERR_UNSUPPORTEDFORMAT";
		case DDERR_UNSUPPORTEDMASK: return "DDERR_UNSUPPORTEDMASK";
		case DDERR_UNSUPPORTEDMODE: return "DDERR_UNSUPPORTEDMODE";
		case DDERR_VERTICALBLANKINPROGRESS: return "DDERR_VERTICALBLANKINPROGRESS";
		case DDERR_VIDEONOTACTIVE: return "DDERR_VIDEONOTACTIVE";
		case DDERR_WASSTILLDRAWING: return "DDERR_WASSTILLDRAWING";
		case DDERR_WRONGMODE: return "DDERR_WRONGMODE";
		case DDERR_XALIGN: return "DDERR_XALIGN";
		default: break;
	}
	return "Unknown error";
}


//
// GetDInputError() -- stinking huge list of error messages since MS didn't want to include
//   them in the DLL
//
static const char * GetDInputError(HRESULT code)
{
	switch (code) {
		case DI_OK: return "DI_OK";
		case DI_BUFFEROVERFLOW: return "DI_BUFFEROVERFLOW";
		case DI_DOWNLOADSKIPPED: return "DI_DOWNLOADSKIPPED";
		case DI_EFFECTRESTARTED: return "DI_EFFECTRESTARTED";
		case DI_POLLEDDEVICE: return "DI_POLLEDDEVICE";
		case DI_TRUNCATED: return "DI_TRUNCATED";
		case DI_TRUNCATEDANDRESTARTED: return "DI_TRUNCATEDANDRESTARTED";
		case DIERR_ACQUIRED: return "DIERR_ACQUIRED";
		case DIERR_ALREADYINITIALIZED: return "DIERR_ALREADYINITIALIZED";
		case DIERR_BADDRIVERVER: return "DIERR_BADDRIVERVER";
		case DIERR_BETADIRECTINPUTVERSION: return "DIERR_BETADIRECTINPUTVERSION";
		case DIERR_DEVICEFULL: return "DIERR_DEVICEFULL";
		case DIERR_DEVICENOTREG: return "DIERR_DEVICENOTREG";
		case DIERR_EFFECTPLAYING: return "DIERR_EFFECTPLAYING";
		case DIERR_HASEFFECTS: return "DIERR_HASEFFECTS";
		case DIERR_GENERIC: return "DIERR_GENERIC";
		case DIERR_HANDLEEXISTS: return "DIERR_HANDLEEXISTS";
		case DIERR_INCOMPLETEEFFECT: return "DIERR_INCOMPLETEEFFECT";
		case DIERR_INPUTLOST: return "DIERR_INPUTLOST";
		case DIERR_INVALIDPARAM: return "DIERR_INVALIDPARAM";
		case DIERR_MOREDATA: return "DIERR_MOREDATA";
		case DIERR_NOAGGREGATION: return "DIERR_NOAGGREGATION";
		case DIERR_NOINTERFACE: return "DIERR_NOINTERFACE";
		case DIERR_NOTACQUIRED: return "DIERR_NOTACQUIRED";
		case DIERR_NOTBUFFERED: return "DIERR_NOTBUFFERED";
		case DIERR_NOTDOWNLOADED: return "DIERR_NOTDOWNLOADED";
		case DIERR_NOTEXCLUSIVEACQUIRED: return "DIERR_NOTEXCLUSIVEACQUIRED";
		case DIERR_NOTFOUND: return "DIERR_NOTFOUND";
		case DIERR_NOTINITIALIZED: return "DIERR_NOTINITIALIZED";
		case DIERR_OLDDIRECTINPUTVERSION: return "DIERR_OLDDIRECTINPUTVERSION";
		case DIERR_OUTOFMEMORY: return "DIERR_OUTOFMEMORY";
		case DIERR_UNSUPPORTED: return "DIERR_UNSUPPORTED";
		case E_PENDING: return "E_PENDING";
		default: break;
	}
	return "Unknown error";
}


//
// GetWindowsErrorMsg() -- gives a pointer to a static buffer containing the Windows error message
//
static const LPTSTR GetWindowsErrorMsg(DWORD code)
{
	static TCHAR lpMsgBuf[1024];

	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL, code,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		(LPTSTR)lpMsgBuf, 1024, NULL);

	return lpMsgBuf;
}


//
// SaveSystemColours() -- save the Windows-reserved colours
//
static void SaveSystemColours(void)
{
	int i;

	if (system_colours_saved) return;

	for (i=0; i<NUM_SYS_COLOURS; i++)
		syscolours[i] = GetSysColor(syscolouridx[i]);

	system_colours_saved = 1;
}


//
// SetBWSystemColours() -- set system colours to a black-and-white scheme
//
static void SetBWSystemColours(void)
{
#define WHI RGB(255,255,255)
#define BLA RGB(0,0,0)
static COLORREF syscoloursbw[NUM_SYS_COLOURS] = {
	WHI, BLA, BLA, WHI, WHI, WHI, WHI, BLA, BLA, WHI,
	WHI, WHI, WHI, BLA, WHI, WHI, BLA, BLA, BLA, BLA,
	BLA, BLA, WHI, BLA, WHI
};
#undef WHI
#undef BLA
	if (!system_colours_saved || bw_colours_set) return;

	SetSysColors(NUM_SYS_COLOURS, syscolouridx, syscoloursbw);
	bw_colours_set = 1;
}


//
// RestoreSystemColours() -- restore the Windows-reserved colours
//
static void RestoreSystemColours(void)
{
	if (!system_colours_saved || !bw_colours_set) return;

	SetSystemPaletteUse(hDC, SYSPAL_STATIC);
	SetSysColors(NUM_SYS_COLOURS, syscolouridx, syscolours);
	bw_colours_set = 0;
}

