// SDL interface layer
// for the Build Engine
// by Jonathon Fowler (jonof@edgenetwk.com)
//
// Use SDL1.2 from http://www.libsdl.org

#include <stdlib.h>
#include "SDL.h"
#include "sdlayer.h"
#include "cache1d.h"
#include "pragmas.h"
#include "a.h"
#include "build.h"
#include "osd.h"
#include "compat.h"

extern char moustat;

#define SURFACE_FLAGS	(SDL_SWSURFACE|SDL_HWPALETTE|SDL_HWACCEL)

// undefine to restrict windowed resolutions to conventional sizes
#define ANY_WINDOWED_SIZE

// timer
//static SDL_TimerID timerid=0;
static Uint32 timerfreq=0;
static Uint32 timerlastsample=0;
static Uint32 timerticspersec=0;
static void (*usertimercallback)(void) = NULL;

// video
static SDL_Surface *sdl_surface;
long xres=-1, yres=-1, fullscreen=0, bytesperline, imageSize;
long frameplace=0, lockcount=0;
char modechange=1;
char offscreenrendering=0;

// input and events
static SDL_Event sdl_event;
char inputdevices=0;
char quitevent=0, appactive=1;
char cursorvisible=1;
short mousex=0,mousey=0,mouseb=0;

char keystatus[256], keyfifo[KEYFIFOSIZ], keyfifoplc, keyfifoend;

static char keytranslation[SDLK_LAST];

void (*keypresscallback)(long,long) = 0;
void (*mousepresscallback)(long,long) = 0;

//static SDL_Surface * loadtarga(const char *fn);		// for loading the icon


//
// initsystem() -- init SDL systems
//
int initsystem(void)
{
	const SDL_VideoInfo *vid;
	const SDL_version *linked = SDL_Linked_Version();
	SDL_version compiled;
	SDL_Surface *icon;
	char drvname[32];

	SDL_VERSION(&compiled);

	printOSD("Initializing SDL system interface "
		  "(compiled with SDL version %d.%d.%d, DLL version %d.%d.%d)\n",
		linked->major, linked->minor, linked->patch,
		compiled.major, compiled.minor, compiled.patch);

	if (SDL_Init(SDL_INIT_VIDEO | SDL_INIT_TIMER | SDL_INIT_NOPARACHUTE)) {
		printOSD("Initialization failed! (%s)\n", SDL_GetError());
		return -1;
	}

	atexit(uninitsystem);

	frameplace = 0;
	lockcount = 0;

/*
	icon = loadtarga("icon.tga");
	if (icon) {
		SDL_WM_SetIcon(icon, 0);
		SDL_FreeSurface(icon);
	}
*/
	if (SDL_VideoDriverName(drvname, 32))
		printOSD("Using \"%s\" video driver\n", drvname);

	// dump a quick summary of the graphics hardware
	vid = SDL_GetVideoInfo();
	printOSD("Video device information:\n");
	printOSD("  Can create hardware surfaces?          %s\n", (vid->hw_available)?"Yes":"No");
	printOSD("  Window manager available?              %s\n", (vid->wm_available)?"Yes":"No");
	printOSD("  Accelerated hardware blits?            %s\n", (vid->blit_hw)?"Yes":"No");
	printOSD("  Accelerated hardware colourkey blits?  %s\n", (vid->blit_hw_CC)?"Yes":"No");
	printOSD("  Accelerated hardware alpha blits?      %s\n", (vid->blit_hw_A)?"Yes":"No");
	printOSD("  Accelerated software blits?            %s\n", (vid->blit_sw)?"Yes":"No");
	printOSD("  Accelerated software colourkey blits?  %s\n", (vid->blit_sw_CC)?"Yes":"No");
	printOSD("  Accelerated software alpha blits?      %s\n", (vid->blit_sw_A)?"Yes":"No");
	printOSD("  Accelerated colour fills?              %s\n", (vid->blit_fill)?"Yes":"No");
	printOSD("  Total video memory:                    %dKB\n", vid->video_mem);

	return 0;
}


//
// uninitsystem() -- uninit SDL systems
//
void uninitsystem(void)
{
	uninitinput();
	uninitmouse();
	uninittimer();

	SDL_Quit();
}


//
// buildkeytranslationtable() -- map SDL codes to AT keyboard scancodes
//
static int buildkeytranslationtable(void)
{
	clearbufbyte(keytranslation,sizeof(keytranslation),0);

#define MAP(x,y) keytranslation[x] = y
	MAP(SDLK_BACKSPACE,	0xe);
	MAP(SDLK_TAB,		0xf);
	MAP(SDLK_RETURN,	0x1c);
	MAP(SDLK_PAUSE,		0x59);	// 0x1d + 0x45 + 0x9d + 0xc5
	MAP(SDLK_ESCAPE,	0x1);
	MAP(SDLK_SPACE,		0x39);
	MAP(SDLK_EXCLAIM,	0x2);	// '1'
	MAP(SDLK_QUOTEDBL,	0x28);	// '''
	MAP(SDLK_HASH,		0x4);	// '3'
	MAP(SDLK_DOLLAR,	0x5);	// '4'
	MAP(37,			0x6);	// '5' <-- where's the keysym SDL guys?
	MAP(SDLK_AMPERSAND,	0x8);	// '7'
	MAP(SDLK_QUOTE,		0x28);	// '''
	MAP(SDLK_LEFTPAREN,	0xa);	// '9'
	MAP(SDLK_RIGHTPAREN,	0xb);	// '0'
	MAP(SDLK_ASTERISK,	0x9);	// '8'
	MAP(SDLK_PLUS,		0xd);	// '='
	MAP(SDLK_COMMA,		0x33);
	MAP(SDLK_MINUS,		0xc);
	MAP(SDLK_PERIOD,	0x34);
	MAP(SDLK_SLASH,		0x35);
	MAP(SDLK_0,		0xb);
	MAP(SDLK_1,		0x2);
	MAP(SDLK_2,		0x3);
	MAP(SDLK_3,		0x4);
	MAP(SDLK_4,		0x5);
	MAP(SDLK_5,		0x6);
	MAP(SDLK_6,		0x7);
	MAP(SDLK_7,		0x8);
	MAP(SDLK_8,		0x9);
	MAP(SDLK_9,		0xa);
	MAP(SDLK_COLON,		0x27);
	MAP(SDLK_SEMICOLON,	0x27);
	MAP(SDLK_LESS,		0x33);
	MAP(SDLK_EQUALS,	0xd);
	MAP(SDLK_GREATER,	0x34);
	MAP(SDLK_QUESTION,	0x35);
	MAP(SDLK_AT,		0x3);	// '2'
	MAP(SDLK_LEFTBRACKET,	0x1a);
	MAP(SDLK_BACKSLASH,	0x2b);
	MAP(SDLK_RIGHTBRACKET,	0x1b);
	MAP(SDLK_CARET,		0x7);	// '7'
	MAP(SDLK_UNDERSCORE,	0xc);
	MAP(SDLK_BACKQUOTE,	0x29);
	MAP(SDLK_a,		0x1e);
	MAP(SDLK_b,		0x30);
	MAP(SDLK_c,		0x2e);
	MAP(SDLK_d,		0x20);
	MAP(SDLK_e,		0x12);
	MAP(SDLK_f,		0x21);
	MAP(SDLK_g,		0x22);
	MAP(SDLK_h,		0x23);
	MAP(SDLK_i,		0x17);
	MAP(SDLK_j,		0x24);
	MAP(SDLK_k,		0x25);
	MAP(SDLK_l,		0x26);
	MAP(SDLK_m,		0x32);
	MAP(SDLK_n,		0x31);
	MAP(SDLK_o,		0x18);
	MAP(SDLK_p,		0x19);
	MAP(SDLK_q,		0x10);
	MAP(SDLK_r,		0x13);
	MAP(SDLK_s,		0x1f);
	MAP(SDLK_t,		0x14);
	MAP(SDLK_u,		0x16);
	MAP(SDLK_v,		0x2f);
	MAP(SDLK_w,		0x11);
	MAP(SDLK_x,		0x2d);
	MAP(SDLK_y,		0x15);
	MAP(SDLK_z,		0x2c);
	MAP(SDLK_DELETE,	0xd3);
	MAP(SDLK_KP0,		0x52);
	MAP(SDLK_KP1,		0x4f);
	MAP(SDLK_KP2,		0x50);
	MAP(SDLK_KP3,		0x51);
	MAP(SDLK_KP4,		0x4b);
	MAP(SDLK_KP5,		0x4c);
	MAP(SDLK_KP6,		0x4d);
	MAP(SDLK_KP7,		0x47);
	MAP(SDLK_KP8,		0x48);
	MAP(SDLK_KP9,		0x49);
	MAP(SDLK_KP_PERIOD,	0x53);
	MAP(SDLK_KP_DIVIDE,	0xb5);
	MAP(SDLK_KP_MULTIPLY,	0x37);
	MAP(SDLK_KP_MINUS,	0x4a);
	MAP(SDLK_KP_PLUS,	0x4e);
	MAP(SDLK_KP_ENTER,	0x9c);
	//MAP(SDLK_KP_EQUALS,	);
	MAP(SDLK_UP,		0xc8);
	MAP(SDLK_DOWN,		0xd0);
	MAP(SDLK_RIGHT,		0xcd);
	MAP(SDLK_LEFT,		0xcb);
	MAP(SDLK_INSERT,	0xd2);
	MAP(SDLK_HOME,		0xc7);
	MAP(SDLK_END,		0xcf);
	MAP(SDLK_PAGEUP,	0xc9);
	MAP(SDLK_PAGEDOWN,	0xd1);
	MAP(SDLK_F1,		0x3b);
	MAP(SDLK_F2,		0x3c);
	MAP(SDLK_F3,		0x3d);
	MAP(SDLK_F4,		0x3e);
	MAP(SDLK_F5,		0x3f);
	MAP(SDLK_F6,		0x40);
	MAP(SDLK_F7,		0x41);
	MAP(SDLK_F8,		0x42);
	MAP(SDLK_F9,		0x43);
	MAP(SDLK_F10,		0x44);
	MAP(SDLK_F11,		0x57);
	MAP(SDLK_F12,		0x58);
	MAP(SDLK_NUMLOCK,	0x45);
	MAP(SDLK_CAPSLOCK,	0x3a);
	MAP(SDLK_SCROLLOCK,	0x46);
	MAP(SDLK_RSHIFT,	0x36);
	MAP(SDLK_LSHIFT,	0x2a);
	MAP(SDLK_RCTRL,		0x9d);
	MAP(SDLK_LCTRL,		0x1d);
	MAP(SDLK_RALT,		0xb8);
	MAP(SDLK_LALT,		0x38);
	MAP(SDLK_LSUPER,	0xdb);	// win l
	MAP(SDLK_RSUPER,	0xdc);	// win r
	MAP(SDLK_PRINT,		-2);	// 0xaa + 0xb7
	MAP(SDLK_SYSREQ,	0x54);	// alt+printscr
	MAP(SDLK_BREAK,		0xb7);	// ctrl+pause
	MAP(SDLK_MENU,		0xdd);	// win menu?
#undef MAP

	return 0;
}


//
// initinput() -- init input system
//
int initinput(void)
{
	buildkeytranslationtable();
	if (SDL_EnableKeyRepeat(250, 30)) printOSD("Error enabling keyboard repeat.\n");
	inputdevices = 1|2;	// keyboard (1) and mouse (2)

	return 0;
}

//
// uninitinput() -- uninit input system
//
void uninitinput(void)
{
}

//
// setkeypresscallback() -- sets a callback which gets notified when keys are pressed
//
void setkeypresscallback(void (*callback)(long, long))
{
	keypresscallback = callback;
}

//
// setmousepresscallback() -- sets a callback which gets notified when a mouse button is pressed
//
void setmousepresscallback(void (*callback)(long, long))
{
	mousepresscallback = callback;
}

//
// handleevents() -- process the SDL message queue
//   returns !0 if there was an important event worth checking (like quitting)
//
int handleevents(void)
{
	int code, rv=0;

#define SetKey(key,state) { \
	keystatus[key] = state; \
		if (state) { \
	keyfifo[keyfifoend] = key; \
	keyfifo[(keyfifoend+1)&(KEYFIFOSIZ-1)] = state; \
	keyfifoend = ((keyfifoend+2)&(KEYFIFOSIZ-1)); \
		} \
}

	while (SDL_PollEvent(&sdl_event)) {
		switch (sdl_event.type) {
			case SDL_KEYDOWN:
			case SDL_KEYUP:
				code = keytranslation[sdl_event.key.keysym.sym];

				// hook in the osd
				if (OSD_HandleKey(code, (sdl_event.key.type == SDL_KEYDOWN)) == 0)
					break;

				if (sdl_event.key.type == SDL_KEYDOWN) {
					if (code == -2) {
						// printscreen
						if (!keystatus[0x36] && !keystatus[0x2a])
							SetKey(0xaa, 1);

						SetKey(0xb7, 1);
						if (keypresscallback)
							keypresscallback(0xb7, 1);
					} else if (!keystatus[code]) {
						SetKey(code, 1);
						if (keypresscallback)
							keypresscallback(code, 1);
					}
				} else {
					if (code == -2) {
						// printscreen
						if (!keystatus[0x36] && !keystatus[0x2a])
							SetKey(0xaa, 0);

						SetKey(0xb7, 0);
						if (keypresscallback)
							keypresscallback(0xb7, 0);
					} else {
						SetKey(code, 0);
						if (keypresscallback)
							keypresscallback(code, 0);
					}
				}
				break;

			case SDL_ACTIVEEVENT:
				// if we lose/gain mouse focus, big woop.
				if ((sdl_event.active.state & SDL_APPACTIVE) ||
				    (sdl_event.active.state & SDL_APPINPUTFOCUS)) {
					appactive = (sdl_event.active.gain == 1);

					if (appactive) grabmouse(1);
					else grabmouse(0);

					rv=-1;
				}
				break;

			case SDL_MOUSEBUTTONDOWN:
			case SDL_MOUSEBUTTONUP:
				if (sdl_event.button.state == SDL_PRESSED)
					mouseb |= (1<<(sdl_event.button.button-1));
				else
					mouseb &= ~(1<<(sdl_event.button.button-1));

				if (mousepresscallback)
					mousepresscallback(sdl_event.button.button, sdl_event.button.state == SDL_PRESSED);
				break;
				
			case SDL_MOUSEMOTION:
				mousex += sdl_event.motion.xrel;
				mousey += sdl_event.motion.yrel;
				break;

			case SDL_QUIT:
				quitevent = 1;
				rv=-1;
				break;

			default:
				//printOSD("Got event (%d)\n", sdl_event.type);
				break;
		}
	}

	sampletimer();

#undef SetKey

	return rv;
}


//
// initmouse() -- init mouse input
//
int initmouse(void)
{
	if (moustat) return 0;

	printOSD("Initializing mouse\n");

	// grab input
	grabmouse(0);
	moustat=1;

	return 0;
}

//
// uninitmouse() -- uninit mouse input
//
void uninitmouse(void)
{
	if (!moustat) return;

	grabmouse(1);
	moustat=0;
}


//
// grabmouse() -- show/hide mouse cursor
//
void grabmouse(char a)
{
	if (a && !cursorvisible) {
		SDL_ShowCursor(SDL_ENABLE);
		SDL_WM_GrabInput(SDL_GRAB_OFF);
	} else if (!a && cursorvisible) {
		SDL_ShowCursor(SDL_DISABLE);
		SDL_WM_GrabInput(SDL_GRAB_ON);
	}

	cursorvisible = a;
}


//
// readmousexy() -- return mouse motion information
//
void readmousexy(short *x, short *y)
{
	//SDL_GetRelativeMouseState((int *)x, (int *)y);
	*x = mousex;
	*y = mousey;
	mousex = mousey = 0;
}

//
// readmousebstatus() -- return mouse button information
//
void readmousebstatus(short *b)
{
	//*b = SDL_GetMouseState(NULL, NULL);
	*b = mouseb;
}


//  This timer stuff is all Ken's idea.

//
// inittimer() -- initialize timer
//
int inittimer(int tickspersecond)
{
	if (timerfreq) return 0;	// already installed

	printOSD("Initializing timer\n");

	timerfreq = 1000;
	timerticspersec = tickspersecond;
	timerlastsample = SDL_GetTicks() * timerticspersec / timerfreq;

	usertimercallback = NULL;

	return 0;
}

//
// uninittimer() -- shut down timer
//
void uninittimer(void)
{
	if (!timerfreq) return;

	timerfreq=0;
}

//
// sampletimer() -- update totalclock
//
void sampletimer(void)
{
	Uint32 i;
	long n;
	
	if (!timerfreq) return;
	
	i = SDL_GetTicks();
	n = (long)(i * timerticspersec / timerfreq) - timerlastsample;
	if (n>0) {
		totalclock += n;
		timerlastsample += n;
	}

	if (usertimercallback) for (; n>0; n--) usertimercallback();
}

//
// getticks() -- returns the sdl ticks count
//
unsigned long getticks(void)
{
	// 32-bit version
	return (unsigned long)((SDL_GetTicks()&0xffffl)*0xffffl/(timerfreq&0xffffl))<<16;

	// 64-bit version
	// return (unsignedlong)((int64)SDL_GetTicks()*0xffffffffll / (int64)timerfreq);
}


//
// installusertimercallback() -- set up a callback function to be called when the timer is fired
//
void (*installusertimercallback(void (*callback)(void)))(void)
{
	void (*oldtimercallback)(void);

	oldtimercallback = usertimercallback;
	usertimercallback = callback;

	return oldtimercallback;
}

//
// checkvideomode() -- makes sure the video mode passed is legal
//
int checkvideomode(long *x, long *y, int fs)
{
	int i, nearest=-1, dx, dy, odx=9999, ody=9999;

	getvalidmodes();

	// fix up the passed resolution values to be multiples of 8
	// and at least 320x200 or at most MAXXDIMxMAXYDIM
	if (*x < 320) *x = 320;
	if (*y < 200) *y = 200;
	if (*x > MAXXDIM) *x = MAXXDIM;
	if (*y > MAXYDIM) *y = MAXYDIM;
	*x &= 0xfffffff8l;

#ifdef ANY_WINDOWED_SIZE
	if (fs) {
#endif
		for (i=0; i<validmodecnt; i++) {
			if (validmodefs[i] != fs) continue;
			dx = klabs(validmodexdim[i] - *x);
			dy = klabs(validmodeydim[i] - *y);
			if (!(dx | dy)) { 	// perfect match
				nearest = i;
				break;
			}
			if ((dx <= odx) && (dy <= ody)) {
				nearest = i;
				odx = dx; ody = dy;
			}
		}

		if (nearest < 0) {
			// no mode that will match (eg. if no fullscreen modes)
			return -1;
		}
		*x = validmodexdim[nearest];
		*y = validmodeydim[nearest];
#ifdef ANY_WINDOWED_SIZE
	}
#endif

	return 0;
}


//
// setvideomode() -- set SDL video mode
//
int setvideomode(int x, int y, int fs)
{
	static char t[384];
	char flags[512] = "";
	char oldvis = cursorvisible;

	if (!oldvis) grabmouse(1);

	if (lockcount) while (lockcount) enddrawing();

	printOSD("Setting video mode %dx%d (%s)\n", x,y, ((fs) ? "fullscreen" : "windowed"));
	sdl_surface = SDL_SetVideoMode(x, y, 8, SURFACE_FLAGS | ((fs) ? SDL_FULLSCREEN : 0));
	if (!sdl_surface) {
		printOSD("Unable to set video mode!\n");
		return -1;
	}

	if (!oldvis) grabmouse(0);

#if 0
#define FLAG(x,y) if ((sdl_surface->flags & x) == x) { strcat(flags, y); strcat(flags, " "); }
	FLAG(SDL_SWSURFACE, "SWSURFACE")
	FLAG(SDL_HWSURFACE, "HWSURFACE")
	FLAG(SDL_ASYNCBLIT, "ASYNCBLIT")
	FLAG(SDL_ANYFORMAT, "ANYFORMAT")
	FLAG(SDL_HWPALETTE, "HWPALETTE")
	FLAG(SDL_DOUBLEBUF, "DOUBLEBUF")
	FLAG(SDL_FULLSCREEN, "FULLSCREEN")
	FLAG(SDL_OPENGL, "OPENGL")
	FLAG(SDL_OPENGLBLIT, "OPENGLBLIT")
	FLAG(SDL_RESIZABLE, "RESIZABLE")
	FLAG(SDL_HWACCEL, "HWACCEL")
	FLAG(SDL_SRCCOLORKEY, "SRCCOLORKEY")
	FLAG(SDL_RLEACCEL, "RLEACCEL")
	FLAG(SDL_SRCALPHA, "SRCALPHA")
	FLAG(SDL_PREALLOC, "PREALLOC")
#undef FLAG
	printOSD("Surface flags: %s\n", flags);
#endif

	sprintf(t, "%s (%dx%d %s)", apptitle, x, y, ((fs) ? "fullscreen" : "windowed"));
	SDL_WM_SetCaption(t, 0);

	xres = x;
	yres = y;
	//bytesperline = sdl_surface->pitch;
	//imageSize = bytesperline*yres;
	fullscreen = fs;
	frameplace = 0;
	lockcount = 0;
	modechange=1;
	OSD_ResizeDisplay(xres,yres);

	return 0;
}


//
// getvalidmodes() -- figure out what video modes are available
//
void getvalidmodes(void)
{
	static int modeschecked=0;
	SDL_Rect **modes;
	SDL_PixelFormat pf = { NULL, 8, 1, 0,0,0,0, 0,0,0,0, 0,0,0,0, 0,0 };
	int i, maxx=0, maxy=0;


	if (modeschecked) return;

	validmodecnt=0;

#define ADDMODE(x,y,f) { \
	validmodexdim[validmodecnt]=x; \
	validmodeydim[validmodecnt]=y; \
	validmodefs[validmodecnt]=f; \
	validmodecnt++; \
	printOSD("Adding mode %dx%d (%s)\n", x, y, (f)?"fullscreen":"windowed"); }

#define CHECK(w,h) if ((w <= maxx) && (h <= maxy))

	// do fullscreen modes first
	modes = SDL_ListModes(&pf, SURFACE_FLAGS | SDL_FULLSCREEN);
	if (modes != (SDL_Rect **)0) {
		if (modes == (SDL_Rect **)-1) {
			ADDMODE(1280,1024,1)
			ADDMODE(1280,960,1)
			ADDMODE(1152,864,1)
			ADDMODE(1024,768,1)
			ADDMODE(800,600,1)
			ADDMODE(640,480,1)
			ADDMODE(640,400,1)
			ADDMODE(512,384,1)
			ADDMODE(480,360,1)
			ADDMODE(400,300,1)
			ADDMODE(320,240,1)
			ADDMODE(320,200,1)
		} else {
			for (i=0; modes[i+1]; i++) ;
			for (; i>=0; i--) {
				if ((modes[i]->w > MAXXDIM) || (modes[i]->h > MAXYDIM)) continue;

				ADDMODE(modes[i]->w, modes[i]->h, 1)

				if ((modes[i]->w > maxx) && (modes[i]->h > maxy)) {
					maxx = modes[i]->w;
					maxy = modes[i]->h;
				}
			}
		}
	} else {
		printOSD("No fullscreen modes available!\n");
		maxx=9999; maxy=9999;
	}

	// add windowed modes next
	CHECK(320,200) ADDMODE(320,200,0)
	CHECK(320,240) ADDMODE(320,240,0)
	CHECK(400,300) ADDMODE(400,300,0)
	CHECK(480,360) ADDMODE(480,360,0)
	CHECK(512,384) ADDMODE(512,384,0)
	CHECK(640,400) ADDMODE(640,400,0)
	CHECK(640,480) ADDMODE(640,480,0)
	CHECK(800,600) ADDMODE(800,600,0)
	CHECK(1024,768) ADDMODE(1024,768,0)
	CHECK(1152,864) ADDMODE(1152,864,0)
	CHECK(1280,960) ADDMODE(1280,960,0)
	CHECK(1280,1024) ADDMODE(1280,1024,0)

#undef CHECK
#undef ADDMODE

	modeschecked=1;
}


//
// begindrawing() -- locks the framebuffer for drawing
//
void begindrawing(void)
{
	long i,j;

	// lock the frame
	if (lockcount++ > 0)
		return;

	if (offscreenrendering) return;
	
	if (SDL_MUSTLOCK(sdl_surface)) SDL_LockSurface(sdl_surface);
	frameplace = sdl_surface->pixels;
	
	if (sdl_surface->pitch != bytesperline || modechange) {
		bytesperline = sdl_surface->pitch;
		imageSize = bytesperline*yres;
		setvlinebpl(bytesperline);

		j = 0;
		for(i=0;i<=ydim;i++) ylookup[i] = j, j += bytesperline;
		modechange=0;
	}
}


//
// enddrawing() -- unlocks the framebuffer
//
void enddrawing(void)
{
	if (!frameplace) return;
	if (lockcount > 1) { lockcount--; return; }
	if (!offscreenrendering) frameplace = 0;
	if (lockcount == 0) return;
	lockcount = 0;

	if (offscreenrendering) return;

	if (SDL_MUSTLOCK(sdl_surface)) SDL_UnlockSurface(sdl_surface);
}


//
// showframe() -- update the display
//
void showframe(int w)
{
	long i,j;

	if (offscreenrendering) return;

	if (lockcount) {
		printf("Frame still locked %d times when showframe() called.\n", lockcount);
		while (lockcount) enddrawing();
	}

	SDL_UpdateRect(sdl_surface, 0,0,0,0);
}


//
// setpalette() -- set palette values
//
int setpalette(int start, int num, char *dapal)
{
	SDL_Color pal[256];
	int i;

	dapal += (start<<2);
	for (i=0; i<num; i++) {
		pal[i].b = dapal[0] << 2;
		pal[i].g = dapal[1] << 2;
		pal[i].r = dapal[2] << 2;
		dapal += 4;
	}

	return SDL_SetPalette(sdl_surface, SDL_LOGPAL|SDL_PHYSPAL, pal, start, num);
}

//
// getpalette() -- get palette values
//
int getpalette(int start, int num, char *dapal)
{
	int i;
	SDL_Palette *pal;

	// we shouldn't need to lock the surface to get the palette
	pal = sdl_surface->format->palette;

	for (i=num; i>0; i--, start++) {
		dapal[0] = pal->colors[start].b >> 2;
		dapal[1] = pal->colors[start].g >> 2;
		dapal[2] = pal->colors[start].r >> 2;
		dapal += 4;
	}

	return 1;
}

/*
static SDL_Surface * loadtarga(const char *fn)
{
	int i, width, height, palsize;
	char head[18];
	SDL_Color palette[256];
	SDL_Surface *surf=0;
	long fil;

	clearbufbyte(palette, sizeof(palette), 0);

	if ((fil = kopen4load(fn,0)) == -1) return NULL;

	kread(fil, head, 18);
	if (head[0] > 0) klseek(fil, head[0], SEEK_CUR);

	if ((head[1] != 1) ||		// colormap type
		(head[2] != 1) ||	// image type
		(head[7] != 24)	||	// colormap entry size
		(head[16] != 8)) {	// image descriptor
		printOSD("%s in unsuitable format for icon\n", fn);
		goto nogo;
	}

	width = head[12] | ((int)head[13] << 8);
	height = head[14] | ((int)head[15] << 8);

	if (width != 32 && height != 32) goto nogo;

	surf = SDL_CreateRGBSurface(SDL_SWSURFACE, width, height, 8, 0,0,0,0);
	if (!surf) goto nogo;

	// palette first
	palsize = head[5] | ((int)head[6] << 8);
	for (i=(head[3]|((int)head[4]<<8)); palsize; --palsize, ++i) {
		kread(fil, &palette[i].b, 1);		// b
		kread(fil, &palette[i].g, 1);		// g
		kread(fil, &palette[i].r, 1);		// r
	}

	// targa renders bottom to top, from left to right
	for (i=height-1; i>=0; i--) {
		kread(fil, surf->pixels+i*width, width);
	}
	SDL_SetPalette(surf, SDL_PHYSPAL|SDL_LOGPAL, palette, 0, 256);

nogo:
	kclose(fil);
	return(surf);
}
*/
