/*
 * JonoF's Nasty-Ass Multiplayer Module
 * for the Build Engine
 * Copyright (c) 2003 Jonathon Fowler
 *
 * See BUILDLIC.TXT for license details.
 */

/*
 Packet Structure:

 unsigned long  crc;			// crc32 of the entire packet data (including header, but excluding this property)
 unsigned short packetlength;		// total length of data this packet ought to contain (including this field)
 unsigned char  messagecount;		// number of messages contained in this packet
 {
 unsigned char  sequence;		// sequence number of the message
 unsigned char  length;			// the length of the message minus 1
 unsigned char  flags;			// bits 0 & 1 represent message type
					//   00 = represents a synchronous message.
					//   01 = represents a message which doesn't affect sync.
					//   10 = represents an oob message which uses the oob message format.
					//   11 = undefined
 unsigned char  messagedata[length+1];	// data for this message
 } * messagecount;			// sorted ascending by sequence number
 unsigned char  acklen;			// byte-length of unacknowledged list
 unsigned char  ack[];			// acknowledged sequence numbers, array size being calculated
					//   from (packetlength - headersize - messages - 1(acklen))


 Out-of-band messages:

 unsigned char  type;			// oob message type byte
*/

//#define DUMMY
#define DUMPEACHPACKET
//#define FAKESEND
#define PEDANTIC

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "osd.h"
#include "compat.h"
#include "crc32.h"

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winsock2.h>
#define ER(x) WSA##x
#define ERS(x) "WSA" x
#else
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#define ER(x) x
#define ERS(x) x
#endif


// data structure to assist in constructing the acknowledgement lists
#define ACKTYPE		char
#define MAXACK		((1ul << (sizeof(ACKTYPE) << 3)) - 1)
#define MAXACK_BYTE	((1ul << (sizeof(ACKTYPE) << 3)) >> 3)
typedef unsigned char ackset[MAXACK_BYTE];
#define ackset_clear(s) memset((s), 0, MAXACK_BYTE)
#define ackset_set(s,f) (s)[ (f) >> 3 ] |= (1 << ((f) & 7))
#define ackset_unset(s,f) (s)[ (f) >> 3 ] &= ~(1 << ((f) & 7))
#define ackset_test(s,f) ((s)[ (f) >> 3 ] & (1 << ((f) & 7)))



// address structure opaque to the game
#define ADDR_NONE (0)
#define ADDR_IPV4 (1)
//#define ADDR_IPV6 (2)
typedef struct {
	unsigned short type;
	union {
		struct sockaddr_in ipv4;
		//struct sockaddr_in6 ipv6;
	} addr;
} multiaddr;


// include this here so we define multiaddr beforehand
#define IN_MMULTI
#include "mmulti.h"


#define MaxCatchUpBacklog 64
#define MaxMessageLen     256	// unsigned char range is 0-255 and we store length as 1 less

#define MessageHeaderLength (1+1+1)
#define HeaderLength      (4+2+1)
#define TrailerLength     (1+1+32+1+32)

// the maximum length of message data (including message headers) to allow in a packet
// before it is considered full. this keeps message sizes smaller so that we don't
// hit fragmentation issues to do with UDP packets getting bigger than MTUs
#define MaxPacketSize     ((MessageHeaderLength+MaxMessageLen)*2)

#define MaxFullPacketLen  (HeaderLength+MaxPacketSize+TrailerLength)

#define loopcirculartail(start,end,len)  ((start)>(end)?((end)+(len)):(end))


// message backups management
#define BACKLISTSIZE (MAXMULTIPLAYERS*MaxCatchUpBacklog*2)
static int backlisthead,
           backlisthandles[BACKLISTSIZE];
static struct {
	char buf[MaxMessageLen];
	unsigned char length;
	unsigned char flags;
	unsigned char sequence;
	unsigned char age;
} backlist[BACKLISTSIZE];


typedef struct {
	unsigned char acksequence;
	// The last acknowleged sequence number from this peer. This gets incremented when
	// the oldest message is acknowledged.

	unsigned char sequence;
	// The next available sequence number for transmitting a message to this peer. This
	// gets incremented after each new message.
	// Any message with a sequence number which does not fall within the circular range
	// starting at 'acksequence' and ending at 'sequence-1' is illegal and should be
	// dropped silently.

	unsigned char retransmitage;
	// The age at which, if no ack is received, the message is retransmitted.
	// Because of the variable latency compensation this allows for delaying message
	// retransmission.

	short messagebackups[256];
	// The backed-up messages to be sent. As a message is acknowledged its handle in this
	// array is set to -1, and if the acknowledgement was for the oldest message, ie. the
	// sequence number given in 'acksequence', then 'acksequence' gets incremented to the
	// next unacknowledged message.
	
	unsigned char toacknowledge[256], toacknowledgehead;
	// Array of flags indicating which packets are to be acknowledged on the next send.
	
	short incomingqueue[256];
	unsigned char incominghead;
	// The array of incoming queued messages. This exists in case some messages come out
	// of order and we have a gap between where we expect the next message to be and what
	// has come. The sequence number the message arrives with is the index it resides in
	// this array. The array contains handles from the same pool that the messagebackups
	// array draws from. If the handle which incominghead addresses is -1 then we have to
	// wait until it becomes the handle to a message before we can continue.

	multiaddr address;
	// The Internet address and port for this peer.
	
	unsigned char workpacket[MaxFullPacketLen];
	unsigned char *workdata;
	short workpacketlen;
	// The scratch space for assembling a packet for sending to the peer.
	
	ackset hassent;
	// Used to prevent duplicate transmission of messages in a packet.
} peer_t;

#define set_packet_crc(pkt,crc) \
{ \
	(pkt)[0] = (crc) & 255; \
	(pkt)[1] = ((crc) >> 8) & 255; \
	(pkt)[2] = ((crc) >> 16) & 255; \
	(pkt)[3] = ((crc) >> 24) & 255; \
}

#define get_packet_crc(pkt) \
( \
	((unsigned long)(pkt)[0]) | \
	((unsigned long)(pkt)[1] << 8) | \
	((unsigned long)(pkt)[2] << 16) | \
	((unsigned long)(pkt)[3] << 24) \
)

#define set_packet_length(pkt,len) \
{ \
	(pkt)[4+0] = (len) & 255; \
	(pkt)[4+1] = ((len) >> 8) & 255; \
}

#define get_packet_length(pkt) \
( \
	((unsigned short)(pkt)[4+0]) | \
	((unsigned short)(pkt)[4+1] << 8) \
)

#define packet_msgcount(pkt) \
	(pkt)[4+2]

static peer_t peers[MAXMULTIPLAYERS];

#define comparesequences(a,op,b) ((signed char)((a) - (b)) op 0)
inline int willfit(short p, unsigned l) { return ((peers[p].workpacketlen + (l) + 3) <= MaxPacketSize); }

char  multioption = 0;	// single-player by default
short myconnectindex, numplayers;
short connecthead, connectpoint2[MAXMULTIPLAYERS];
char  syncstate = 0;

int   ipport = 0x4A46;

static int sk = -1;

#ifdef WINDOWS
static char winsockinited = 0;
#endif


extern long totalclock;		// from the game, for message aging
static long ototalclock;

//////////////////////
// Internal functions
//

#define FLAG_TYPE_MASK   (3)
#define FLAG_TYPE_SYNC   (0)
#define FLAG_TYPE_UNSYNC (1)
#define FLAG_TYPE_OOB    (2)

static int startnetwork(void);
static void stopnetwork(void);
	
static void networksend(multiaddr *addr, void *buf, unsigned len);
static int networkrecv(multiaddr *addr, void *buf, unsigned *len);

static int dosendpacket(short other, short handle);

static const char *networkerrorstr(int c)
{
	return "Other error";
}

//
// clearpeer() -- Clears a peer
//
static void clearpeer(short i)
{
	peers[i].acksequence = 0;
	peers[i].sequence    = 0;
	
	peers[i].retransmitage = 2;

	memset(peers[i].messagebackups, -1, 256*sizeof(short));
	
	memset(peers[i].toacknowledge, 0, 256*sizeof(char));
	peers[i].toacknowledgehead = 0;

	memset(peers[i].incomingqueue, -1, 256*sizeof(short));
	peers[i].incominghead = 0;
	
	memset(&peers[i].address, 0, sizeof(multiaddr));

	// set the workpacketdata pointer to the beginning of where
	// the message payload would reside in the work buffer.
	// this is so we can fill in the rest of the packet details
	// with zero copying when we send.
	peers[i].workdata = peers[i].workpacket + 4+2+1;
	memset(peers[i].workpacket, 0, sizeof(peers[i].workpacket));
	peers[i].workpacketlen = 0;

	ackset_clear(peers[i].hassent);
}

//
// whichpeer() -- Internal function to determine which peer is being addressed
//   by the passed address.
//
static short whichpeer(multiaddr *addr)
{
	short peer;

	for (peer=connecthead; peer!=-1; peer=connectpoint2[peer]) {
		if (!memcmp(&peers[peer].address, addr, sizeof(multiaddr)))
			return peer;
	}

	return -1;
}

//
// backupmessage() -- Internal function to backup a message buffer and return
//   the handle of the backup buffer.
//
static short backupmessage(char *buf, long len, unsigned char flags, unsigned char sequence)
{
	short i;

	if (backlisthead == 0) {
		OSD_Printf("backupmessage(): backup handles exhausted!!!\n");
		return -1;	// no backup space left
	}
	
	i = backlisthandles[--backlisthead];
	
	// copy data into a backup buffer and point backlistptrs[i] to it
	memcpy(backlist[i].buf, buf, len);
	backlist[i].length = len-1;
	backlist[i].flags = flags;
	backlist[i].sequence = sequence;
	backlist[i].age = 0;

	return i;
}


//
// freemessage() -- Internal function to free a message backup buffer given
//   by the handle.
//
static void freemessage(short handle)
{
	int i;
	
	if (handle < 0) return;

	// sanity check for already freed handle
#ifdef PEDANTIC
	for (i=0;i<backlisthead;i++)
		if (backlisthandles[i] == handle) return;
#endif

	backlisthandles[backlisthead++] = handle;
}


//
// agemessages() -- Increments age counters and stuff.
//
static void agemessages(void)
{
	short i;
	unsigned char inc;

	inc = (unsigned char)(totalclock - ototalclock);
	if (!inc) return;

	for (i=0; i<BACKLISTSIZE; i++)
		backlist[i].age += inc;

	ototalclock += inc;
}


//
// housekeep() -- Dusts the furniture, re-sorts the sock drawer.
//
static void housekeep(void)
{
	short peer;
	int j,jj;

	agemessages();
	
	// check to see if any messages need resending and begin stuffing packets
	for (peer=connecthead; peer!=-1; peer=connectpoint2[peer]) {
		for (j=peers[peer].acksequence;
		     j < loopcirculartail(peers[peer].acksequence, peers[peer].sequence, 256);
		     j++) {
			jj = j&255;
		
			if (peers[peer].messagebackups[jj] == -1) continue;	// message not backed up

			// test if this message has reached its retransmit age
			if (backlist[ peers[peer].messagebackups[jj] ].age < peers[peer].retransmitage)
				continue;

			dosendpacket(peer, peers[peer].messagebackups[jj]);
		}
	}
}


//
// pushpacket() -- Pushes out a packet if there is some packet data waiting.
//   Tacks on the header and acknowledgement lists, checksums it, and
//   sends it on its merry way.
//
static void pushpacket(short peer)
{
#ifndef DUMMY
	unsigned long crc;
	unsigned char unacklistlen = 0, acklistlen = 0;
	ackset unacklist, acklist;
	int unackfirst = -1, ackfirst = -1;
	int i,j,jj;
	unsigned char c;

	// add the acknowledgement lists
	
	// first, the unacknowledged messages summary
	ackset_clear(unacklist);
	for (j=peers[peer].acksequence;
	     j < loopcirculartail(peers[peer].acksequence, peers[peer].sequence, 256);
	     j++) {
		jj = j&255;
		
		if (peers[peer].messagebackups[jj] == -1) continue;	// message not backed up
		if (unackfirst == -1)
			unackfirst = j;
		else {
			i = j - unackfirst - 1;
			ackset_set(unacklist,i);
			unacklistlen = i+1;
		}
	}
	unacklistlen = ((unacklistlen+7)>>3);

	// then, the acknowledged messages
	ackset_clear(acklist);
	for (j=peers[peer].toacknowledgehead;
	     j < peers[peer].toacknowledgehead+256;
	     j++) {
		jj = j&255;
		
		if (peers[peer].toacknowledge[jj] == 0) continue;	// message not needing acknowledgement
		if (ackfirst == -1)
			ackfirst = j;
		else {
			i = j - ackfirst - 1;
			ackset_set(acklist,i);
			acklistlen = i+1;
		}
	}
	acklistlen = ((acklistlen+7)>>3);

	// packet is empty, so don't send anything
	if (!unacklistlen && !acklistlen && !peers[peer].workpacketlen) return;
	
	// copy the lists into the packet
	if (unackfirst != -1 || ackfirst != -1) {
		peers[peer].workdata[ peers[peer].workpacketlen++ ] = unacklistlen;
		if (unackfirst != -1) {
			peers[peer].workdata[ peers[peer].workpacketlen++ ] = unackfirst & 255;
			for (i=0; i<unacklistlen; i++)
				peers[peer].workdata[ peers[peer].workpacketlen++ ] = unacklist[i];
		}
	
		if (ackfirst != -1) {
			peers[peer].workdata[ peers[peer].workpacketlen++ ] = ackfirst & 255;
			for (i=0; i<acklistlen; i++)
				peers[peer].workdata[ peers[peer].workpacketlen++ ] = acklist[i];
		}
	}

	// set the packet length word
	peers[peer].workpacketlen += 2+1;	// header size
	set_packet_length(peers[peer].workpacket, peers[peer].workpacketlen);
	
	// the message count byte is already initialized

	// crc32 the data
	peers[peer].workpacketlen += 4;
	set_packet_length(peers[peer].workpacket, peers[peer].workpacketlen);
	crc = crc32(&peers[peer].workpacket[4], peers[peer].workpacketlen-4);
	set_packet_crc(peers[peer].workpacket, crc);

	// send the packet
	networksend(&peers[peer].address, peers[peer].workpacket, peers[peer].workpacketlen);

	// reset the world for the next time
	peers[peer].workpacketlen = 0;
	//memset(peers[peer].workpacket, 0, sizeof(peers[peer].workpacket));
	packet_msgcount(peers[peer].workpacket) = 0;
	memset(peers[peer].toacknowledge, 0, 256*sizeof(char));

	ackset_clear(peers[peer].hassent);
#endif
}


//
// putpackets() -- Pushes out any packets waiting on all peers.
//
static void pushpackets(void)
{
	short peer;

	for (peer=connecthead; peer!=-1; peer=connectpoint2[peer]) {
		pushpacket(peer);
	}
}


//
// processoobmessage() -- Processes a message marked out-of-band
// 
static void processoobmessage(multiaddr *addr, unsigned char *buf, unsigned int len)
{
}


//
// dosendpacket() -- Internal function to send a packet
//
static int dosendpacket(short other, short handle)
{
	if (ackset_test(peers[other].hassent, backlist[handle].sequence))
		return 0;	// this message has already been sent
	
	// see if we can fit the message in the current packet
	if (!willfit(other, (long)backlist[handle].length+1)) {
		// the message won't fit in the current packet so flush
		// what's waiting to the output and then proceed
		pushpacket(other);
	}

	// copy the new message into the packet
	peers[other].workdata[ peers[other].workpacketlen++ ] = backlist[handle].sequence;
	peers[other].workdata[ peers[other].workpacketlen++ ] = backlist[handle].length;
	peers[other].workdata[ peers[other].workpacketlen++ ] = backlist[handle].flags;
	memcpy( &peers[other].workdata[ peers[other].workpacketlen ], backlist[handle].buf, (long)backlist[handle].length+1);
	peers[other].workpacketlen += (long)backlist[handle].length+1;
	packet_msgcount(peers[other].workpacket)++;			// increment included message count
	
	ackset_set(peers[other].hassent, backlist[handle].sequence);

	return 0;
}


//
// dogetpacket() -- Internal function to get a packet and unpack it
//
static int dogetpacket(void)
{
	int rv;
	unsigned long crc;
	unsigned char count;

	multiaddr addr;
	short peer;

	unsigned char workbuf[MaxFullPacketLen], *workptr;
	unsigned workbuflen;
	unsigned char sequence;
	unsigned int  length;
	unsigned char flags;
	short han;

	// fetch a waiting packet into the working buffer
	workbuflen = MaxFullPacketLen;
	if ((rv = networkrecv(&addr, workbuf, &workbuflen)) > 0) {
		printf("Got %d bytes from network\n", rv);

		// check the crc
		crc = crc32(&workbuf[4], workbuflen-4);
		if (crc != get_packet_crc(workbuf)) return 0;		// corrupt or truncated

		peer = whichpeer(&addr);

		// put messages into waiting queue
		workptr = workbuf + HeaderLength;
		for (count=0; count < packet_msgcount(workbuf); count++, workptr+=MessageHeaderLength+length) {
			sequence = workptr[0];
			length   = (unsigned int)workptr[1] + 1;
			flags    = workptr[2];

			if (length > (workbuflen - (workptr-workbuf))) {
				printf("Oi! %d %d\n", length, (workbuflen - (workptr-workbuf)));
				break;	// someone trying to f*ck with us by mis-setting the message length byte
			}

			if (flags & FLAG_TYPE_MASK == FLAG_TYPE_OOB) {
				processoobmessage(&addr, workptr + MessageHeaderLength, length);
			} else {
				if (peer < 0) continue;	// can't take game messages from a host not in the game
				
				if (peers[peer].incomingqueue[ sequence ] != -1) {
					// message already pending for this sequence. drop it on the floor
				} else {
					// queue message for getpacket() to retrieve later
					han = backupmessage(workptr + MessageHeaderLength, length, flags, sequence);

					// put handle into the incoming queue
					peers[peer].incomingqueue[ sequence ] = han;
					
					printf("Unpacked %d byte message (sequence %d) handle %d\n", length, sequence, han);
				}
			}
		}
		
		// process the acknowledgement lists
	} else {
		if (rv < 0) return rv;
	}

	return 0;
}



//
// startnetwork() -- Initialises networking stuff.
//
static int startnetwork(void)
{
#ifdef WINDOWS
	WSADATA wsa;
	int rv;
	char *e;
	struct sockaddr_in sadr;

	if (!winsockinited) {
		rv = WSAStartup(MAKEWORD(2,2), &wsa);
		if (rv) {
			switch (rv) {
				case WSASYSNOTREADY: e = "Networking subsystem not ready (WSASYSNOTREADY)"; break;
				case WSAVERNOTSUPPORTED: e = "Requested Winsock version not available (WSAVERNOTSUPPORTED)"; break;
				case WSAEINPROGRESS: e = "Winsock 1.1 operation in progress (WSAINPROGRESS)"; break;
				case WSAEPROCLIM: e = "Task limit reached (WSAPROCLIM)"; break;
				case WSAEFAULT: e = "Invalid data pointer passed (WSAEFAULT)"; break;
				default: e = "Unknown Winsock error"; break;
			}
			OSD_Printf("Windows Sockets startup error: %s\n", e);
			return -1;
		}
	
		if (wsa.wVersion != MAKEWORD(2,2)) {
			OSD_Printf("Windows Sockets startup error: Incorrect Winsock version returned\n");
			WSACleanup();
			return -1;
		}

		OSD_Printf("Windows Sockets initialized\n"
	           "  Available version:     %d.%d\n"
		   "  Max supported version: %d.%d\n"
		   "  Driver description:    %s\n",
		   LOBYTE(wsa.wVersion), HIBYTE(wsa.wVersion),
		   LOBYTE(wsa.wHighVersion), HIBYTE(wsa.wHighVersion),
		   wsa.szDescription
		  );
	
		winsockinited = 1;
		sk = -1;
	}
#endif

	if (sk == -1) {
		sk = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP);
		if (sk < 0) return -1;
	}

	rv = 1;
#ifdef WINDOWS
	if (ioctlsocket(sk, FIONBIO, (u_long*)&rv) < 0) {
#else
	if (ioctl(sk, FIONBIO, &rv) < 0) {
#endif
		stopnetwork();
		return -1;
	}

	if (setsockopt(sk, SOL_SOCKET, SO_BROADCAST, (char *)&rv, sizeof(rv)) < 0) {
		stopnetwork();
		return -1;
	}
	
	sadr.sin_family = AF_INET;
	sadr.sin_addr.s_addr = INADDR_ANY;
	sadr.sin_port = htons(ipport);
	if (bind(sk, (struct sockaddr *)&sadr, sizeof(sadr)) < 0) {
		stopnetwork();
		return -1;
	}

	OSD_Printf("Networking initialised.\n");

	return 0;
}

//
// stopnetwork() -- Shuts down networking stuff.
//
static void stopnetwork(void)
{
	if (sk != -1) {
#ifdef WINDOWS
		closesocket(sk);
#else
		close(sk);
#endif
		sk = -1;
	}

#ifdef WINDOWS
	if (winsockinited) {
		WSACleanup();
		winsockinited = 0;
	}
#endif
}

//
// networksend() -- Sends a buffer to an address.
//
static void networksend(multiaddr *addr, void *buf, unsigned len)
{
#ifdef DUMPEACHPACKET
	char straddr[64];
	unsigned i;
	int err;
#endif

	if (!addr || !buf) return;

#ifdef DUMPEACHPACKET
	multiaddrtostring(addr, straddr, 64);
	printf("To:     %s\n"
	       "Length: %d\n"
	       "Contents:\n",
	       straddr, len);
	for (i=0;i<len;i++) {
		if (i % 8 == 0) printf("   %4x: ", i);
		printf("%02x ", ((unsigned char *)buf)[i]);
		if (i % 8 == 7) printf("\n");
	}
	if (len&7) printf("\n");
	printf("\n");
#endif
	
	if (addr->type == ADDR_NONE) return;
	if (addr->type == ADDR_IPV4) {
#ifndef FAKESEND
		if ((err = sendto(sk, buf, len, 0, (struct sockaddr *)&addr->addr, sizeof(struct sockaddr_in)))< 0) {
#ifdef WINDOWS
			err = WSAGetLastError();
#else
			err = errno;
#endif
			printf("sendto() error %d\n", err);
			return;
		}
		printf("sendto() returned %d\n",err);
#endif
		return;
	}
}


//
// networkrecv() -- Receives network data into a buffer.
//
static int networkrecv(multiaddr *addr, void *buf, unsigned *len)
{
	int fromlen;
	struct sockaddr *sad;
	int rv;
	
	if (!addr) return -1;

	sad = (struct sockaddr *)&addr->addr;

	fromlen = sizeof(addr->addr);
	rv = recvfrom(sk, buf, *len, 0, sad, &fromlen);

	if (sad->sa_family == AF_INET)
		addr->type = ADDR_IPV4;
	else
		addr->type = ADDR_NONE;

	if (rv >= 0) *len = rv;
	else *len = 0;

	return rv;
}



////////////////////////////
// Basic Game-land functions
//

//
// initmultiplayers() -- Start up multiplayer
//
void initmultiplayers(char damultioption, char dacomrateoption, char dapriority)
{
	int i;

	if (damultioption == 0 && multioption > 0)
		uninitmultiplayers();	// going back to singleplayer
	
	numplayers = 1; myconnectindex = 0;
	connecthead = -1;

	multioption = damultioption;
	if (damultioption == 0) return;
	
	backlisthead = BACKLISTSIZE;
	for (i=0;i<BACKLISTSIZE;i++)
		backlisthandles[i] = BACKLISTSIZE-i-1;

	for (i=0;i<MAXMULTIPLAYERS;i++)
		clearpeer(i);

	if (startnetwork() < 0) {
		OSD_Printf("Network system initialization failed!\n");
		multioption = 0;	// failed initialising network so fall back to singleplayer
	}

	ototalclock = totalclock;
}

//
// uninitmultiplayers() -- Shut down multiplayer
//
void uninitmultiplayers(void)
{
	stopnetwork();
	numplayers = 1; myconnectindex = 0;
	connecthead = 0; connectpoint2[0] = -1;
	multioption = 0;
}

//
// setpackettimeout() -- Unused
//
void setpackettimeout(long datimeoutcount, long daresendagaincount)
{
}

//
// sendlogon() -- Unused
//
void sendlogon(void)
{
}

//
// sendlogoff() - Unused
//
void sendlogoff(void)
{
}

//
// getoutputcirclesize() -- Unused
//
long getoutputcirclesize(void)
{
	return 0;
}

//
// setsocket() -- Unused
//
void setsocket(short newsocket)
{
}

//
// sendpacket() -- The game interface for sending a message
//
void sendpacket(long other, char *bufptr, long messleng)
{
#ifndef DUMMY
	short h;

	if (messleng > MaxMessageLen || messleng == 0) return;
	
#ifdef PEDANTIC
	for (h=connecthead; h!=-1 && h!=other; h=connectpoint2[h]) ;
	if (h==-1) return;
#endif

	// back up the message
	h = backupmessage(bufptr, messleng, 0, peers[other].sequence);

	if (h<0) return;
	
	peers[other].messagebackups[ peers[other].sequence ] = h;	// store the handle
	peers[other].sequence++;	// will auto-wrap

	dosendpacket(other,h);
	
	housekeep();

	pushpackets();
#endif
}

//
// getpacket() -- The game interface for getting a message
//
short getpacket (short *other, char *bufptr)
{
#ifndef DUMMY
	short h,b;
	
	dogetpacket();
	
	housekeep();
	
	pushpackets();

	for (h = connecthead; h != -1; h = connectpoint2[h]) {
		b = peers[h].incomingqueue[ peers[h].incominghead ];
		if (b == -1) continue;

		*other = h;
		memcpy(bufptr, backlist[b].buf, (int)backlist[b].length+1);
		freemessage(b);

		peers[h].incomingqueue[ peers[h].incominghead++ ] = -1;

		return (int)backlist[b].length+1;
	}
#endif
	return 0;
}

//
// flushpackets() -- Unused
//
void flushpackets(void)
{
}

//
// genericmultifunction() -- Interface for sending multiplayer requests
//   using the COMMIT interface
//
#define  CMD_SEND               1
#define  CMD_GET                2
#define  CMD_SENDTOALL          3
#define  CMD_SENDTOALLOTHERS    4
#define  CMD_SCORE              5
void genericmultifunction(long other, char *bufptr, long messleng, long command)
{
	short p;
	
	switch (command) {
		case CMD_GET:	// kinda unused
			break;
				
		case CMD_SEND:
			if (other == 0) other = myconnectindex;
			else other--;
			sendpacket(other, bufptr, messleng);
			break;
			
		case CMD_SENDTOALL:
			for (p = connecthead; p != -1; p = connectpoint2[p])
				sendpacket(p, bufptr, messleng);
			break;
			
		case CMD_SENDTOALLOTHERS:
			for (p = connecthead; p != -1; p = connectpoint2[p])
				if (p != myconnectindex) sendpacket(p, bufptr, messleng);
			break;
			
		case CMD_SCORE:
			break;
	}
}


///////////////////
// Game-land extras

//
// multiaddrtostring() -- Converts the multiaddr address passed into a string
//   which is more human-readable
//
int multiaddrtostring(multiaddr *addr, char *buf, int len)
{
	if (!addr) return -1;
	
	if (addr->type == ADDR_IPV4) {
#ifdef WINDOWS
		if (!WSAAddressToString((LPSOCKADDR)&addr->addr.ipv4, sizeof(struct sockaddr_in), NULL, buf, (LPDWORD)&len))
			return 0;
#else
		char t[64];
		int l=64;
	
		if (inet_ntop(AF_INET, &addr->addr.ipv4, t, l)) {
			snprintf(buf,len,"%s:%d",t,ntohs(addr->addr.ipv4.sin_port));
			return 0;
		}
#endif
	} else {
		strncpy(buf,"Invalid",len);
	}

	return -1;
}


//
// multistringtoaddr() -- Resolves a hostname and stores the result in a multiaddr
//
static int multistringtoaddr(char *str, multiaddr *addr)
{
	struct hostent *he;
	int port = ipport, t;
	char *host, *p, *p2;

	host = alloca(strlen(str)+1);
	strcpy(host,str);

	p = strchr(host,':');
	if (p) {
		*(p++) = 0;
		t = strtol(p, &p2, 10);
		if (p != p2 && *p2 == 0) port = t;
	}

	memset(addr, 0, sizeof(multiaddr));
	he = gethostbyname(host);
	if (!he) {
		switch (h_errno) {
			case HOST_NOT_FOUND: OSD_Printf("Name resolution error: %s: Host not found\n", host); break;
			case NO_DATA: OSD_Printf("Name resolution error: %s: No address for this hostname\n", host); break;
			case NO_RECOVERY: OSD_Printf("Name resolution error: %s: Unrecoverable error\n", host); break;
			case TRY_AGAIN: OSD_Printf("Name resolution error: %s: Temporary name server error\n", host); break;
		}
		
		return -1;
	}

	if (he->h_addrtype == AF_INET) {
		addr->type = ADDR_IPV4;
		addr->addr.ipv4.sin_family = AF_INET;
		memcpy(&addr->addr.ipv4.sin_addr, he->h_addr_list[0], he->h_length);
		addr->addr.ipv4.sin_port = htons(port);
	}

	return 0;
}


//
// multiaddplayer() -- Hack trick to force the engine to recognize a player point
//
void multiaddplayer(char *str)
{
	short p;
	multiaddr addr;

	if (multistringtoaddr(str, &addr)) return;

	if (connecthead == -1) {
		connecthead = 0;
		connectpoint2[0] = -1;
		p = 0;
	} else {
		for (p = connecthead; connectpoint2[p] != -1; p = connectpoint2[p]) ;
		connectpoint2[p] = p+1;
		connectpoint2[p+1] = -1;
		p++;
	}

	peers[p].address = addr;
}


//
// multidumpstats() -- Dumps current status information
//
void multidumpstats(void)
{
	short p,i;
	char adr[32];

	OSD_Printf("Multiplayer system status:\n"
		   "myconnectindex = %d\n"
		   "numplayers = %d\n",
		   myconnectindex,
		   numplayers);

	for (p = connecthead; p != -1; p = connectpoint2[p]) {
		OSD_Printf("Player %d:\n", p);
		OSD_Printf("   acksequence = %d\n"
			   "   sequence = %d\n"
			   "   retransmitage = %d\n"
			   "   messagebackups[] =",
			   	peers[p].acksequence,
				peers[p].sequence,
				peers[p].retransmitage);
		for (i=peers[p].acksequence; i!=peers[p].sequence; i=(i+1)&255)
			OSD_Printf(" %d", peers[p].messagebackups[i]);
		multiaddrtostring(&peers[p].address, adr, 32);
		OSD_Printf("\n"
			   "   address = %s\n", adr);
	}

}

