// On-screen Display (ie. console)
// for the Build Engine
// by Jonathon Fowler (jonof@edgenetwk.com)

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include <stdlib.h>
#include "build.h"
#include "osd.h"
#include "compat.h"


typedef struct _symbol {
	const char *name;
	struct _symbol *next, *nexttype;

	int type;
	union {
		struct {
			int         minparms;
			const char *help;
			int (*func)(const osdfuncparm_t *);
		} func;
		struct {
			int         type;
			void       *var;
			int         extra;
			int (*validator)(void *);
		} var;
	} i;
} symbol_t;

#define SYMBTYPE_FUNC 0
#define SYMBTYPE_VAR  1

static symbol_t *symbols = NULL, *symbols_func = NULL, *symbols_var = NULL;
static symbol_t *addnewsymbol(int type);
static symbol_t *findsymbol(const char *name, symbol_t *startingat);
static symbol_t *findexactsymbol(const char *name);

static int _internal_validate_string(void *);
static int _internal_validate_integer(void *);

static int _validate_osdlines(void *);

static int _internal_osdfunc_listsymbols(const osdfuncparm_t *);
static int _internal_osdfunc_help(const osdfuncparm_t *);


static int white=-1;			// colour of white (used by default display routines)
static void _internal_drawosdchar(int, int, char, int, int);
static void _internal_drawosdstr(int, int, char*, int, int, int);
static void _internal_drawosdcursor(int,int,int,int);
static int _internal_getcolumnwidth(int);
static int _internal_getrowheight(int);
static void _internal_clearbackground(int,int);
static int _internal_gettime(void);
static void _internal_onshowosd(int);

#define TEXTSIZE 16384
#define DEFAULTLOG "console.txt"

// history display
static char osdtext[TEXTSIZE];
static int  osdpos=0;			// position next character will be written at
static int  osdlines=1;			// # lines of text in the buffer
static int  osdrows=20;			// # lines of the buffer that are visible
static int  osdcols=60;			// width of onscreen display in text columns
static int  osdmaxrows=20;		// maximum number of lines which can fit on the screen
static int  osdmaxlines=TEXTSIZE/60;	// maximum lines which can fit in the buffer
static char osdvisible=0;		// onscreen display visible?
static int  osdhead=0; 			// topmost visible line number
static FILE *osdlog=NULL;		// log filehandle
static char osdinited=0;		// text buffer initialised?
static int  osdkey=0x45;		// numlock shows the osd
static int  keytime=0;

// command prompt editing
#define EDITLENGTH 512
static int  osdovertype=0;		// insert (0) or overtype (1)
static char osdeditbuf[EDITLENGTH+1];	// editing buffer
static int  osdeditlen=0;		// length of characters in edit buffer
static int  osdeditcursor=0;		// position of cursor in edit buffer
static int  osdeditshift=0;		// shift state
static int  osdeditcontrol=0;		// control state
static int  osdeditcaps=0;		// capslock
static int  osdeditwinstart=0;
static int  osdeditwinend=60-1-3;
#define editlinewidth (osdcols-1-3)

// command processing
#define HISTORYDEPTH 16
static int  osdhistorypos=-1;		// position we are at in the history buffer
static int  osdhistorybuf[HISTORYDEPTH][EDITLENGTH+1];	// history strings
static int  osdhistorysize=0;		// number of entries in history

// execution buffer
// the execution buffer works from the command history
static int  osdexeccount=0;		// number of lines from the head of the history buffer to execute

// presentation parameters
static int  osdpromptshade=0;
static int  osdpromptpal=0;
static int  osdeditshade=0;
static int  osdeditpal=0;
static int  osdtextshade=0;
static int  osdtextpal=0;
static int  osdcursorshade=0;
static int  osdcursorpal=0;

// application callbacks
static void (*drawosdchar)(int, int, char, int, int) = _internal_drawosdchar;
static void (*drawosdstr)(int, int, char*, int, int, int) = _internal_drawosdstr;
static void (*drawosdcursor)(int, int, int, int) = _internal_drawosdcursor;
static int (*getcolumnwidth)(int) = _internal_getcolumnwidth;
static int (*getrowheight)(int) = _internal_getrowheight;
static void (*clearbackground)(int,int) = _internal_clearbackground;
static int (*gettime)(void) = _internal_gettime;
static void (*onshowosd)(int) = _internal_onshowosd;


// translation table for turning scancode into ascii characters
static char sctoasc[2][256] = {
	{
//      0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f
	0,   27,  '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=', 8,   9,   // 0x00
	'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']', 13,  0,   'a', 's', // 0x10
	'd', 'f', 'g', 'h', 'j', 'k', 'l', ';', '\'','`', 0,   '\\','z', 'x', 'c', 'v', // 0x20
	'b', 'n', 'm', ',', '.', '/', 0,   '*', 0,   ' ', 0,   0,   0,   0,   0,   0,   // 0x30
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   '-', 0,   0,   0,   '+', 0,   // 0x40
	0,   0,   0,   '.', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x50
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x60
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x70
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x80
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   13,  0,   0,   0,   // 0x90
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xa0
	0,   0,   0,   0,   0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xb0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xc0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xd0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xe0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0    // 0xf0
	},
	{
//      0    1    2    3    4    5    6    7    8    9    a    b    c    d    e    f
	0,   27,  '!', '@', '#', '$', '%', '^', '&', '*', '(', ')', '_', '+', 8,   9,   // 0x00
	'Q', 'W', 'E', 'R', 'T', 'Y', 'U', 'I', 'O', 'P', '{', '}', 13,  0,   'A', 'S', // 0x10
	'D', 'F', 'G', 'H', 'J', 'K', 'L', ':', '"', '~', 0,   '|', 'Z', 'X', 'C', 'V', // 0x20
	'B', 'N', 'M', '<', '>', '?', 0,   '*', 0,   ' ', 0,   0,   0,   0,   0,   0,   // 0x30
	0,   0,   0,   0,   0,   0,   0,   '7', '8', '9', '-', '4', '5', '6', '+', '1', // 0x40
	'2', '3', '0', '.', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x50
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x60
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x70
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0x80
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   13,  0,   0,   0,   // 0x90
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xa0
	0,   0,   0,   0,   0,   '/', 0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xb0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xc0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xd0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   // 0xe0
	0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0,   0    // 0xf0
	}
};



static void _internal_drawosdchar(int x, int y, char ch, int shade, int pal)
{
	int i,j,k;
	char st[2] = { ch,0 };

	if (white<0) {
		// find the palette index closest to white
		k=0;
		for(i=0;i<256;i++)
		{
			j = ((int)palette[i*3])+((int)palette[i*3+1])+((int)palette[i*3+2]);
			if (j > k) { k = j; white = i; }
		}
	}

	printext256(4+(x<<3),4+(y<<3), white, -1, st, 0);
}

static void _internal_drawosdstr(int x, int y, char *ch, int len, int shade, int pal)
{
	int i,j,k;
	char st[1024];

	if (len>1023) len=1023;
	memcpy(st,ch,len);
	st[len]=0;
	
	if (white<0) {
		// find the palette index closest to white
		k=0;
		for(i=0;i<256;i++)
		{
			j = ((int)palette[i*3])+((int)palette[i*3+1])+((int)palette[i*3+2]);
			if (j > k) { k = j; white = i; }
		}
	}

	printext256(4+(x<<3),4+(y<<3), white, -1, st, 0);
}

static void _internal_drawosdcursor(int x, int y, int type, int lastkeypress)
{
	int i,j,k;
	char st[2] = { '_',0 };

	if (type) st[0] = '#';

	if (white<0) {
		// find the palette index closest to white
		k=0;
		for(i=0;i<256;i++)
		{
			j = ((int)palette[i*3])+((int)palette[i*3+1])+((int)palette[i*3+2]);
			if (j > k) { k = j; white = i; }
		}
	}

	printext256(4+(x<<3),4+(y<<3)+2, white, -1, st, 0);
}

static int _internal_getcolumnwidth(int w)
{
	return w/8 - 1;
}

static int _internal_getrowheight(int w)
{
	return w/8;
}

static void _internal_clearbackground(int cols, int rows)
{
}

static int _internal_gettime(void)
{
	return 0;
}

static void _internal_onshowosd(int a)
{
}

////////////////////////////

static int _internal_validate_string(void *a)
{
	return 0;	// no problems
}

static int _internal_validate_integer(void *a)
{
	return 0;	// no problems
}

static int _validate_osdlines(void *a)
{
	int *v = (int *)a;

	if (*v < 1) {
		*v = 1;
		return 1;
	}
	if (*v > osdmaxrows) {
		*v = osdmaxrows;
		return 1;
	}

	return 0;	
}

static int _internal_osdfunc_listsymbols(const osdfuncparm_t *parm)
{
	symbol_t *i;

	OSD_Printf("Symbol listing\n  Functions:\n");
	for (i=symbols_func; i!=NULL; i=i->nexttype)
		OSD_Printf("     %s\n", i->name);
	
	OSD_Printf("\n  Variables:\n");
	for (i=symbols_var; i!=NULL; i=i->nexttype)
		OSD_Printf("     %s\n", i->name);

	return OSDCMD_OK;
}

static int _internal_osdfunc_help(const osdfuncparm_t *parm)
{
	symbol_t *symb;

	symb = findexactsymbol(parm->parms[0]);
	if (!symb) {
		OSD_Printf("Help Error: \"%s\" is not a defined variable or function\n", parm->parms[0]);
	} else {
		if (symb->type == SYMBTYPE_FUNC) {
			OSD_Printf("%s\n", symb->i.func.help);
		} else {
			OSD_Printf("\"%s\" is a", symb->name);
			switch (symb->i.var.type) {
				case OSDVAR_INTEGER: OSD_Printf("n integer"); break;
				case OSDVAR_STRING:  OSD_Printf(" string"); break;
			}
			OSD_Printf(" variable\n");
		}
	}
	
	return OSDCMD_OK;
}

////////////////////////////


//
// OSD_Cleanup() -- Cleans up the on-screen display
//
void OSD_Cleanup(void)
{
	symbol_t *s;

	for (; symbols; symbols=s) {
		s=symbols->next;
		free(symbols);
	}

	osdinited=0;
}


//
// OSD_Init() -- Initialises the on-screen display
//
void OSD_Init(void)
{
	memset(osdtext, 32, TEXTSIZE);
	osdlines=0;

	OSD_RegisterFunction("listsymbols",0,"listsymbols: lists all the recognized symbols",_internal_osdfunc_listsymbols);
	OSD_RegisterFunction("help",1,"help: displays help on the named symbol",_internal_osdfunc_help);
	OSD_RegisterVariable("osdrows", OSDVAR_INTEGER, &osdrows, 0, _validate_osdlines);

	atexit(OSD_Cleanup);

	osdinited=1;
}


//
// OSD_SetLogFile() -- Sets the text file where printed text should be echoed
//
void OSD_SetLogFile(char *fn)
{
	if (osdlog) fclose(osdlog);
	osdlog = fopen(DEFAULTLOG,"w");
}


//
// OSD_SetFunctions() -- Sets some callbacks which the OSD uses to understand its world
//
void OSD_SetFunctions(
		void (*drawchar)(int,int,char,int,int),
		void (*drawstr)(int,int,char*,int,int,int),
		void (*drawcursor)(int,int,int,int),
		int (*colwidth)(int),
		int (*rowheight)(int),
		void (*clearbg)(int,int),
		int (*gtime)(void),
		void (*showosd)(int)
	)
{
	drawosdchar = drawchar;
	drawosdstr = drawstr;
	drawosdcursor = drawcursor;
	getcolumnwidth = colwidth;
	getrowheight = rowheight;
	clearbackground = clearbg;
	gettime = gtime;
	onshowosd = showosd;

	if (!drawosdchar) drawosdchar = _internal_drawosdchar;
	if (!drawosdstr) drawosdstr = _internal_drawosdstr;
	if (!drawosdcursor) drawosdcursor = _internal_drawosdcursor;
	if (!getcolumnwidth) getcolumnwidth = _internal_getcolumnwidth;
	if (!getrowheight) getrowheight = _internal_getrowheight;
	if (!clearbackground) clearbackground = _internal_clearbackground;
	if (!gettime) gettime = _internal_gettime;
	if (!onshowosd) onshowosd = _internal_onshowosd;
}


//
// OSD_SetParameters() -- Sets the parameters for presenting the text
//
void OSD_SetParameters(
		int promptshade, int promptpal,
		int editshade, int editpal,
		int textshade, int textpal
	)
{
	osdpromptshade = promptshade;
	osdpromptpal   = promptpal;
	osdeditshade   = editshade;
	osdeditpal     = editpal;
	osdtextshade   = textshade;
	osdtextpal     = textpal;
}


//
// OSD_CaptureKey() -- Sets the scancode for the key which activates the onscreen display
//
void OSD_CaptureKey(int sc)
{
	osdkey = sc;
}


//
// OSD_HandleKey() -- Handles keyboard input when capturing input.
// 	Returns 0 if the key was handled internally, or the scancode if it should
// 	be passed on to the game.
//
int OSD_HandleKey(int sc, int press)
{
	char ch;
	int i;
	
	if (!osdinited) return sc;

	if (sc == osdkey) {
		if (press) OSD_ShowDisplay(osdvisible ^ 1);
		return sc;
	} else if (!osdvisible) {
		return sc;
	}

	if (!press) {
		if (sc == 0x2a || sc == 0x36) // shift
			osdeditshift = 0;
		if (sc == 0x1d || sc == 0x9d)	// control
			osdeditcontrol = 0;
		return 0;	// ignore releases
	}

	keytime = gettime();

	if (sc == 1) OSD_ShowDisplay(0);	// escape
	else if (sc == 0xc9) {		// page up
		if (osdhead < osdlines-1)
			osdhead++;
	} else if (sc == 0xd1) {	// page down
		if (osdhead > 0)
			osdhead--;
	} else if (sc == 0xc7) {	// home
		if (osdeditcontrol) {
			osdhead = osdlines-1;
		} else {
			osdeditcursor = 0;
			osdeditwinstart = osdeditcursor;
			osdeditwinend = osdeditwinstart+editlinewidth;
		}
	} else if (sc == 0xcf) {	// end
		if (osdeditcontrol) {
			osdhead = 0;
		} else {
			osdeditcursor = osdeditlen;
			osdeditwinend = osdeditcursor;
			osdeditwinstart = osdeditwinend-editlinewidth;
			if (osdeditwinstart<0) {
				osdeditwinstart=0;
				osdeditwinend = editlinewidth;
			}
		}
	} else if (sc == 0xd2) {	// insert
		osdovertype ^= 1;
	} else if (sc == 0xcb) {	// left
		if (osdeditcursor>0) {
			if (osdeditcontrol) {
				while (osdeditcursor>0) {
					if (osdeditbuf[osdeditcursor-1] != 32) break;
					osdeditcursor--;
				}
				while (osdeditcursor>0) {
					if (osdeditbuf[osdeditcursor-1] == 32) break;
					osdeditcursor--;
				}
			} else osdeditcursor--;
		}
		if (osdeditcursor<osdeditwinstart)
			osdeditwinend-=(osdeditwinstart-osdeditcursor),
			osdeditwinstart-=(osdeditwinstart-osdeditcursor);
	} else if (sc == 0xcd) {	// right
		if (osdeditcursor<osdeditlen) {
			if (osdeditcontrol) {
				while (osdeditcursor<osdeditlen) {
					if (osdeditbuf[osdeditcursor] == 32) break;
					osdeditcursor++;
				}
				while (osdeditcursor<osdeditlen) {
					if (osdeditbuf[osdeditcursor] != 32) break;
					osdeditcursor++;
				}
			} else osdeditcursor++;
		}
		if (osdeditcursor>=osdeditwinend)
			osdeditwinstart+=(osdeditcursor-osdeditwinend),
			osdeditwinend+=(osdeditcursor-osdeditwinend);
	} else if (sc == 0xc8) {	// up
		if (osdhistorypos < osdhistorysize-1) {
			osdhistorypos++;
			memcpy(osdeditbuf, osdhistorybuf[osdhistorypos], EDITLENGTH+1);
			osdeditlen = osdeditcursor = 0;
			while (osdeditbuf[osdeditcursor]) osdeditlen++, osdeditcursor++;
			if (osdeditcursor<osdeditwinstart) {
				osdeditwinend = osdeditcursor;
				osdeditwinstart = osdeditwinend-editlinewidth;
				
				if (osdeditwinstart<0)
					osdeditwinend-=osdeditwinstart,
					osdeditwinstart=0;
			} else if (osdeditcursor>=osdeditwinend)
				osdeditwinstart+=(osdeditcursor-osdeditwinend),
				osdeditwinend+=(osdeditcursor-osdeditwinend);
		}
	} else if (sc == 0xd0) {	// down
		if (osdhistorypos >= 0) {
			if (osdhistorypos == 0) {
				osdeditlen=0;
				osdeditcursor=0;
				osdeditwinstart=0;
				osdeditwinend=editlinewidth;
				osdhistorypos = -1;
			} else {
				osdhistorypos--;
				memcpy(osdeditbuf, osdhistorybuf[osdhistorypos], EDITLENGTH+1);
				osdeditlen = osdeditcursor = 0;
				while (osdeditbuf[osdeditcursor]) osdeditlen++, osdeditcursor++;
				if (osdeditcursor<osdeditwinstart) {
					osdeditwinend = osdeditcursor;
					osdeditwinstart = osdeditwinend-editlinewidth;
					
					if (osdeditwinstart<0)
						osdeditwinend-=osdeditwinstart,
						osdeditwinstart=0;
				} else if (osdeditcursor>=osdeditwinend)
					osdeditwinstart+=(osdeditcursor-osdeditwinend),
					osdeditwinend+=(osdeditcursor-osdeditwinend);
			}
		}
	} else if (sc == 0x2a || sc == 0x36) {	// shift
		osdeditshift = 1;
	} else if (sc == 0x1d || sc == 0x9d) {	// control
		osdeditcontrol = 1;
	} else if (sc == 0x3a) {	// capslock
		osdeditcaps ^= 1;
	} else if (sc == 0x1c || sc == 0x9c) {	// enter
		if (osdeditlen>0) {
			osdeditbuf[osdeditlen] = 0;
			memmove(osdhistorybuf[1], osdhistorybuf[0], HISTORYDEPTH*(EDITLENGTH+1));
			memmove(osdhistorybuf[0], osdeditbuf, EDITLENGTH+1);
			if (osdhistorysize < HISTORYDEPTH) osdhistorysize++;
			if (osdexeccount == HISTORYDEPTH)
				printOSD("Command Buffer Warning: Failed queueing command for execution. Buffer full.\n");
			else
				osdexeccount++;
			osdhistorypos=-1;
		}

		osdeditlen=0;
		osdeditcursor=0;
		osdeditwinstart=0;
		osdeditwinend=editlinewidth;
	} else if (sc == 0xe) {		// backspace
		if (!osdeditcursor || !osdeditlen) return 0;
		if (!osdovertype) {
			if (osdeditcursor < osdeditlen)
				memmove(osdeditbuf+osdeditcursor-1, osdeditbuf+osdeditcursor, osdeditlen-osdeditcursor);
			osdeditlen--;
		}
		osdeditcursor--;
		if (osdeditcursor<osdeditwinstart) osdeditwinstart--,osdeditwinend--;
	} else if (sc == 0xd3) {	// delete
		if (osdeditcursor == osdeditlen || !osdeditlen) return 0;
		if (osdeditcursor <= osdeditlen-1) memmove(osdeditbuf+osdeditcursor, osdeditbuf+osdeditcursor+1, osdeditlen-osdeditcursor-1);
		osdeditlen--;
	} else {
		ch = sctoasc[osdeditshift^osdeditcaps][sc];
		if (!ch) return 0;

		if (osdeditcontrol) {
			if (ch == 'u' || ch == 'U') {
				if (osdeditcursor>0 && osdeditlen) {
					if (osdeditcursor<osdeditlen)
						memmove(osdeditbuf, osdeditbuf+osdeditcursor, osdeditlen-osdeditcursor);
					osdeditlen-=osdeditcursor;
					osdeditcursor = 0;
					osdeditwinstart = 0;
					osdeditwinend = editlinewidth;
				}
			} else if (ch == 'w' || ch == 'W') {
				if (osdeditcursor>0 && osdeditlen>0) {
					i=osdeditcursor;
					while (i>0 && osdeditbuf[i-1]==32) i--;
					while (i>0 && osdeditbuf[i-1]!=32) i--;
					if (osdeditcursor<osdeditlen)
						memmove(osdeditbuf+i, osdeditbuf+osdeditcursor, osdeditlen-osdeditcursor);
					osdeditlen -= (osdeditcursor-i);
					osdeditcursor = i;
					if (osdeditcursor < osdeditwinstart) {
						osdeditwinstart=osdeditcursor;
						osdeditwinend=osdeditwinstart+editlinewidth;
					}
				}
			}
			return 0;
		}

		if (!osdovertype && osdeditlen == EDITLENGTH)	// buffer full, can't insert another char
			return 0;

		if (!osdovertype) {
			if (osdeditcursor < osdeditlen) 
				memmove(osdeditbuf+osdeditcursor+1, osdeditbuf+osdeditcursor, osdeditlen-osdeditcursor);
			osdeditlen++;
		}
		osdeditbuf[osdeditcursor] = ch;
		osdeditcursor++;
		if (osdeditcursor>osdeditwinend) osdeditwinstart++,osdeditwinend++;
	}
	
	return 0;
}


//
// OSD_ResizeDisplay() -- Handles readjustment of the display when the screen resolution
// 	changes on us.
//
void OSD_ResizeDisplay(int w, int h)
{
	int newcols;
	int newmaxlines;
	char newtext[TEXTSIZE];
	int i,j,k;

	newcols = getcolumnwidth(w);
	newmaxlines = TEXTSIZE / newcols;

	j = min(newmaxlines, osdmaxlines);
	k = min(newcols, osdcols);

	memset(newtext, 32, TEXTSIZE);
	for (i=0;i<j;i++) {
		memcpy(newtext+newcols*i, osdtext+osdcols*i, k);
	}

	memcpy(osdtext, newtext, TEXTSIZE);
	osdcols = newcols;
	osdmaxlines = newmaxlines;
	osdmaxrows = getrowheight(h)-2;
	
	if (osdrows > osdmaxrows) osdrows = osdmaxrows;
	
	osdpos = 0;
	osdhead = 0;
	osdeditwinstart = 0;
	osdeditwinend = editlinewidth;
}


//
// OSD_ShowDisplay() -- Shows or hides the onscreen display
//
void OSD_ShowDisplay(int onf)
{
	osdvisible = (onf != 0);
	osdeditcontrol = 0;
	osdeditshift = 0;

	onshowosd(osdvisible);
}


//
// OSD_Draw() -- Draw the onscreen display
//
void OSD_Draw(void)
{
	unsigned topoffs;
	int row, lines, x, len;
	
	if (!osdvisible || !osdinited) return;

	topoffs = osdhead * osdcols;
	row = osdrows-1;
	lines = min( osdlines-osdhead, osdrows );
	
	begindrawing();

	clearbackground(osdcols,osdrows+1);

	for (; lines>0; lines--, row--) {
		drawosdstr(0,row,osdtext+topoffs,osdcols,osdtextshade,osdtextpal);
		topoffs+=osdcols;
	}

	drawosdchar(2,osdrows,'>',osdpromptshade,osdpromptpal);
	if (osdeditcaps) drawosdchar(0,osdrows,'C',osdpromptshade,osdpromptpal);
	if (osdeditshift) drawosdchar(1,osdrows,'H',osdpromptshade,osdpromptpal);
	
	len = min(osdcols-1-3, osdeditlen-osdeditwinstart);
	for (x=0; x<len; x++)
		drawosdchar(3+x,osdrows,osdeditbuf[osdeditwinstart+x],osdeditshade,osdeditpal);
	
	drawosdcursor(3+osdeditcursor-osdeditwinstart,osdrows,osdovertype,keytime);

	enddrawing();
}


//
// OSD_Printf() -- Print a string to the onscreen display
//   and write it to the log file
//

static inline void linefeed(void)
{
	memmove(osdtext+osdcols, osdtext, TEXTSIZE-osdcols);
	memset(osdtext, 32, osdcols);

	if (osdlines < osdmaxlines) osdlines++;
}

void OSD_Printf(const char *fmt, ...)
{
	char tmpstr[1024], *chp;
	va_list va;
		
	if (!osdlog) osdlog = fopen(DEFAULTLOG, "w");

	if (!osdinited) OSD_Init();

	va_start(va, fmt);
	vsnprintf(tmpstr, 1024, fmt, va);
	va_end(va);

	if (osdlog) fputs(tmpstr, osdlog);

	for (chp = tmpstr; *chp; chp++) {
		if (*chp == '\r') osdpos=0;
		else if (*chp == '\n') {
			osdpos=0;
			linefeed();
		} else {
			osdtext[osdpos++] = *chp;
			if (osdpos == osdcols) {
				osdpos = 0;
				linefeed();
			}
		}
	}
}


//
// OSD_DispatchQueued() -- Executes any commands queued in the buffer
//
void OSD_DispatchQueued(void)
{
	int cmd;
	
	if (!osdexeccount) return;

	cmd=osdexeccount-1;
	osdexeccount=0;

	for (; cmd>=0; cmd--) {
		OSD_Dispatch(osdhistorybuf[cmd]);
	}
}


//
// OSD_Dispatch() -- Executes a command string
//

static char *strtoken(char *s, char **ptrptr)
{
	char *p, *p2, *start;

	if (!ptrptr) return NULL;
	
	if (s) p = s;
	else p = *ptrptr;

	if (!p) return NULL;

	while (*p != 0 && *p != ';' && *p == ' ') p++;
	if (*p == 0 || *p == ';') {
		*ptrptr = NULL;
		return NULL;
	}
	if (*p == '\"') {
		// quoted string
		start = ++p;
		p2 = p;
		while (*p != 0 && *p != ';') {
			if (*p == '\"') {
				p++;
				break;
			} else if (*p == '\\') {
				switch (*(++p)) {
					case 'n': *p2 = '\n'; break;
					case 'r': *p2 = '\r'; break;
					default: *p2 = *p; break;
				}
			} else {
				*p2 = *p;
			}
			p2++, p++;
		}
		*p2 = 0;
	} else {
		start = p;
		while (*p != 0 && *p != ';' && *p != ' ') p++;
	}
	
	if (*p == 0 || *p == ';') *ptrptr = NULL;
	else {
		*(p++) = 0;
		*ptrptr = p;
	}

	return start;
}


#define MAXPARMS 512
int OSD_Dispatch(const char *cmd)
{
	char *workbuf, *wp, *wtp;
	char *parms[MAXPARMS];
	int  numparms;
	osdfuncparm_t ofp;
	symbol_t *symb;
	//int i;
	
	int intvar;
	char *strvar;

	workbuf = strdup(cmd);
	if (!workbuf) return -1;

	numparms = 0;
	memset(parms, 0, sizeof(parms));
	wp = strtoken(workbuf, &wtp);
	
	symb = findexactsymbol(wp);
	if (!symb) {
		OSD_Printf("Error: \"%s\" is not a defined variable or function\n", wp);
		free(workbuf);
		return -1;
	}

	while (wtp) {
		wp = strtoken(NULL, &wtp);
		if (wp && numparms < MAXPARMS) parms[numparms++] = wp;
	}

	//OSD_Printf("Symbol: %s\nParameters:\n",symb->name);
	//for (i=0;i<numparms;i++) OSD_Printf("Parm %d: %s\n",i,parms[i]);

	if (symb->type == SYMBTYPE_FUNC) {
		if (numparms < symb->i.func.minparms) {
			OSD_Printf("%s\n", symb->i.func.help);
		} else {
			ofp.numparms = numparms;
			ofp.parms    = parms;
			ofp.raw      = cmd;
			switch (symb->i.func.func(&ofp)) {
				case OSDCMD_OK: break;
				case OSDCMD_SHOWHELP: OSD_Printf("%s\n", symb->i.func.help); break;
			}
		}
	} else if (symb->type == SYMBTYPE_VAR) {
		if (numparms < 1) {
			// display variable value
			switch (symb->i.var.type) {
				case OSDVAR_STRING:
					OSD_Printf("%s[%d] = \"%s\"\n", symb->name, symb->i.var.extra, symb->i.var.var);
					break;
				
				case OSDVAR_INTEGER:
					if (symb->i.var.extra)
						OSD_Printf("%s = %d\n", symb->name, *((signed int*)symb->i.var.var));
					else
						OSD_Printf("%s = %u\n", symb->name, *((unsigned int*)symb->i.var.var));
					break;
			}
		} else {
			switch (symb->i.var.type) {
				case OSDVAR_STRING:
					strvar = (char *)alloca(symb->i.var.extra);
					strncpy(strvar, parms[0], symb->i.var.extra-1);
					strvar[symb->i.var.extra-1] = 0;

					if (symb->i.var.validator(strvar) >= 0)
						memcpy(symb->i.var.var, strvar, symb->i.var.extra);
					break;
				
				case OSDVAR_INTEGER:
					if (symb->i.var.extra)
						intvar = strtol(parms[0], &strvar, 0);
					else
						intvar = strtoul(parms[0], &strvar, 0);
					if (!strvar[0])
						if (symb->i.var.validator(&intvar) >= 0)
							memcpy(symb->i.var.var, &intvar, 4);
					break;
			}
		}
	}
	
	free(workbuf);
	
	return 0;
}


//
// OSD_RegisterFunction() -- Registers a new function
//
int OSD_RegisterFunction(const char *name, int minparm, const char *help, int (*func)(const osdfuncparm_t*))
{
	symbol_t *symb;
	const char *cp;

	if (!name) {
		printf("OSD_RegisterFunction(): may not register a function with a null name\n");
		return -1;
	}
	if (!name[0]) {
		printf("OSD_RegisterFunction(): may not register a function with no name\n");
		return -1;
	}

	// check for illegal characters in name
	for (cp = name; *cp; cp++) {
		if ((cp == name) && (*cp >= '0') && (*cp <= '9')) {
			printf("OSD_RegisterFunction(): first character of function name \"%s\" must not be a numeral\n", name);
			return -1;
		}
		if ((*cp < '0') ||
		    (*cp > '9' && *cp < 'A') ||
		    (*cp > 'Z' && *cp < 'a' && *cp != '_') ||
		    (*cp > 'z')) {
			printf("OSD_RegisterFunction(): illegal character in function name \"%s\"\n", name);
			return -1;
		}
	}

	if (minparm < 0) minparm = 0;
	if (!help) help = "(no description for this function)";
	if (!func) {
		printf("OSD_RegisterFunction(): may not register a null function\n");
		return -1;
	}

	symb = findexactsymbol(name);
	if (symb) {
		printf("OSD_RegisterFunction(): \"%s\" is already defined as a ");
		switch (symb->type) {
			case SYMBTYPE_FUNC: printf("function"); break;
			case SYMBTYPE_VAR:  printf("variable"); break;
		}
		printf("\n");
		return -1;
	}
	
	symb = addnewsymbol(SYMBTYPE_FUNC);
	if (!symb) {
		printf("OSD_RegisterFunction(): Failed registering function \"%s\"\n", name);
		return -1;
	}

	symb->name = name;
	symb->type = SYMBTYPE_FUNC;
	symb->i.func.minparms = minparm;
	symb->i.func.help = help;
	symb->i.func.func = func;

	return 0;
}


//
// OSD_RegisterVariable() -- Registers a new variable
//
int OSD_RegisterVariable(const char *name, int type, void *var, int extra, int (*validator)(void*))
{
	symbol_t *symb;
	const char *cp;

	if (!name) {
		printf("OSD_RegisterFunction(): may not register a variable with a null name\n");
		return -1;
	}
	if (!name[0]) {
		printf("OSD_RegisterFunction(): may not register a variable with no name\n");
		return -1;
	}

	// check for illegal characters in name
	for (cp = name; *cp; cp++) {
		if ((cp == name) && (*cp >= '0') && (*cp <= '9')) {
			printf("OSD_RegisterFunction(): first character of variable name \"%s\" must not be a numeral\n", name);
			return -1;
		}
		if ((*cp < '0') ||
		    (*cp > '9' && *cp < 'A') ||
		    (*cp > 'Z' && *cp < 'a' && *cp != '_') ||
		    (*cp > 'z')) {
			printf("OSD_RegisterFunction(): illegal character in variable name \"%s\"\n", name);
			return -1;
		}
	}

	if (type != OSDVAR_INTEGER && type != OSDVAR_STRING) {
		printf("OSD_RegisterFunction(): unrecognised variable type for \"%s\"\n", name);
		return -1;
	}
	if (!var) {
		printf("OSD_RegisterFunction(): may not register a null variable\n");
		return -1;
	}
	if (!validator) {
		switch (type) {
			case OSDVAR_INTEGER: validator = _internal_validate_integer; break;
			case OSDVAR_STRING:  validator = _internal_validate_string; break;
		}
	}
	
	symb = findexactsymbol(name);
	if (symb) {
		printf("OSD_RegisterFunction(): \"%s\" is already defined as a ");
		switch (symb->type) {
			case SYMBTYPE_FUNC: printf("function"); break;
			case SYMBTYPE_VAR:  printf("variable"); break;
		}
		printf("\n");
		return -1;
	}
	
	symb = addnewsymbol(SYMBTYPE_VAR);
	if (!symb) {
		printf("OSD_RegisterFunction(): Failed registering variable \"%s\"\n", name);
		return -1;
	}

	symb->name = name;
	symb->type = SYMBTYPE_VAR;
	symb->i.var.type = type;
	symb->i.var.var  = var;
	symb->i.var.extra = extra;
	symb->i.var.validator = validator;

	return 0;
}


//
// addnewsymbol() -- Allocates space for a new symbol and attaches it
//   appropriately to the lists.
//
static symbol_t *addnewsymbol(int type)
{
	symbol_t *newsymb, *s;

	if (type != SYMBTYPE_FUNC && type != SYMBTYPE_VAR) { return NULL; }

	newsymb = (symbol_t *)malloc(sizeof(symbol_t));
	if (!newsymb) { return NULL; }

	memset(newsymb, 0, sizeof(symbol_t));
	newsymb->type = type;

	// link it to the main chain
	if (!symbols) {
		symbols = newsymb;
	} else {
		s = symbols;
		while (s->next) s=s->next;
		s->next = newsymb;
	}

	// link it to the appropriate type chain
	if (type == SYMBTYPE_FUNC) {
		if (!symbols_func) {
			symbols_func = newsymb;
		} else {
			s = symbols_func;
			while (s->nexttype) s=s->nexttype;
			s->nexttype = newsymb;
		}
	} else {
		if (!symbols_var) {
			symbols_var = newsymb;
		} else {
			s = symbols_var;
			while (s->nexttype) s=s->nexttype;
			s->nexttype = newsymb;
		}
	}

	return newsymb;
}


//
// findsymbol() -- Finds a symbol, possibly partially named
// 
static symbol_t *findsymbol(const char *name, symbol_t *startingat)
{
	if (!startingat) startingat = symbols;
	if (!startingat) return NULL;

	for (; startingat; startingat=startingat->next)
		if (!strncasecmp(name, startingat->name, strlen(name))) return startingat;

	return NULL;
}


//
// findexactsymbol() -- Finds a symbol, complete named
// 
static symbol_t *findexactsymbol(const char *name)
{
	symbol_t *startingat;
	if (!symbols) return NULL;

	startingat = symbols;

	for (; startingat; startingat=startingat->next)
		if (!strcasecmp(name, startingat->name)) return startingat;

	return NULL;
}

