// Windows DIB/DirectDraw interface layer
// for the Build Engine
// by Jonathon Fowler (jonof@edgenetwk.com)
//
// Written for DirectX 6.0


#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <ddraw.h>
#include <dinput.h>

#include "dxdidf.h"
#include <mmsystem.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>

#include "winlayer.h"
#include "pragmas.h"
#include "build.h"
#include "osd.h"
#include "compat.h"


extern char moustat;

// undefine to restrict windowed resolutions to conventional sizes
#define ANY_WINDOWED_SIZE


// Windows crud
static HINSTANCE hInstance = 0;
static HWND hWindow = 0;
#define WINDOW_STYLE (WS_POPUP|WS_CAPTION|WS_SYSMENU|WS_MINIMIZEBOX)
extern long app_main(long argc, char *argv[]);
static BOOL window_class_registered = FALSE;

// DirectInput objects
static HMODULE              hDInputDLL    = NULL;
static LPDIRECTINPUTA       lpDI          = NULL;
static LPDIRECTINPUTDEVICEA lpDIDKeyboard = NULL;
static LPDIRECTINPUTDEVICEA lpDIDMouse    = NULL;
static BOOL                 bDInputInited = FALSE;
#define INPUT_BUFFER_SIZE	32
//#define USE_BUFFERED_KEYBOARD

// DirectDraw objects
static HMODULE              hDDrawDLL      = NULL;
static LPDIRECTDRAW         lpDD           = NULL;
static LPDIRECTDRAWSURFACE  lpDDSPrimary   = NULL;
static LPDIRECTDRAWSURFACE  lpDDSBack      = NULL;
static LPDIRECTDRAWSURFACE  lpDDSOffscreen = NULL;
static LPDIRECTDRAWPALETTE  lpDDPalette = NULL;
static BOOL                 bDDrawInited = FALSE;

// DIB stuff
static HDC      hDC         = NULL;
static HDC      hDCSection  = NULL;
static HBITMAP  hDIBSection = NULL;
static HPALETTE hPalette    = NULL;

#define NUM_SYS_COLOURS	25
static int syscolouridx[NUM_SYS_COLOURS] = {
	COLOR_SCROLLBAR,		// 1
	COLOR_BACKGROUND,
	COLOR_ACTIVECAPTION,
	COLOR_INACTIVECAPTION,
	COLOR_MENU,
	COLOR_WINDOW,
	COLOR_WINDOWFRAME,
	COLOR_MENUTEXT,
	COLOR_WINDOWTEXT,
	COLOR_CAPTIONTEXT,		// 10
	COLOR_ACTIVEBORDER,
	COLOR_INACTIVEBORDER,
	COLOR_APPWORKSPACE,
	COLOR_HIGHLIGHT,
	COLOR_HIGHLIGHTTEXT,
	COLOR_BTNFACE,
	COLOR_BTNSHADOW,
	COLOR_GRAYTEXT,
	COLOR_BTNTEXT,
	COLOR_INACTIVECAPTIONTEXT,	// 20
	COLOR_BTNHIGHLIGHT,
	COLOR_3DDKSHADOW,
	COLOR_3DLIGHT,
	COLOR_INFOTEXT,
	COLOR_INFOBK			// 25
};
static DWORD syscolours[NUM_SYS_COLOURS];
static char system_colours_saved = 0;

static const LPTSTR GetWindowsErrorMsg(DWORD code);
static const char * GetDDrawError(HRESULT code);
static const char * GetDInputError(HRESULT code);
static void ShowErrorBox(const char *m);
static void ShowDDrawErrorBox(const char *m, HRESULT r);
static void ShowDInputErrorBox(const char *m, HRESULT r);
static BOOL CheckWinVersion(void);
static LRESULT CALLBACK WndProcCallback(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam);
static BOOL InitDirectDraw(void);
static void UninitDirectDraw(void);
static void ReleaseSurfaces(void);
static BOOL InitDirectInput(void);
static void UninitDirectInput(void);
static void AcquireInputDevices(char acquire);
static void ProcessInputDevices(void);
static void UninitDIB(void);
static BOOL RegisterWindowClass(void);
static BOOL CreateAppWindow(int width, int height, int fs);
static void DestroyAppWindow(void);
static void SaveSystemColours(void);
static void RestoreSystemColours(void);


// timer
static int64 timerfreq=0;
static long timerlastsample=0;
static int timerticspersec=0;
static void (*usertimercallback)(void) = NULL;


// video
static long desktopxdim, desktopydim, desktoppal;
long xres=-1, yres=-1, fullscreen=0, bytesperline=0, imageSize=0;
long frameplace=0, lockcount=0;
static PALETTEENTRY curPalette[256];
static int windowposx, windowposy;
static char modechange=1;


// input and events
char quitevent=0, pauseevent=0, appactive=1, appminimized=0;
short mousex=0, mousey=0, mouseb=0, mousegrab=1;

char keystatus[256], keyfifo[KEYFIFOSIZ], keyfifoplc, keyfifoend;
static BYTE keys[2*256], keysofs=0;
static unsigned long lastKeyDown = 0;
static unsigned long lastKeyTime = 0;

#define SetKey(key,state) { \
	keystatus[key] = state; \
	keyfifo[keyfifoend] = key; \
	keyfifo[(keyfifoend+1)&(KEYFIFOSIZ-1)] = state; \
	keyfifoend = ((keyfifoend+2)&(KEYFIFOSIZ-1)); \
}





//-------------------------------------------------------------------------------------------------
//  MAIN CRAP
//=================================================================================================


//
// SignalHandler() -- called when we've sprung a leak
//
static void SignalHandler(int signum)
{
	switch (signum) {
		case SIGSEGV:
			printOSD("Fatal Signal caught: SIGSEGV. Bailing out.\n");
			uninitsystem();
			break;
		default:
			break;
	}
}


//
// WinMain() -- main Windows entry point
//
int WINAPI WinMain(HINSTANCE hInst, HINSTANCE hPrevInst, LPSTR lpCmdLine, int nCmdShow)
{
	int r;
	char *argp;
	FILE *fp;
	int argc = 1;
	char *argv[30] = { "buildapp" };
	HDC hdc;

	hInstance = hInst;

	if (CheckWinVersion() || hPrevInst) {
		MessageBox(0, "This application must be run under Windows 95/98/Me or Windows 2000/XP or better.",
			apptitle, MB_OK|MB_ICONSTOP);
		return -1;
	}

	hdc = GetDC(NULL);
	r = GetDeviceCaps(hdc, BITSPIXEL);
	ReleaseDC(NULL, hdc);
	if (r < 8) {
		MessageBox(0, "This application requires a desktop colour depth of 256-colours or more.",
			apptitle, MB_OK|MB_ICONSTOP);
		return -1;
	}

	// carve up the commandline into more recognizable pieces
	argp = lpCmdLine;
	while (*argp && (argc < 30)) {
		while (((*argp <= 32) || (*argp > 126)) && (*argp != 0)) argp++;
		if (*argp == 0) break;

		argv[argc++] = argp;

		while ((*argp > 32) && (*argp <= 126)) argp++;
		if (*argp == 0) break;

		*(argp++) = 0;
	}

	// pipe standard outputs to files
	fp = freopen("stdout.txt", "w", stdout);
	if (!fp) {
		fp = fopen("stdout.txt", "w");
	}
	if (fp) setvbuf(fp, 0, _IONBF, 0);
	*stdout = *fp;

	// install signal handlers
	signal(SIGSEGV, SignalHandler);

	if (RegisterWindowClass()) return -1;

	MessageBox(0,
		"IMPORTANT:\n"
		"\tThis is a source port by Jonathon Fowler (jonof@edgenetwk.com) of the Build Engine, "
		"editor and test game by Ken Silverman to the Windows and Linux operating systems. It is "
		"distributed under the terms listed in BUILDLIC.TXT included with this package.",
		"Build Engine",
		MB_OK|MB_ICONINFORMATION);

	r = app_main(argc, argv);

	fclose(stdout);

	return r;
}


//
// initsystem() -- init systems
//
int initsystem(void)
{
	HDC hdc;
	int i;

	printOSD("Initializing Windows DirectX/GDI system interface\n");

	hdc = GetDC(NULL);

	// get the desktop dimensions before anything changes them
	desktopxdim = GetDeviceCaps(hdc, HORZRES);
	desktopydim = GetDeviceCaps(hdc, VERTRES);
	desktoppal  = ((GetDeviceCaps(hdc, RASTERCAPS) & RC_PALETTE) == RC_PALETTE);

	ReleaseDC(NULL, hdc);

	if (desktoppal)
		// save the system colours
		SaveSystemColours();

	memset(curPalette, 0, sizeof(PALETTEENTRY) * 256);

	atexit(uninitsystem);

	frameplace=0;
	lockcount=0;

	// try and start DirectDraw
	if (InitDirectDraw())
		printOSD("DirectDraw initialization failed. Fullscreen modes will be unavailable.\n");

	return 0;
}


//
// uninitsystem() -- uninit systems
//
void uninitsystem(void)
{
	DestroyAppWindow();

	uninitinput();
	uninittimer();
}


//
// handleevents() -- process the Windows message queue
//   returns !0 if there was an important event worth checking (like quitting)
//
int handleevents(void)
{
	int rv=0;
	MSG msg;

	//if (appminimized) WaitMessage();

	if (frameplace && fullscreen) printf("Offscreen buffer is locked!\n");

	while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE)) {
		if (msg.message == WM_QUIT)
			quitevent = 1;
		TranslateMessage(&msg);
		DispatchMessage(&msg);
	}

	ProcessInputDevices();

	if (!appactive || quitevent) rv = -1;

	sampletimer();

	return rv;
}







//-------------------------------------------------------------------------------------------------
//  INPUT (MOUSE/KEYBOARD/JOYSTICK?)
//=================================================================================================

//
// initinput() -- init input system
//
int initinput(void)
{
	moustat=0;
	memset(keystatus, 0, sizeof(keystatus));
	memset(keys, 0, sizeof(keys));
	keysofs = 0;

	if (InitDirectInput())
		return -1;

	AcquireInputDevices(1);

	return 0;
}


//
// uninitinput() -- uninit input system
//
void uninitinput(void)
{
	uninitmouse();
	UninitDirectInput();
}


//
// initmouse() -- init mouse input
//
int initmouse(void)
{
	if (moustat) return 0;

	printOSD("Initializing mouse\n");

	// grab input
	moustat=1;
	grabmouse(1);

	return 0;
}


//
// uninitmouse() -- uninit mouse input
//
void uninitmouse(void)
{
	if (!moustat) return;

	grabmouse(0);
	moustat=0;
}


//
// grabmouse() -- show/hide mouse cursor
//
void grabmouse(char a)
{
	if (!moustat) return;

	mousegrab = a;

	AcquireInputDevices(a);

	mousex = 0;
	mousey = 0;
	mouseb = 0;
}


//
// readmousexy() -- return mouse motion information
//
void readmousexy(short *x, short *y)
{
	*x = mousex;
	*y = mousey;
	mousex = 0;
	mousey = 0;
}


//
// readmousebstatus() -- return mouse button information
//
void readmousebstatus(short *b)
{
	*b = mouseb;
}







//-------------------------------------------------------------------------------------------------
//  TIMER
//=================================================================================================

#if 0
//
// timercallback() -- multimedia timer callback
//
static void CALLBACK timercallback(UINT uID, UINT uMsg, DWORD dwUser, DWORD dw1, DWORD dw2)
{
	totalclock++;
	//ProcessInputDevices();
	if (usertimercallback) usertimercallback();
	//handleevents();
}


//
// inittimer() -- install timer callback
//
int inittimer(int tickspersecond)
{
	TIMECAPS tc;

	if (timerid) return 0;	// already installed

	printOSD("Initializing timer\n");

	if (timeGetDevCaps(&tc, sizeof(tc)) != TIMERR_NOERROR) {
		ShowErrorBox("Failed calling timeGetDevCaps()");
		return -1;
	}

	timerresolution = 1000 / tickspersecond;
	if (timerresolution < tc.wPeriodMin) {
		timerresolution = tc.wPeriodMin;
		printOSD("Warning: %d tickspersecond is too fast for this system. Falling back to minimum of %d\n", tickspersecond, 1000/timerresolution);
	}

	if (timeBeginPeriod(timerresolution) != TIMERR_NOERROR) {
		ShowErrorBox("Failed calling timeBeginPeriod()");
		return -1;
	}

	usertimercallback = NULL;

	halttimer(0);

	return 0;
}

//
// uninittimer() -- remove timer callback
//
void uninittimer(void)
{
	if (!timerid) return;

	halttimer(1);

	timeEndPeriod(timerresolution);

	timerid=0;
}

//
// halttimer() -- freezes and thaws time
//
int halttimer(char halt)
{
	if (halt && !timerid) return 0;		// already stopped
	else if (!halt && timerid) return 0;	// already running

	if (halt) {
		timeKillEvent(timerid);
		timerid = 0;
	} else {
		timerid = timeSetEvent(timerresolution, timerresolution/2, timercallback, 0, TIME_PERIODIC);
		if (!timerid) {
			printOSD("Timer initialization failed: %s\n", GetWindowsErrorMsg(GetLastError()));
			return -1;	// error
		}
	}

	return 0;
}

//
// getticks() -- returns the windows ticks count
//
long getticks(void)
{
	return (long)timeGetTime();
}

#endif


//  This timer stuff is all Ken's idea.

//
// installusertimercallback() -- set up a callback function to be called when the timer is fired
//
void (*installusertimercallback(void (*callback)(void)))(void)
{
	void (*oldtimercallback)(void);

	oldtimercallback = usertimercallback;
	usertimercallback = callback;

	return oldtimercallback;
}


//
// inittimer() -- initialize timer
//
int inittimer(int tickspersecond)
{
	int64 t;
	
	if (timerfreq) return 0;	// already installed

	printOSD("Initializing timer\n");

	QueryPerformanceFrequency((LARGE_INTEGER*)&timerfreq);
	timerticspersec = tickspersecond;
	QueryPerformanceCounter((LARGE_INTEGER*)&t);
	timerlastsample = (long)(t*timerticspersec / timerfreq);

	usertimercallback = NULL;

	return 0;
}

//
// uninittimer() -- shut down timer
//
void uninittimer(void)
{
	if (!timerfreq) return;

	timerfreq=0;
}

//
// sampletimer() -- update totalclock
//
void sampletimer(void)
{
	int64 i;
	long n;
	
	if (!timerfreq) return;

	QueryPerformanceCounter((LARGE_INTEGER*)&i);
	n = (long)(i*timerticspersec / timerfreq) - timerlastsample;
	if (n>0) {
		totalclock += n;
		timerlastsample += n;
	}

	if (usertimercallback) for (; n>0; n--) usertimercallback();
}


//
// getticks() -- returns the windows ticks count
//
long getticks(void)
{
	int64 i;
	QueryPerformanceCounter((LARGE_INTEGER*)&i);
	return (long)(i*0x7fffffffll/timerfreq);
}




//-------------------------------------------------------------------------------------------------
//  VIDEO
//=================================================================================================

//
// checkvideomode() -- makes sure the video mode passed is legal
//
int checkvideomode(long *x, long *y, int fs)
{
	int i, nearest=-1, dx, dy, odx=9999, ody=9999;

	getvalidmodes();

	// fix up the passed resolution values to be multiples of 8
	// and at least 320x200 or at most MAXXDIMxMAXYDIM
	if (*x < 320) *x = 320;
	if (*y < 200) *y = 200;
	if (*x > MAXXDIM) *x = MAXXDIM;
	if (*y > MAXYDIM) *y = MAXYDIM;
	*x &= 0xfffffff8l;

#ifdef ANY_WINDOWED_SIZE
	if (fs) {
#endif
		for (i=0; i<validmodecnt; i++) {
			if (validmodefs[i] != fs) continue;
			dx = klabs(validmodexdim[i] - *x);
			dy = klabs(validmodeydim[i] - *y);
			if (!(dx | dy)) { 	// perfect match
				nearest = i;
				break;
			}
			if ((dx <= odx) && (dy <= ody)) {
				nearest = i;
				odx = dx; ody = dy;
			}
		}

		if (nearest < 0) {
			// no mode that will match (eg. if no fullscreen modes)
			return -1;
		}
		*x = validmodexdim[nearest];
		*y = validmodeydim[nearest];
#ifdef ANY_WINDOWED_SIZE
	}
#endif

	return 0;
}


//
// setvideomode() -- set the video mode
//
int setvideomode(int x, int y, int fs)
{
	char t[384];
	HRESULT result;
	DDSURFACEDESC ddsd;

	if ((fs == fullscreen) && (x == xres) && (x == yres)) return 0;

	AcquireInputDevices(0);

	printOSD("Setting video mode %dx%d (%s)\n", x,y, ((fs) ? "fullscreen" : "windowed"));
	if (CreateAppWindow(x, y, fs)) return -1;

	sprintf(t, "%s (%dx%d %s)", apptitle, x, y, ((fs) ? "fullscreen" : "windowed"));
	SetWindowText(hWindow, t);

	AcquireInputDevices(1);
	modechange=1;

	return 0;
}


//
// getvalidmodes() -- figure out what video modes are available
//
#define ADDMODE(x,y,f) { \
	validmodexdim[validmodecnt]=x; \
	validmodeydim[validmodecnt]=y; \
	validmodefs[validmodecnt]=f; \
	validmodecnt++; \
	printOSD("Adding mode %dx%d (%s)\n", x, y, (f)?"fullscreen":"windowed"); }

#define CHECK(w,h) if ((w <= maxx) && (h <= maxy))

// mode enumerator
static HRESULT WINAPI getvalidmodes_enum(DDSURFACEDESC *ddsd, VOID *udata)
{
	unsigned maxx = MAXXDIM, maxy = MAXYDIM;

	if (ddsd->ddpfPixelFormat.dwRGBBitCount == 8) {
		CHECK(ddsd->dwWidth, ddsd->dwHeight) ADDMODE(ddsd->dwWidth, ddsd->dwHeight, 1);
	}

	return(DDENUMRET_OK);
}

void getvalidmodes(void)
{
	static int modeschecked=0;
	int i, maxx=0, maxy=0;
	HRESULT result;

	if (modeschecked) return;

	validmodecnt=0;

	if (bDDrawInited) {
		// if DirectDraw initialization didn't fail enumerate fullscreen modes

		result = IDirectDraw_EnumDisplayModes(lpDD, 0, NULL, 0, getvalidmodes_enum);
		if (result != DD_OK) {
			printOSD("Unable to enumerate fullscreen modes. Using default list.\n");
			ADDMODE(1280,1024,1)
			ADDMODE(1280,960,1)
			ADDMODE(1152,864,1)
			ADDMODE(1024,768,1)
			ADDMODE(800,600,1)
			ADDMODE(640,480,1)
			ADDMODE(640,400,1)
			ADDMODE(512,384,1)
			ADDMODE(480,360,1)
			ADDMODE(400,300,1)
			ADDMODE(320,240,1)
			ADDMODE(320,200,1)
		}
	}

	// windowed modes cant be bigger than the current desktop resolution
	maxx = desktopxdim-1;
	maxy = desktopydim-1;

	// add windowed modes next
	CHECK(320,200) ADDMODE(320,200,0)
	CHECK(320,240) ADDMODE(320,240,0)
	CHECK(400,300) ADDMODE(400,300,0)
	CHECK(480,360) ADDMODE(480,360,0)
	CHECK(512,384) ADDMODE(512,384,0)
	CHECK(640,400) ADDMODE(640,400,0)
	CHECK(640,480) ADDMODE(640,480,0)
	CHECK(800,600) ADDMODE(800,600,0)
	CHECK(1024,768) ADDMODE(1024,768,0)
	CHECK(1152,864) ADDMODE(1152,864,0)
	CHECK(1280,960) ADDMODE(1280,960,0)
	CHECK(1280,1024) ADDMODE(1280,1024,0)

	modeschecked=1;
}

#undef CHECK
#undef ADDMODE


//
// begindrawing() -- locks the framebuffer for drawing
//
void begindrawing(void)
{
	HRESULT result;
	DDSURFACEDESC ddsd;
	long i,j;

	if (!fullscreen) return;	// gdi has no locking

	if (frameplace) {
		lockcount++;
		return;		// already locked
	}

	lockcount++;

	memset(&ddsd, 0, sizeof(ddsd));
	ddsd.dwSize = sizeof(ddsd);

	result = IDirectDrawSurface_Lock(lpDDSOffscreen, NULL, &ddsd, DDLOCK_NOSYSLOCK | DDLOCK_WAIT, NULL);
	if (result != DD_OK) {
		printOSD("Failed locking offscreen surface: %s\n", GetDDrawError(result));
		return;
	}

	frameplace = (long)ddsd.lpSurface;

	if (ddsd.lPitch != bytesperline || modechange) {
		bytesperline = ddsd.lPitch;
		imageSize = bytesperline*yres;
		setvlinebpl(bytesperline);

		j = 0;
		for(i=0;i<=ydim;i++) ylookup[i] = j, j += bytesperline;
	}
}


//
// enddrawing() -- unlocks the framebuffer
//
void enddrawing(void)
{
	HRESULT result;

	if (!fullscreen) return;	// gdi does no locking

	if (!frameplace) return;
	if (--lockcount > 0) return;

	result = IDirectDrawSurface_Unlock(lpDDSOffscreen, NULL);
	if (result != DD_OK) {
		printOSD("Failed unlocking offscreen surface: %s\n", GetDDrawError(result));
		return;
	}

	frameplace = 0;
}


//
// showframe() -- update the display
//
void showframe(void)
{
	HRESULT result;
	RECT rect;

	if (!fullscreen) {
		BitBlt(hDC, 0, 0, xres, yres, hDCSection, 0, 0, SRCCOPY);
	} else {
		if (lockcount) {
			printf("Frame still locked %d times when showframe() called.\n", lockcount);
			while (lockcount) enddrawing();
		}

		rect.left = 0;
		rect.top = 0;
		rect.right = xres - 1;
		rect.bottom = yres - 1;

		result = IDirectDrawSurface_BltFast(lpDDSBack, 0, 0, lpDDSOffscreen, &rect, /*DDBLTFAST_WAIT*/0);
		if (result == DDERR_SURFACELOST) {
			IDirectDrawSurface_Restore(lpDDSBack);
			result = IDirectDrawSurface_BltFast(lpDDSBack, 0, 0, lpDDSOffscreen, &rect, /*DDBLTFAST_WAIT*/0);
		}

		if (result == DD_OK) {
			result = IDirectDrawSurface_Flip(lpDDSPrimary, NULL, 0);
			if (result == DDERR_SURFACELOST) {
				IDirectDrawSurface_Restore(lpDDSPrimary);
				result = IDirectDrawSurface_Flip(lpDDSPrimary, NULL, 0);
			}
		}
	}
}


//
// setpalette() -- set palette values
//
int setpalette(int start, int num, char *dapal)
{
	int i, n;
	HRESULT result;
	RGBQUAD *rgb;
	HPALETTE hPalPrev;

	struct logpal {
		WORD palVersion;
		WORD palNumEntries;
		PALETTEENTRY palPalEntry[256];
	} *lpal;

	for (i=start, n=num; n>0; i++, n--) {
		curPalette[i].peBlue = dapal[0] << 2;
		curPalette[i].peGreen = dapal[1] << 2;
		curPalette[i].peRed = dapal[2] << 2;
		curPalette[i].peFlags = PC_RESERVED | PC_NOCOLLAPSE;
		dapal += 4;
	}

	if (!fullscreen) {
		if (num > 0) {
			rgb = (RGBQUAD *)alloca(sizeof(RGBQUAD)*num);
			for (i=start, n=0; n<num; i++, n++) {
				rgb[n].rgbBlue = curPalette[i].peBlue;
				rgb[n].rgbGreen = curPalette[i].peGreen;
				rgb[n].rgbRed = curPalette[i].peRed;
				rgb[n].rgbReserved = 0;
			}

			SetDIBColorTable(hDCSection, start, num, rgb);
		}

		if (!desktoppal) return 0;	// only if an 8bit desktop do we do what follows

		lpal = (struct logpal *)alloca(sizeof(struct logpal));

		memcpy(lpal->palPalEntry, curPalette, sizeof(PALETTEENTRY)*256);

		// set 0 and 255 to black and white
		lpal->palVersion = 0x300;
		lpal->palNumEntries = 256;
		lpal->palPalEntry[0].peBlue = 0;
		lpal->palPalEntry[0].peGreen = 0;
		lpal->palPalEntry[0].peRed = 0;
		lpal->palPalEntry[0].peFlags = 0;
		lpal->palPalEntry[255].peBlue = 255;
		lpal->palPalEntry[255].peGreen = 255;
		lpal->palPalEntry[255].peRed = 255;
		lpal->palPalEntry[255].peFlags = 0;

		if (SetSystemPaletteUse(hDC, SYSPAL_NOSTATIC) == SYSPAL_ERROR) {
			//printOSD("Problem setting system palette use: %s\n", GetWindowsErrorMsg(GetLastError()));
			return -1;
		}

		if (hPalette) {
			if (num == 0) { start = 0; num = 256; }		// refreshing the palette only
			SetPaletteEntries(hPalette, start, num, lpal->palPalEntry+start);
		} else {
			hPalette = CreatePalette((LOGPALETTE *)lpal);
			if (!hPalette) {
				printOSD("Problem creating palette: %s\n", GetWindowsErrorMsg(GetLastError()));
				return -1;
			}
		}

		if (SelectPalette(hDC, hPalette, FALSE) == NULL) {
			printOSD("Problem selecting palette: %s\n", GetWindowsErrorMsg(GetLastError()));
			return -1;
		}

		if (RealizePalette(hDC) == GDI_ERROR) {
			printOSD("Failure realizing palette: %s\n", GetWindowsErrorMsg(GetLastError()));
			return -1;
		}
	} else {
		result = IDirectDrawPalette_SetEntries(lpDDPalette, 0, start, num, &curPalette[start]);
		if (result != DD_OK) {
			printOSD("Palette set failed: %s\n", GetDDrawError(result));
			return -1;
		}
	}

	return 0;
}

//
// getpalette() -- get palette values
//
int getpalette(int start, int num, char *dapal)
{
	int i;

	for (i=num; i>0; i--, start++) {
		dapal[0] = curPalette[start].peBlue >> 2;
		dapal[1] = curPalette[start].peGreen >> 2;
		dapal[2] = curPalette[start].peRed >> 2;
		dapal += 4;
	}

	return 0;
}






//-------------------------------------------------------------------------------------------------
//  MOSTLY STATIC INTERNAL WINDOWS THINGS
//=================================================================================================

//
// ShowErrorBox() -- shows an error message box
//
static void ShowErrorBox(const char *m)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetWindowsErrorMsg(GetLastError()));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// ShowDDrawErrorBox() -- shows an error message box for a DirectDraw error
//
static void ShowDDrawErrorBox(const char *m, HRESULT r)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetDDrawError(r));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// ShowDInputErrorBox() -- shows an error message box for a DirectInput error
//
static void ShowDInputErrorBox(const char *m, HRESULT r)
{
	TCHAR msg[1024];

	wsprintf(msg, "%s: %s", m, GetDInputError(r));
	MessageBox(0, msg, apptitle, MB_OK|MB_ICONSTOP);
}


//
// CheckWinVersion() -- check to see what version of Windows we happen to be running under
//
static BOOL CheckWinVersion(void)
{
	OSVERSIONINFO osv;

	osv.dwOSVersionInfoSize = sizeof(osv);
	if (!GetVersionEx(&osv)) return TRUE;

	// haha, yeah, like it will work on Win32s
	if (osv.dwPlatformId == VER_PLATFORM_WIN32s) return TRUE;

	// we don't like NT 3.51
	if (osv.dwMajorVersion < 4) return TRUE;

	// nor do we like NT 4
	if (osv.dwPlatformId == VER_PLATFORM_WIN32_NT &&
	    osv.dwMajorVersion == 4) return TRUE;

	return FALSE;
}


//
// WndProcCallback() -- the Windows window callback
//
static LRESULT CALLBACK WndProcCallback(HWND hWnd, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	RECT rect;
	POINT pt;
	HRESULT result;

	switch (uMsg) {
		case WM_ACTIVATE:
			appminimized = HIWORD(wParam);
			appactive = (LOWORD(wParam) != WA_INACTIVE);

			if (desktoppal) {
				if (appactive) {
					setpalette(0,0,0);
				} else {
					RestoreSystemColours();
				}
			}

			AcquireInputDevices(appactive && !appminimized);
			break;

		case WM_ENTERMENULOOP:
		case WM_ENTERSIZEMOVE:
			// un-acquire device when entering menu or re-sizing
			// this will show the mouse cursor again
			appactive = 0;
			AcquireInputDevices(0);
			//halttimer(1);
			break;

		case WM_EXITMENULOOP:
		case WM_EXITSIZEMOVE:
			// re-acquire device when leaving menu or re-sizing
			// this will show the mouse cursor again

			// even though the menu is going away, the app
			// might have lost focus or be an icon
			if (GetActiveWindow() == hWnd && !IsIconic(hWnd))
				appactive = 1;
			else
				appactive = 0;

			//halttimer(0);
			AcquireInputDevices(appactive);
			break;

		case WM_PAINT:
			// blit the back-buffer to the screen
			break;

		case WM_MOVE:
			windowposx = LOWORD(lParam);
			windowposy = HIWORD(lParam);
			return 0;

		case WM_CLOSE:
			quitevent = 1;
			return 0;

		case WM_SYSKEYDOWN:
		case WM_SYSKEYUP:
			return 0;

		case WM_SYSCOMMAND:
			if (wParam == SC_SCREENSAVE) return 0;
			break;

		case WM_DESTROY:
			hWindow = 0;
			PostQuitMessage(0);
			return 0;

		default:
			break;
	}

	return DefWindowProc(hWnd, uMsg, wParam, lParam);
}


//
// RegisterWindowClass() -- register the window class
//
static BOOL RegisterWindowClass(void)
{
	WNDCLASSEX wcx;

	if (window_class_registered) return FALSE;

	printOSD("Registering window class\n");

	wcx.cbSize	= sizeof(wcx);
	wcx.style	= 0;
	wcx.lpfnWndProc	= WndProcCallback;
	wcx.cbClsExtra	= 0;
	wcx.cbWndExtra	= 0;
	wcx.hInstance	= hInstance;
	wcx.hIcon	= LoadImage(hInstance, MAKEINTRESOURCE(100), IMAGE_ICON,
				GetSystemMetrics(SM_CXICON), GetSystemMetrics(SM_CYICON), LR_DEFAULTCOLOR);
	wcx.hCursor	= LoadCursor(hInstance, IDC_ARROW);
	wcx.hbrBackground = (HBRUSH)COLOR_GRAYTEXT;
	wcx.lpszMenuName = NULL;
	wcx.lpszClassName = "buildapp";
	wcx.hIconSm	= LoadImage(hInstance, MAKEINTRESOURCE(100), IMAGE_ICON,
				GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON), LR_DEFAULTCOLOR);
	if (!RegisterClassEx(&wcx)) {
		ShowErrorBox("Failed to register window class");
		return TRUE;
	}

	window_class_registered = TRUE;

	return FALSE;
}


//
// InitDirectDraw() -- get DirectDraw started
//
static BOOL InitDirectDraw(void)
{
	HRESULT result;
	HRESULT (WINAPI *aDirectDrawCreate)(GUID *, LPDIRECTDRAW *, IUnknown *);

	if (bDDrawInited) return FALSE;

	printOSD("Initializing DirectDraw...\n");

	// load up the DirectDraw DLL
	if (!hDDrawDLL) {
		printOSD("  - Loading DDRAW.DLL\n");
		hDDrawDLL = LoadLibrary("DDRAW.DLL");
		if (!hDDrawDLL) {
			ShowErrorBox("Error loading DDRAW.DLL");
			return TRUE;
		}
	}

	// get the pointer to DirectDrawCreate
	aDirectDrawCreate = (void *)GetProcAddress(hDDrawDLL, "DirectDrawCreate");
	if (!aDirectDrawCreate) {
		ShowErrorBox("Error fetching DirectDrawCreate()");
		UninitDirectDraw();
		return TRUE;
	}

	// create a new DirectDraw object
	printOSD("  - Creating DirectDraw object\n");
	result = aDirectDrawCreate(NULL, &lpDD, NULL);
	if (result != DD_OK) {
		ShowDDrawErrorBox("DirectDrawCreate() failed", result);
		UninitDirectDraw();
		return TRUE;
	}

	bDDrawInited = TRUE;

	return FALSE;
}


//
// UninitDirectDraw() -- clean up DirectDraw
//
static void UninitDirectDraw(void)
{
	if (bDDrawInited) printOSD("Uninitializing DirectDraw...\n");

	AcquireInputDevices(0);
	ReleaseSurfaces();

	if (lpDD) {
		printOSD("  - Releasing DirectDraw object\n");
		IDirectDraw_Release(lpDD);
		lpDD = NULL;
	}

	if (hDDrawDLL) {
		printOSD("  - Unloading DDRAW.DLL\n");
		FreeLibrary(hDDrawDLL);
		hDDrawDLL = NULL;
	}

	bDDrawInited = FALSE;
}


//
// ReleaseSurfaces() -- release the front and back buffers
//
static void ReleaseSurfaces(void)
{
	if (lpDDPalette) {
		printOSD("  - Releasing palette\n");
		IDirectDrawPalette_Release(lpDDPalette);
		lpDDPalette = NULL;
	}

	if (lpDDSPrimary) {
		printOSD("  - Releasing primary surface\n");
		IDirectDrawSurface_Release(lpDDSPrimary);
		lpDDSPrimary = NULL;
	}

	if (lpDDSOffscreen) {
		printOSD("  - Releasing offscreen surface\n");
		IDirectDrawSurface_Release(lpDDSOffscreen);
		lpDDSOffscreen = NULL;
	}
}


//
// InitDirectInput() -- get DirectInput started
//
static BOOL InitDirectInput(void)
{
	HRESULT result;
	HRESULT (WINAPI *aDirectInputCreateA)(HINSTANCE, DWORD, LPDIRECTINPUTA *, LPUNKNOWN);
	DIPROPDWORD dipdw;

	if (bDInputInited) return FALSE;

	printOSD("Initializing DirectInput...\n");

	// load up the DirectInput DLL
	if (!hDInputDLL) {
		printOSD("  - Loading DINPUT.DLL\n");
		hDInputDLL = LoadLibrary("DINPUT.DLL");
		if (!hDInputDLL) {
			ShowErrorBox("Error loading DINPUT.DLL");
			return TRUE;
		}
	}

	// get the pointer to DirectInputCreate
	aDirectInputCreateA = (void *)GetProcAddress(hDInputDLL, "DirectInputCreateA");
	if (!aDirectInputCreateA) {
		ShowErrorBox("Error fetching DirectInputCreateA()");
		UninitDirectInput();
		return TRUE;
	}

	// create a new DirectInput object
	printOSD("  - Creating DirectDraw object\n");
	result = aDirectInputCreateA(hInstance, DIRECTINPUT_VERSION, &lpDI, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("DirectInputCreateA() failed", result);
		UninitDirectInput();
		return TRUE;
	}

	// create the keyboard device
	printOSD("  - Creating keyboard device\n");
	result = IDirectInput_CreateDevice(lpDI, &GUID_SysKeyboard, &lpDIDKeyboard, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed creating keyboard device", result);
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice_SetDataFormat(lpDIDKeyboard, &c_dfDIKeyboard);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting keyboard data format", result);
		UninitDirectInput();
		return TRUE;
	}

#ifdef USE_BUFFERED_KEYBOARD
	// set up keyboard for buffered IO
	dipdw.diph.dwSize = sizeof(DIPROPDWORD);
	dipdw.diph.dwHeaderSize = sizeof(DIPROPHEADER);
	dipdw.diph.dwObj = 0;
	dipdw.diph.dwHow = DIPH_DEVICE;
	dipdw.dwData = INPUT_BUFFER_SIZE;

	result = IDirectInputDevice_SetProperty(lpDIDKeyboard, DIPROP_BUFFERSIZE, &dipdw.diph);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting keyboard buffering", result);
		UninitDirectInput();
		return TRUE;
	}
#endif

	// create the mouse device
	printOSD("  - Creating mouse device\n");
	result = IDirectInput_CreateDevice(lpDI, &GUID_SysMouse, &lpDIDMouse, NULL);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed creating mouse device", result);
		UninitDirectInput();
		return TRUE;
	}

	result = IDirectInputDevice_SetDataFormat(lpDIDMouse, &c_dfDIMouse);
	if (result != DI_OK) {
		ShowDInputErrorBox("Failed setting mouse data format", result);
		UninitDirectInput();
		return TRUE;
	}


	bDInputInited = TRUE;
	return FALSE;
}


//
// UninitDirectInput() -- clean up DirectInput
//
static void UninitDirectInput(void)
{
	if (bDInputInited) printOSD("Uninitializing DirectInput...\n");

	AcquireInputDevices(0);

	if (lpDIDKeyboard) {
		printOSD("  - Releasing keyboard device\n");
		IDirectInputDevice_Release(lpDIDKeyboard);
		lpDIDKeyboard = NULL;
	}

	if (lpDIDMouse) {
		printOSD("  - Releasing mouse device\n");
		IDirectInputDevice_Release(lpDIDMouse);
		lpDIDMouse = NULL;
	}

	if (lpDI) {
		printOSD("  - Releasing DirectInput object\n");
		IDirectInput_Release(lpDI);
		lpDI = NULL;
	}

	if (hDInputDLL) {
		printOSD("  - Unloading DINPUT.DLL\n");
		FreeLibrary(hDInputDLL);
		hDInputDLL = NULL;
	}

	bDInputInited = FALSE;
}


//
// AcquireInputDevices() -- (un)acquires the input devices
//
static void AcquireInputDevices(char acquire)
{
	DWORD flags;

	if (!bDInputInited) return;

	if (acquire) {
		if (lpDIDKeyboard) {
			IDirectInputDevice_SetCooperativeLevel(lpDIDKeyboard, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			IDirectInputDevice_Acquire(lpDIDKeyboard);
		}

		if (lpDIDMouse) {
			if (fullscreen || mousegrab) flags = DISCL_FOREGROUND|DISCL_EXCLUSIVE;
			else flags = DISCL_FOREGROUND|DISCL_NONEXCLUSIVE;
			IDirectInputDevice_SetCooperativeLevel(lpDIDMouse, hWindow, flags);
			IDirectInputDevice_Acquire(lpDIDMouse);
		}
	} else {
		if (lpDIDKeyboard) {
			IDirectInputDevice_SetCooperativeLevel(lpDIDKeyboard, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			IDirectInputDevice_Unacquire(lpDIDKeyboard);
		}

		if (lpDIDMouse) {
			IDirectInputDevice_SetCooperativeLevel(lpDIDMouse, hWindow, DISCL_FOREGROUND|DISCL_NONEXCLUSIVE);
			IDirectInputDevice_Unacquire(lpDIDMouse);
		}
	}

	// reset mouse and keypresses
	mousex = 0;
	mousey = 0;
	mouseb = 0;

	memset(keystatus, 0, sizeof(keystatus));
	memset(keyfifo, 0, sizeof(keyfifo));
	keyfifoplc = 0;
	keyfifoend = 0;
	lastKeyDown = 0;
}


//
// ProcessInputDevices() -- processes the input devices
//
static void ProcessInputDevices(void)
{
	DIMOUSESTATE dims;
	DWORD i;
	HRESULT result;
#ifdef USE_BUFFERED_KEYBOARD
	DIDEVICEOBJECTDATA didod[INPUT_BUFFER_SIZE];
	DWORD dwElements;
#else
	BYTE *ks, *oks, a,b;
#endif

	if (lpDIDKeyboard) {
#ifdef USE_BUFFERED_KEYBOARD
		// buffered
		//
		dwElements = INPUT_BUFFER_SIZE;
		result = IDirectInputDevice_GetDeviceData(lpDIDKeyboard, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
		if (result == DIERR_INPUTLOST) {
			IDirectInputDevice_Acquire(lpDIDKeyboard);
			result = IDirectInputDevice_GetDeviceData(lpDIDKeyboard, sizeof(DIDEVICEOBJECTDATA), (LPDIDEVICEOBJECTDATA)&didod, &dwElements, 0);
		}

		if (result == DI_OK) {
			// process the key events
			for (i=0; i<dwElements; i++) {
				SetKey(didod[i].dwOfs, (didod[i].dwData & 0x80) == 0x80);

				if ((lastKeyDown == didod[i].dwOfs) && !(didod[i].dwData & 0x80))
					lastKeyDown = 0;
				else if (didod[i].dwData & 0x80) {
					lastKeyDown = didod[i].dwOfs;
					lastKeyTime = timeGetTime() + 250;
				}
			}

			if (lastKeyDown > 0) {
				if (timeGetTime() >= lastKeyTime) {
					SetKey(lastKeyDown, 1);
					lastKeyTime = timeGetTime() + 30;
				}
			}
//		} else {
//			printOSD("Failed querying keyboard device data: %s\n", GetDInputError(result));
		}
#else
		// unbuffered
		//
		ks = oks = keys;
		if (!keysofs) ks+=256;
		else oks+=256;
		keysofs = !keysofs;

		result = IDirectInputDevice_GetDeviceState(lpDIDKeyboard, sizeof(BYTE)*256, ks);
		if (result == DIERR_INPUTLOST) {
			IDirectInputDevice_Acquire(lpDIDKeyboard);
			result = IDirectInputDevice_GetDeviceState(lpDIDKeyboard, sizeof(BYTE)*256, ks);
		}

		if (result == DI_OK) {
			for (i=0; i<256; i++) {
				a = ks[i]&0x80;
				b = oks[i]&0x80;
				if (a == b) continue;	// no change in state

				// if the last key pressed was this current one,
				// and the key has just been released, stop repeating
				if ((i == lastKeyDown) && b && !a)
					lastKeyDown = 0;

				SetKey(i, a == 0x80);
				if (lastKeyDown) continue;
				if (a) {
					lastKeyDown = i;
					lastKeyTime = timeGetTime() + 250;
				}
			}

			if (lastKeyDown > 0) {
				if (timeGetTime() >= lastKeyTime) {
					SetKey(lastKeyDown, 1);
					lastKeyTime = timeGetTime() + 30;
				}
			}
		}
#endif
	}

	if (lpDIDMouse && moustat) {
		result = IDirectInputDevice_GetDeviceState(lpDIDMouse, sizeof(dims), &dims);
		if (result == DIERR_INPUTLOST) {
			IDirectInputDevice_Acquire(lpDIDMouse);
			result = IDirectInputDevice_GetDeviceState(lpDIDMouse, sizeof(dims), &dims);
		}

		if (result == DI_OK) {
			// update the mouse statuses
			//   5,4,3,2,1,0 = wheelup,wheeldown,thumb,middle,right,left
			mousex = (short)dims.lX;
			mousey = (short)dims.lY;
			mouseb = 0;
			if (dims.lZ < 0) mouseb |= 16;
			else if (dims.lZ > 0) mouseb |= 32;
			if (dims.rgbButtons[0] & 0x80) mouseb |= 1;
			if (dims.rgbButtons[1] & 0x80) mouseb |= 2;
			if (dims.rgbButtons[2] & 0x80) mouseb |= 4;
			if (dims.rgbButtons[3] & 0x80) mouseb |= 8;
//		} else {
//			printOSD("Failed querying mouse device data: %s\n", GetDInputError(result));
		}
	}
}


//
// UninitDIB() -- clean up the DIB renderer
//
static void UninitDIB(void)
{
	AcquireInputDevices(0);

	if (desktoppal)
		RestoreSystemColours();

	if (hPalette) {
		DeleteObject(hPalette);
		hPalette = NULL;
	}

	if (hDCSection) {
		DeleteDC(hDCSection);
		hDCSection = NULL;
	}

	if (hDIBSection) {
		DeleteObject(hDIBSection);
		hDIBSection = NULL;
	}

	if (hDC) {
		ReleaseDC(hWindow, hDC);
		hDC = NULL;
	}
}


//
// CreateAppWindow() -- create the application window
//
static BOOL CreateAppWindow(int width, int height, int fs)
{
	TCHAR msg[1024];
	RECT rect;
	int w, h, x, y, i;
	struct binfo {
		BITMAPINFOHEADER header;
		RGBQUAD colours[256];
	} dibsect;
	VOID *pixels;

	HRESULT result;
	DDSURFACEDESC ddsd;


	if (!hWindow) {
		hWindow = CreateWindowEx(
			0,
			"buildapp",
			apptitle,
			WINDOW_STYLE,
			CW_USEDEFAULT,
			CW_USEDEFAULT,
			320,
			200,
			NULL,
			NULL,
			hInstance,
			0);
		if (!hWindow) {
			ShowErrorBox("Unable to create window");
			return TRUE;
		}
	}

	// if we're coming out of fullscreen mode, reset the current display mode
	// so the window resizing doesn't get screwed up (???)
//	if (fullscreen && !fs) {
		ReleaseSurfaces();

		if (fullscreen) {
			// restore previous display mode and set to normal cooperative level
			result = IDirectDraw_RestoreDisplayMode(lpDD);
			if (result != DD_OK) {
				ShowDDrawErrorBox("Error restoring display mode", result);
				UninitDirectDraw();
				return TRUE;
			}

			result = IDirectDraw_SetCooperativeLevel(lpDD, hWindow, DDSCL_NORMAL);
			if (result != DD_OK) {
				ShowDDrawErrorBox("Error setting cooperative level", result);
				UninitDirectDraw();
				return TRUE;
			}
		}
//	}

	// resize the window
	if (!fs) {
		rect.left = 0;
		rect.top = 0;
		rect.right = width-1;
		rect.bottom = height-1;
		AdjustWindowRect(&rect, WINDOW_STYLE, FALSE);

		w = (rect.right - rect.left);
		h = (rect.bottom - rect.top);
		x = (desktopxdim - w) / 2;
		y = (desktopydim - h) / 2;

		SetWindowPos(hWindow, HWND_TOP, x, y, w, h, 0);
	} else {
		SetWindowPos(hWindow, HWND_TOP, 0, 0, width, height, 0);
	}

	SetFocus(hWindow);
	ShowWindow(hWindow, SW_SHOWNORMAL);
	UpdateWindow(hWindow);
	SetForegroundWindow(hWindow);

	// fullscreen?
	if (!fs) {
		// no, use DIB section

		if (!hDC) {
			hDC = GetDC(hWindow);
			if (!hDC) {
				ShowErrorBox("Error getting device context");
				return TRUE;
			}
		}

		if (hDCSection) {
			DeleteDC(hDCSection);
			hDCSection = NULL;
		}

		// destroy the previous DIB section if it existed
		if (hDIBSection) {
			DeleteObject(hDIBSection);
			hDIBSection = NULL;
		}

		// create the new DIB section
		memset(&dibsect, 0, sizeof(dibsect));
		dibsect.header.biSize = sizeof(dibsect.header);
		dibsect.header.biWidth = width;
		dibsect.header.biHeight = -height;
		dibsect.header.biPlanes = 1;
		dibsect.header.biBitCount = 8;
		dibsect.header.biCompression = BI_RGB;
		dibsect.header.biClrUsed = 256;
		dibsect.header.biClrImportant = 256;
		for (i=0; i<256; i++) {
			dibsect.colours[i].rgbBlue = curPalette[i].peBlue;
			dibsect.colours[i].rgbGreen = curPalette[i].peGreen;
			dibsect.colours[i].rgbRed = curPalette[i].peRed;
		}

		hDIBSection = CreateDIBSection(hDC, (BITMAPINFO *)&dibsect, DIB_RGB_COLORS, &pixels, NULL, 0);
		if (!hDIBSection) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;

			ShowErrorBox("Error creating DIB section");
			return TRUE;
		}

		memset(pixels, 0, width*height);

		// create a compatible memory DC
		hDCSection = CreateCompatibleDC(hDC);
		if (!hDCSection) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;

			ShowErrorBox("Error creating compatible DC");
			return TRUE;
		}

		// select the DIB section into the memory DC
		if (!SelectObject(hDCSection, hDIBSection)) {
			ReleaseDC(hWindow, hDC);
			hDC = NULL;
			DeleteDC(hDCSection);
			hDCSection = NULL;

			ShowErrorBox("Error creating compatible DC");
			return TRUE;
		}

		xres = width;
		yres = height;

		bytesperline = width;
		frameplace = (long)pixels;

		imageSize = bytesperline*yres;

		fullscreen = fs;
	} else {
		// yes, set up DirectDraw

		// clean up after the DIB renderer if it was being used
		UninitDIB();

		if (!bDDrawInited) {
			DestroyWindow(hWindow);
			hWindow = NULL;
			return TRUE;
		}

		ReleaseSurfaces();	// discard previous mode's surfaces

		// set exclusive cooperative level
		result = IDirectDraw_SetCooperativeLevel(lpDD, hWindow, DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error setting cooperative level", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDraw_RestoreDisplayMode(lpDD);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error restoring display mode", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDraw_SetDisplayMode(lpDD, width, height, 8);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Error setting display mode", result);
			UninitDirectDraw();
			return TRUE;
		}

		// now create the DirectDraw surfaces
		ZeroMemory(&ddsd, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE | DDSCAPS_FLIP | DDSCAPS_COMPLEX;
		ddsd.dwBackBufferCount = 2;

		printOSD("  - Creating primary surface\n");
		result = IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSPrimary, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating primary surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		ZeroMemory(&ddsd.ddsCaps, sizeof(ddsd.ddsCaps));
		ddsd.ddsCaps.dwCaps = DDSCAPS_BACKBUFFER;

		printOSD("  - Getting back buffer\n");
		result = IDirectDrawSurface_GetAttachedSurface(lpDDSPrimary, &ddsd.ddsCaps, &lpDDSBack);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure fetching back-buffer surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		ZeroMemory(&ddsd, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
		ddsd.dwWidth = width;
		ddsd.dwHeight = height;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN | DDSCAPS_SYSTEMMEMORY;

		printOSD("  - Creating offscreen surface\n");
		result = IDirectDraw_CreateSurface(lpDD, &ddsd, &lpDDSOffscreen, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating offscreen surface", result);
			UninitDirectDraw();
			return TRUE;
		}

		// attach a palette to the primary surface
		printOSD("  - Creating palette\n");
		result = IDirectDraw_CreatePalette(lpDD, DDPCAPS_8BIT | DDPCAPS_ALLOW256, curPalette, &lpDDPalette, NULL);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure creating palette", result);
			UninitDirectDraw();
			return TRUE;
		}

		result = IDirectDrawSurface_SetPalette(lpDDSPrimary, lpDDPalette);
		if (result != DD_OK) {
			ShowDDrawErrorBox("Failure setting palette", result);
			UninitDirectDraw();
			return TRUE;
		}

		/*
		memset(&ddsd, 0, sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);

		result = IDirectDrawSurface_Lock(lpDDSOffscreen, NULL, &ddsd, DDLOCK_NOSYSLOCK | DDLOCK_WAIT, NULL);
		if (result != DD_OK) {
			printOSD("Failed locking offscreen surface: %s\n", GetDDrawError(result));
			return -1;
		}
		*/

		xres = width;
		yres = height;

		//bytesperline = ddsd.lPitch;
		//frameplace = (long)ddsd.lpSurface;
		frameplace = 0;
		lockcount = 0;

		//imageSize = bytesperline*yres;

		fullscreen = fs;
	}
	modechange=1;

	return FALSE;
}


//
// DestroyAppWindow() -- destroys the application window
//
static void DestroyAppWindow(void)
{
	UninitDirectDraw();
	UninitDIB();

	if (hWindow) {
		DestroyWindow(hWindow);
		hWindow = NULL;
	}
}


//
// GetDDrawError() -- stinking huge list of error messages since MS didn't want to include
//   them in the DLL
//
static const char * GetDDrawError(HRESULT code)
{
	switch (code) {
		case DD_OK: return "DD_OK";
		case DDERR_ALREADYINITIALIZED: return "DDERR_ALREADYINITIALIZED";
		case DDERR_BLTFASTCANTCLIP: return "DDERR_BLTFASTCANTCLIP";
		case DDERR_CANNOTATTACHSURFACE: return "DDERR_CANNOTATTACHSURFACE";
		case DDERR_CANNOTDETACHSURFACE: return "DDERR_CANNOTDETACHSURFACE";
		case DDERR_CANTCREATEDC: return "DDERR_CANTCREATEDC";
		case DDERR_CANTDUPLICATE: return "DDERR_CANTDUPLICATE";
		case DDERR_CANTLOCKSURFACE: return "DDERR_CANTLOCKSURFACE";
		case DDERR_CANTPAGELOCK: return "DDERR_CANTPAGELOCK";
		case DDERR_CANTPAGEUNLOCK: return "DDERR_CANTPAGEUNLOCK";
		case DDERR_CLIPPERISUSINGHWND: return "DDERR_CLIPPERISUSINGHWND";
		case DDERR_COLORKEYNOTSET: return "DDERR_COLORKEYNOTSET";
		case DDERR_CURRENTLYNOTAVAIL: return "DDERR_CURRENTLYNOTAVAIL";
		case DDERR_DCALREADYCREATED: return "DDERR_DCALREADYCREATED";
		case DDERR_DEVICEDOESNTOWNSURFACE: return "DDERR_DEVICEDOESNTOWNSURFACE";
		case DDERR_DIRECTDRAWALREADYCREATED: return "DDERR_DIRECTDRAWALREADYCREATED";
		case DDERR_EXCEPTION: return "DDERR_EXCEPTION";
		case DDERR_EXCLUSIVEMODEALREADYSET: return "DDERR_EXCLUSIVEMODEALREADYSET";
		case DDERR_EXPIRED: return "DDERR_EXPIRED";
		case DDERR_GENERIC: return "DDERR_GENERIC";
		case DDERR_HEIGHTALIGN: return "DDERR_HEIGHTALIGN";
		case DDERR_HWNDALREADYSET: return "DDERR_HWNDALREADYSET";
		case DDERR_HWNDSUBCLASSED: return "DDERR_HWNDSUBCLASSED";
		case DDERR_IMPLICITLYCREATED: return "DDERR_IMPLICITLYCREATED";
		case DDERR_INCOMPATIBLEPRIMARY: return "DDERR_INCOMPATIBLEPRIMARY";
		case DDERR_INVALIDCAPS: return "DDERR_INVALIDCAPS";
		case DDERR_INVALIDCLIPLIST: return "DDERR_INVALIDCLIPLIST";
		case DDERR_INVALIDDIRECTDRAWGUID: return "DDERR_INVALIDDIRECTDRAWGUID";
		case DDERR_INVALIDMODE: return "DDERR_INVALIDMODE";
		case DDERR_INVALIDOBJECT: return "DDERR_INVALIDOBJECT";
		case DDERR_INVALIDPARAMS: return "DDERR_INVALIDPARAMS";
		case DDERR_INVALIDPIXELFORMAT: return "DDERR_INVALIDPIXELFORMAT";
		case DDERR_INVALIDPOSITION: return "DDERR_INVALIDPOSITION";
		case DDERR_INVALIDRECT: return "DDERR_INVALIDRECT";
		case DDERR_INVALIDSTREAM: return "DDERR_INVALIDSTREAM";
		case DDERR_INVALIDSURFACETYPE: return "DDERR_INVALIDSURFACETYPE";
		case DDERR_LOCKEDSURFACES: return "DDERR_LOCKEDSURFACES";
		case DDERR_MOREDATA: return "DDERR_MOREDATA";
		case DDERR_NO3D: return "DDERR_NO3D";
		case DDERR_NOALPHAHW: return "DDERR_NOALPHAHW";
		case DDERR_NOBLTHW: return "DDERR_NOBLTHW";
		case DDERR_NOCLIPLIST: return "DDERR_NOCLIPLIST";
		case DDERR_NOCLIPPERATTACHED: return "DDERR_NOCLIPPERATTACHED";
		case DDERR_NOCOLORCONVHW: return "DDERR_NOCOLORCONVHW";
		case DDERR_NOCOLORKEY: return "DDERR_NOCOLORKEY";
		case DDERR_NOCOLORKEYHW: return "DDERR_NOCOLORKEYHW";
		case DDERR_NOCOOPERATIVELEVELSET: return "DDERR_NOCOOPERATIVELEVELSET";
		case DDERR_NODC: return "DDERR_NODC";
		case DDERR_NODDROPSHW: return "DDERR_NODDROPSHW";
		case DDERR_NODIRECTDRAWHW: return "DDERR_NODIRECTDRAWHW";
		case DDERR_NODIRECTDRAWSUPPORT: return "DDERR_NODIRECTDRAWSUPPORT";
		case DDERR_NOEMULATION: return "DDERR_NOEMULATION";
		case DDERR_NOEXCLUSIVEMODE: return "DDERR_NOEXCLUSIVEMODE";
		case DDERR_NOFLIPHW: return "DDERR_NOFLIPHW";
		case DDERR_NOFOCUSWINDOW: return "DDERR_NOFOCUSWINDOW";
		case DDERR_NOGDI: return "DDERR_NOGDI";
		case DDERR_NOHWND: return "DDERR_NOHWND";
		case DDERR_NOMIPMAPHW: return "DDERR_NOMIPMAPHW";
		case DDERR_NOMIRRORHW: return "DDERR_NOMIRRORHW";
		case DDERR_NONONLOCALVIDMEM: return "DDERR_NONONLOCALVIDMEM";
		case DDERR_NOOPTIMIZEHW: return "DDERR_NOOPTIMIZEHW";
		case DDERR_NOOVERLAYDEST: return "DDERR_NOOVERLAYDEST";
		case DDERR_NOOVERLAYHW: return "DDERR_NOOVERLAYHW";
		case DDERR_NOPALETTEATTACHED: return "DDERR_NOPALETTEATTACHED";
		case DDERR_NOPALETTEHW: return "DDERR_NOPALETTEHW";
		case DDERR_NORASTEROPHW: return "DDERR_NORASTEROPHW";
		case DDERR_NOROTATIONHW: return "DDERR_NOROTATIONHW";
		case DDERR_NOSTRETCHHW: return "DDERR_NOSTRETCHHW";
		case DDERR_NOT4BITCOLOR: return "DDERR_NOT4BITCOLOR";
		case DDERR_NOT4BITCOLORINDEX: return "DDERR_NOT4BITCOLORINDEX";
		case DDERR_NOT8BITCOLOR: return "DDERR_NOT8BITCOLOR";
		case DDERR_NOTAOVERLAYSURFACE: return "DDERR_NOTAOVERLAYSURFACE";
		case DDERR_NOTEXTUREHW: return "DDERR_NOTEXTUREHW";
		case DDERR_NOTFLIPPABLE: return "DDERR_NOTFLIPPABLE";
		case DDERR_NOTFOUND: return "DDERR_NOTFOUND";
		case DDERR_NOTINITIALIZED: return "DDERR_NOTINITIALIZED";
		case DDERR_NOTLOADED: return "DDERR_NOTLOADED";
		case DDERR_NOTLOCKED: return "DDERR_NOTLOCKED";
		case DDERR_NOTPAGELOCKED: return "DDERR_NOTPAGELOCKED";
		case DDERR_NOTPALETTIZED: return "DDERR_NOTPALETTIZED";
		case DDERR_NOVSYNCHW: return "DDERR_NOVSYNCHW";
		case DDERR_NOZBUFFERHW: return "DDERR_NOZBUFFERHW";
		case DDERR_NOZOVERLAYHW: return "DDERR_NOZOVERLAYHW";
		case DDERR_OUTOFCAPS: return "DDERR_OUTOFCAPS";
		case DDERR_OUTOFMEMORY: return "DDERR_OUTOFMEMORY";
		case DDERR_OUTOFVIDEOMEMORY: return "DDERR_OUTOFVIDEOMEMORY";
		case DDERR_OVERLAPPINGRECTS: return "DDERR_OVERLAPPINGRECTS";
		case DDERR_OVERLAYCANTCLIP: return "DDERR_OVERLAYCANTCLIP";
		case DDERR_OVERLAYCOLORKEYONLYONEACTIVE: return "DDERR_OVERLAYCOLORKEYONLYONEACTIVE";
		case DDERR_OVERLAYNOTVISIBLE: return "DDERR_OVERLAYNOTVISIBLE";
		case DDERR_PALETTEBUSY: return "DDERR_PALETTEBUSY";
		case DDERR_PRIMARYSURFACEALREADYEXISTS: return "DDERR_PRIMARYSURFACEALREADYEXISTS";
		case DDERR_REGIONTOOSMALL: return "DDERR_REGIONTOOSMALL";
		case DDERR_SURFACEALREADYATTACHED: return "DDERR_SURFACEALREADYATTACHED";
		case DDERR_SURFACEALREADYDEPENDENT: return "DDERR_SURFACEALREADYDEPENDENT";
		case DDERR_SURFACEBUSY: return "DDERR_SURFACEBUSY";
		case DDERR_SURFACEISOBSCURED: return "DDERR_SURFACEISOBSCURED";
		case DDERR_SURFACELOST: return "DDERR_SURFACELOST";
		case DDERR_SURFACENOTATTACHED: return "DDERR_SURFACENOTATTACHED";
		case DDERR_TOOBIGHEIGHT: return "DDERR_TOOBIGHEIGHT";
		case DDERR_TOOBIGSIZE: return "DDERR_TOOBIGSIZE";
		case DDERR_TOOBIGWIDTH: return "DDERR_TOOBIGWIDTH";
		case DDERR_UNSUPPORTED: return "DDERR_UNSUPPORTED";
		case DDERR_UNSUPPORTEDFORMAT: return "DDERR_UNSUPPORTEDFORMAT";
		case DDERR_UNSUPPORTEDMASK: return "DDERR_UNSUPPORTEDMASK";
		case DDERR_UNSUPPORTEDMODE: return "DDERR_UNSUPPORTEDMODE";
		case DDERR_VERTICALBLANKINPROGRESS: return "DDERR_VERTICALBLANKINPROGRESS";
		case DDERR_VIDEONOTACTIVE: return "DDERR_VIDEONOTACTIVE";
		case DDERR_WASSTILLDRAWING: return "DDERR_WASSTILLDRAWING";
		case DDERR_WRONGMODE: return "DDERR_WRONGMODE";
		case DDERR_XALIGN: return "DDERR_XALIGN";
		default: break;
	}
	return "Unknown error";
}


//
// GetDInputError() -- stinking huge list of error messages since MS didn't want to include
//   them in the DLL
//
static const char * GetDInputError(HRESULT code)
{
	switch (code) {
		case DI_OK: return "DI_OK";
		case DI_BUFFEROVERFLOW: return "DI_BUFFEROVERFLOW";
		case DI_DOWNLOADSKIPPED: return "DI_DOWNLOADSKIPPED";
		case DI_EFFECTRESTARTED: return "DI_EFFECTRESTARTED";
		case DI_POLLEDDEVICE: return "DI_POLLEDDEVICE";
		case DI_TRUNCATED: return "DI_TRUNCATED";
		case DI_TRUNCATEDANDRESTARTED: return "DI_TRUNCATEDANDRESTARTED";
		case DIERR_ACQUIRED: return "DIERR_ACQUIRED";
		case DIERR_ALREADYINITIALIZED: return "DIERR_ALREADYINITIALIZED";
		case DIERR_BADDRIVERVER: return "DIERR_BADDRIVERVER";
		case DIERR_BETADIRECTINPUTVERSION: return "DIERR_BETADIRECTINPUTVERSION";
		case DIERR_DEVICEFULL: return "DIERR_DEVICEFULL";
		case DIERR_DEVICENOTREG: return "DIERR_DEVICENOTREG";
		case DIERR_EFFECTPLAYING: return "DIERR_EFFECTPLAYING";
		case DIERR_HASEFFECTS: return "DIERR_HASEFFECTS";
		case DIERR_GENERIC: return "DIERR_GENERIC";
		case DIERR_HANDLEEXISTS: return "DIERR_HANDLEEXISTS";
		case DIERR_INCOMPLETEEFFECT: return "DIERR_INCOMPLETEEFFECT";
		case DIERR_INPUTLOST: return "DIERR_INPUTLOST";
		case DIERR_INVALIDPARAM: return "DIERR_INVALIDPARAM";
		case DIERR_MOREDATA: return "DIERR_MOREDATA";
		case DIERR_NOAGGREGATION: return "DIERR_NOAGGREGATION";
		case DIERR_NOINTERFACE: return "DIERR_NOINTERFACE";
		case DIERR_NOTACQUIRED: return "DIERR_NOTACQUIRED";
		case DIERR_NOTBUFFERED: return "DIERR_NOTBUFFERED";
		case DIERR_NOTDOWNLOADED: return "DIERR_NOTDOWNLOADED";
		case DIERR_NOTEXCLUSIVEACQUIRED: return "DIERR_NOTEXCLUSIVEACQUIRED";
		case DIERR_NOTFOUND: return "DIERR_NOTFOUND";
		case DIERR_NOTINITIALIZED: return "DIERR_NOTINITIALIZED";
		case DIERR_OLDDIRECTINPUTVERSION: return "DIERR_OLDDIRECTINPUTVERSION";
		case DIERR_OUTOFMEMORY: return "DIERR_OUTOFMEMORY";
		case DIERR_UNSUPPORTED: return "DIERR_UNSUPPORTED";
		case E_PENDING: return "E_PENDING";
		default: break;
	}
	return "Unknown error";
}


//
// GetWindowsErrorMsg() -- gives a pointer to a static buffer containing the Windows error message
//
static const LPTSTR GetWindowsErrorMsg(DWORD code)
{
	static TCHAR lpMsgBuf[1024];

	FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
		NULL, code,
		MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
		(LPTSTR)lpMsgBuf, 1024, NULL);

	return lpMsgBuf;
}


//
// SaveSystemColours() -- save the Windows-reserved colours
//
static void SaveSystemColours(void)
{
	int i;

	if (system_colours_saved) return;

	for (i=0; i<NUM_SYS_COLOURS; i++)
		syscolours[i] = GetSysColor(syscolouridx[i]);

	system_colours_saved = 1;
}


//
// RestoreSystemColours() -- restore the Windows-reserved colours
//
static void RestoreSystemColours(void)
{
	if (!system_colours_saved) return;

	SetSystemPaletteUse(hDC, SYSPAL_STATIC);
	SetSysColors(NUM_SYS_COLOURS, syscolouridx, syscolours);
}

