// Creative VOC loader
// Build DirectSound system
// by Jonathon Fowler (jonof@edgenetwk.com)


#include <string.h>
#include <stdio.h>
#include "cache1d.h"
#include "compat.h"
#include "buildsound.h"


#define REGRESSTC(tc) (256000000/(65536-(tc)))
#define REGRESSSR(sr) (1000000/(256-(sr)))

//
// ReadVOCInfo() -- read header information and calculate buffer size
//    required to load sound into memory
//
//   Returns 0 on success. snd->soundlength is the required buffer length.
//   Returns -1 if not a VOC file.
//   Returns -2 if version checksum was incorrect.
//   Returns -3 if data format is unsupported.
//
int ReadVOCInfo(int fh, SoundInfo *snd)
{
	char sig[0x14];
	short offset;
	short version;
	short version2;
	char blocktype=0;
	int  blocklen=0;
	unsigned char blockprop[12];


	memset(snd, 0, sizeof(SoundInfo));

	klseek(fh, 0, SEEK_SET);

	kread(fh, sig, 0x14);
	if (memcmp(sig, "Creative Voice File\x1A", 0x14))
		return -1;

	kread(fh, &offset, 2);
	kread(fh, &version, 2);
	kread(fh, &version2, 2);

	if (~version + 0x1234 != version2)
		return -2;

	while (kread(fh, &blocktype, 1) == 1) {
		if (blocktype == 0)
			break;

		kread(fh, &blocklen, 3);

		switch (blocktype) {
			case 0: /* terminator. we should have broken out above... */
				break;

			case 1: /* sound data begin block */
				kread(fh, blockprop, 2);
				if (!snd->samplerate)
					snd->samplerate = REGRESSSR(blockprop[0]);
				if (blockprop[1] != 0) {
					/* only 8-bit files please */
					return -3;
				}
				if (!snd->channels) snd->channels = 1;
				snd->bitspersample = 8;
				snd->soundlength += blocklen-2;
				klseek(fh, blocklen-2, SEEK_CUR);
				break;

			case 2: /* sound continue */
				snd->soundlength += blocklen;
				klseek(fh, blocklen, SEEK_CUR);
				break;

#if 0
			case 3: /* silence */
				kread(fh, blockprop, 3);
				/*
				length = blockprop[0] | (blockprop[1] << 8)
				sample rate = REGRESSSR(blockprop[2]))
				*/
				break;

			case 4:	/* marker */
				kread(fh, &blockprop, 2);
				/*
				id = blockprop[0] | (blockprop[1] << 8))
				*/
				break;

			case 5: /* ASCII data */
				klseek(fh, blocklen, SEEK_CUR);
				/*
				asciiz string
				*/
				break;

			case 6: /* repeat */
				kread(fh, blockprop, 2);
				/*
				num repetitions = (blockprop[0] | (blockprop[1] << 8)) - 1
				*/
				break;

			case 7: /* end repeat */
				break;
#endif

			case 8: /* sound attribute extension block */
				kread(fh, blockprop, 4);
				snd->samplerate = REGRESSTC(blockprop[0] | (blockprop[1] << 8));
				snd->bitspersample = 8;
				if (blockprop[3] == 1) {
					snd->samplerate >>= 1;
					snd->channels = 2;
				} else
					snd->channels = 1;
				if (blockprop[2] != 0) {
					/* only 8-bit files please */
					return -3;
				}
				/* a block 1 follows */
				break;

			case 9: /* sound data format 3 */
				kread(fh, blockprop, 12);
				snd->samplerate = *((long*)(blockprop));
				snd->bitspersample = blockprop[4];
				snd->channels = blockprop[5];
				if ((blockprop[6] | (blockprop[7] << 8)) != 0 &&
				    (blockprop[6] | (blockprop[7] << 8)) != 4) {
					/* only PCM please */
					return -3;
				}
				snd->soundlength += blocklen-12;
				klseek(fh, blocklen-12, SEEK_CUR);
				break;

			default:
				klseek(fh, blocklen, SEEK_CUR);
				break;
		}
	}

	return 0;
}


//
// ReadVOCData -- read the VOC sound data into the preallocated buffer
//   of length indicated by bufferlen
//
int ReadVOCData(int fh, char *data, int bufferlen)
{
	short offset;
	char blocktype=0;
	int  blocklen=0, br;
	unsigned char blockprop[12];

	klseek(fh, 0x14, SEEK_SET);

	kread(fh, &offset, 2);
	klseek(fh, offset, SEEK_SET);

	while (bufferlen > 0 && kread(fh, &blocktype, 1) == 1) {
		if (blocktype == 0)
			break;

		kread(fh, &blocklen, 3);

		switch (blocktype) {
			case 0: /* terminator. we should have broken out above... */
				break;

			case 1: /* sound data */
				kread(fh, blockprop, 2);

				br = kread(fh, data, min(blocklen-2, bufferlen));
				bufferlen -= br;
				data += br;
				break;

			case 2: /* sound continue */
				br = kread(fh, data, min(blocklen, bufferlen));
				bufferlen -= br;
				data += br;
				break;

			case 9: /* sound data format 3 */
				kread(fh, blockprop, 12);

				br = kread(fh, data, min(blocklen-12, bufferlen));
				bufferlen -= br;
				data += br;
				break;

			default:
				klseek(fh, blocklen, SEEK_CUR);
				break;
		}
	}

	return 0;
}
