// On-screen Display (ie. console)
// for the Build Engine
// by Jonathon Fowler (jonof@edgenetwk.com)

#include <stdio.h>
#include <string.h>
#include <stdarg.h>
#include "build.h"
#include "osd.h"


#define BACKSCROLLSIZE 2048	// bytes

static char osdtext[BACKSCROLLSIZE+1], *osdptr=osdtext;
int  osdlines=8;			// # lines of the buffer that are visible
int  osdcols=60;			// # columns of the buffer that are visible
char osdvisible=0;			// onscreen display visible?
int  osdhead=0; 			// topmost visible line number
char *osdlogfile="console.txt";		// file to log to
static int white=-1;			// colour of white
static char osdinited=0;		// text buffer initialised?
static FILE *osdlog=NULL;		// log filehandle


//
// drawOSD() -- Draw the onscreen display
//
void drawOSD(void)
{
	int i,j,k,chars,lines,lines2;
	char *p,*q,*b;

	if (!osdvisible || !osdinited) return;

	if (white<0) {
		// find the palette index closest to white
		k=0;
		for(i=0;i<256;i++)
		{
			j = ((int)palette[i*3])+((int)palette[i*3+1])+((int)palette[i*3+2]);
			if (j > k) { k = j; white = i; }
		}
	}

	if (osdptr == osdtext) return;	// empty buffer, nothing to print
	p = osdptr-1;
	b = (char *)alloca(osdcols+1);

	if (osdhead > 0) {
		i=osdhead;
		if (*p == '\n') p--;
		while (i>0 && p != osdtext) {
			if (*p == '\n') i--;
			p--;
		}
		if (p == osdtext) {
			// line counter wants to go past the top line
			// of the buffer so pull it back
			osdhead -= i;
		}
	}

	begindrawing();
	for (i=osdlines;i>0;) {
		chars=0;
		if (*p == '\n') { p--; }
		while (p!=osdtext && *p != '\n') {
			p--;
			chars++;
		}
		if (p == osdtext) i=0;	// break loop on end of this cycle

		q=p+1;
		
		// skip the lines that scrolled off the top of the osd
		while ((chars+osdcols)/osdcols > i) {
			q+=osdcols;
			chars-=osdcols;
		}

		lines=(chars+osdcols)/osdcols;
		lines2=0;

		if (chars==0) i--;
		while (chars > 0) {
			if (chars>osdcols) k=osdcols;
			else k=chars;
			for (j=0;j<k;j++) b[j]=q[j];
			b[j]=0;
			q += k;
			chars -= k;

			printext256(4,4+((i-lines+lines2)*8), white, -1, b, 0);
			lines2++;
		}
		i-=lines;
	}

	if (osdhead > 0)
		printext256(4,4+(osdlines*8), white, -1, "  (--more--)",0);
	enddrawing();
}


//
// printOSD() -- Print a string to the onscreen display
//   and write it to the log file
//
void printOSD(const char *fmt, ...)
{
	int i=0,j,chars,maxchars;
	char ch, *p;
	va_list va;

	if (!osdlog && osdlogfile)
		osdlog = fopen(osdlogfile, "w");

	if (!osdinited) {
		memset(osdtext, 0, BACKSCROLLSIZE);
		osdtext[BACKSCROLLSIZE]=0;
		osdptr=osdtext;
		osdinited=1;
	}
	
	while (1) {
		va_start(va, fmt);
		maxchars = osdtext+BACKSCROLLSIZE-osdptr;
		chars = vsnprintf(osdptr, maxchars, fmt, va);
		va_end(va);

		//printf("maxchars=%d chars=%d ",maxchars,chars);

		if (maxchars==BACKSCROLLSIZE) {
			if (chars < 0 || chars > maxchars) {
				chars = maxchars;
		//		printf("line truncated\n");
			} else {
		//		printf("full line outputted\n");
			}
			break;
		} else {
			// maxchars is less than the full buffer
			if (chars < 0 || chars > maxchars) {
				// string we're trying to print is longer than what's left
				// so we need to push out the first line of the buffer
				// to try and make room
				p = memchr(osdtext, '\n', osdptr-osdtext);
				if (!p) {
					osdptr = osdtext;
		//			printf("no cr found, so discarding buffer\n");
				} else {
					memmove(osdtext, p+1, osdptr-p-1);
					osdptr -= (p+1-osdtext);
		//			printf("discarding a line\n");
				}
			} else {
				// string fitted in the space
		//		printf("full line outputted\n");
				break;
			}
		}
	}

	if (osdlog)
		fwrite(osdptr, chars, 1, osdlog);
	//printf("***\n%s\n***\n",osdtext);

	osdptr += chars;
}

