//
// kgroup.c
//
// Small utility which collects many files into 1 uncompressed file (a group file)
//
// Original utility by Ken Silverman
// Gold version 0.0 by Mathieu Olivier <elric@mcm.net>
//

// "Build Engine & Tools" Copyright (c) 1993-1997 Ken Silverman
// Ken Silverman's official web site: "http://www.advsys.net/ken"
// See the included license file "BUILDLIC.TXT" for license info.
// This file has been modified from Ken Silverman's original release


#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* --------------------------------- */
/* --- Plateform dependent stuff --- */
/* --------------------------------- */

// Win32
#ifdef WIN32
   #include <io.h>

// Unix (_SHOULD_ works on most Unix OS (it seems to be a POSIX standard)
// Tested on a Linux Redhat 6.2
#elif defined (unix)
   #include <fcntl.h>
   #include <unistd.h>

#else
   #error "Unsupported operating system"
#endif

// If the "min" macro hasn't been previously define, do it
#ifndef min
   // Return the minimum value of 2 values
   #define min(a,b) (((a) < (b)) ? (a) : (b))
#endif


/* ----------------- */
/* --- Constants --- */
/* ----------------- */

// Maximum number of files in a group file: (taken from the original "kextract.c")
#define MAX_NB_FILES 4096


/* ------------- */
/* --- Types --- */
/* ------------- */

// Boolean
#ifndef __cplusplus  // Standard C++ has already this "bool"
   typedef enum { false, true } bool;
#endif

// Properties of a grouped file
typedef struct {
   char         Name [13];   // 8.3 format, zero-terminated
   char         Path [128];  // Path (can contain directories)
   unsigned int Length;
} FileProperties_t;


/* ------------------------- */
/* --- Globals variables --- */
/* ------------------------- */

// Properties of the grouped files
static FileProperties_t FilesProperties [MAX_NB_FILES];

// Total number of files to put into the group file
static unsigned int NbFiles = 0;


/* ----------------- */
/* --- Functions --- */
/* ----------------- */

/* ------ Prototypes ------ */

// Register the files to add into the group file
static void RegisterFiles (const char* FileSpec);


/* ------ Implementations ------ */

/*
====================
main

Main procedure
====================
*/
int main (int ArgC, char* ArgV [])
{
   // Variables
   const char* FileName;
   FILE *GroupFile, *CrtFile;
   unsigned char Buffer [64 * 1024];
   unsigned int Ind, Ind2, Ind3;
   unsigned int DataSize;

   // Header
   printf ("\n"
           "KGROUP Gold version 0.0 by Mathieu Olivier\n"
           "  Original utility by Kenneth Silverman\n"
           "==========================================\n\n"
          );

   // Check arguments
   if (ArgC < 3)
   {
      printf ("KGROUP [grouped file][@file or filespec...]\n"
              "   This program collects many files into 1 big uncompressed file called a\n"
              "   group file\n"
              "   Ex: kgroup stuff.dat *.art *.map *.k?? palette.dat tables.dat\n"
              "      (stuff.dat is the group file, the rest are the files to add)\n");
      return -1;
   }

   // Register files
   NbFiles = 0;
   for (Ind = 2; Ind < (unsigned int)ArgC; Ind++)
   {
      FileName = ArgV[Ind];

      // If it's the name of a list of files
      if (FileName[0] == '@')
      {
         // Open the list of files
         CrtFile = fopen (&FileName[1], "rt");
         if (CrtFile == NULL)
         {
            printf ("Error: can't open files list %s\n", &FileName[1]);
            continue;
         }

         // Register the files in the list for extraction
         while (fgets ((char*)Buffer, sizeof (Buffer), CrtFile) != NULL)
         {
            Ind2 = 0;

            // Skip spaces and other strange chars
            while (Buffer[Ind2] != '\0' && Buffer[Ind2] <= ' ')
               Ind2++;

            // Register all files named in "Buffer"
            while (Buffer[Ind2] != '\0')
            {
               // Get the next file name
               Ind3 = Ind2;
               while (Buffer[Ind3] != '\0' && Buffer[Ind3] > ' ')
                  Ind3++;

               // Register it
               Buffer[Ind3] = '\0';
               RegisterFiles ((char*)&Buffer[Ind2]);
               Ind2 = Ind3 + 1;

               // Skip spaces and other strange chars
               while (Buffer[Ind2] != '\0' && Buffer[Ind2] <= ' ')
                  Ind2++;
            }
         }
         fclose (CrtFile);
      }

      // Else, register the chosen file
      else
         RegisterFiles (FileName);
   }

   // If there's no file to put into the group file, we quit
   if (NbFiles == 0)
   {
      printf ("No files found to create the group file\n");
      return 0;
   }

   // Open the group file
   FileName = ArgV[1];
   GroupFile = fopen (FileName, "wb");
   if (GroupFile == NULL)
   {
      printf ("Error: %s could not be opened\n", FileName);
      return -1;
   }

   // Write the header
   fwrite ("KenSilverman", 1, 12, GroupFile);
   fwrite (&NbFiles, 4, 1, GroupFile);  // Works only on a little-endian processor (Intel)
   for (Ind = 0; Ind < NbFiles; Ind++)
   {
      fwrite (FilesProperties[Ind].Name, 1, 12, GroupFile);
      fwrite (&FilesProperties[Ind].Length, 4, 1, GroupFile);  // Works only on a little-endian processor (Intel)
   }

   // Write the data
   for (Ind = 0; Ind < NbFiles; Ind++)
   {
      printf ("Adding %s... ", FilesProperties[Ind].Name);
      fflush (stdout);

      // Open the current file
      CrtFile = fopen (FilesProperties[Ind].Path, "rb");
      if (CrtFile == NULL)
      {
         printf ("error (can't open file)\n");
         continue;
      }

      // Dump its content in the group file
      for (Ind2 = 0; Ind2 < FilesProperties[Ind].Length; Ind2 += sizeof (Buffer))
      {
         // Read the data from the source file
         DataSize = min (FilesProperties[Ind].Length - Ind2, sizeof (Buffer));
         if (fread (Buffer, 1, DataSize, CrtFile) != DataSize)
         {
            printf ("error (can't read the whole file)\n");
            break;
         }

         // Write the data to the group file
         if (fwrite (Buffer, 1, DataSize, GroupFile) != DataSize)
         {
            printf ("error (can't write into the group file)\n");
            fclose (CrtFile);
            fclose (GroupFile);
            return -1;
         }
      }
      printf ("done\n");
      fclose (CrtFile);
   }

   // Finalization
   fclose (GroupFile);
   printf ("%s closed\n\n", ArgV[1]);
   return 0;
}


/*
====================
RegisterFiles

Register the files to add into the group file
====================
*/
static void RegisterFiles (const char* FileSpec)
{
/* ------------ Win32 ------------ */
#ifdef WIN32
   // Variables
   struct _finddata_t FileData;
   unsigned int PathLength;
   long Handle;

   // Compute the path length
   for (PathLength = 0; FileSpec[PathLength] != '\0'; PathLength++);
   while (FileSpec[PathLength] != '\\' && PathLength > 0)
      PathLength--;
   if (FileSpec[PathLength] == '\\')
      PathLength++;

   // Find the selected files
   Handle = _findfirst (FileSpec, &FileData);
   if (Handle == -1)
      return;
   do
   {
      // Check the maximal number of files
      if (NbFiles == MAX_NB_FILES)
      {
         printf ("FATAL ERROR: TOO MANY FILES SELECTED! (MAX is 4096)\n");
         exit (EXIT_FAILURE);
      }

      // Save the file infos
      strncpy (FilesProperties[NbFiles].Name, FileData.name, 12);
      FilesProperties[NbFiles].Name[12] = 0;
      FilesProperties[NbFiles].Length = FileData.size;

      // Save the path
      strncpy (FilesProperties[NbFiles].Path, FileSpec, PathLength);
      strncpy (&FilesProperties[NbFiles].Path[PathLength], FileData.name, 127 - PathLength);
      FilesProperties[NbFiles].Path[127] = '\0';

      NbFiles++;

   } while (_findnext (Handle, &FileData) == 0);

/* ------------ Unix ------------ */
#elif defined (unix)
   // Variables
   int File;
   const char* FileName;

   // Get the file length
   File = open (FileSpec, O_RDONLY);
   if (File == -1)
   {
      printf ("Error: can't find file %s\n", FileSpec);
      return;
   }
   FilesProperties[NbFiles].Length = lseek (File, 0, SEEK_END);
   close (File);
   if (FilesProperties[NbFiles].Length == (unsigned int)-1)
   {
      printf ("Error: can't get the length of file %s\n", FileSpec);
      return;
   }
   
   // Extract its name from its path
   strncpy (FilesProperties[NbFiles].Path, FileSpec, 127);
   FilesProperties[NbFiles].Path[127] = '\0';
   FileName = strrchr (FileSpec, '/');
   if (FileName == NULL)
      FileName = FileSpec;
   strncpy (FilesProperties[NbFiles].Name, FileName, 12);
   FilesProperties[NbFiles].Name[12] = '\0';

   NbFiles++;

#endif
}
