//
// kextract.c
//
// Small utility which extracts files from a group file
//
// Original utility by Ken Silverman
// Gold version 0.0 by Mathieu Olivier <elric@mcm.net>
//

// "Build Engine & Tools" Copyright (c) 1993-1997 Ken Silverman
// Ken Silverman's official web site: "http://www.advsys.net/ken"
// See the included license file "BUILDLIC.TXT" for license info.
// This file has been modified from Ken Silverman's original release


#include <stdio.h>
#include <string.h>


/* ----------------- */
/* --- Constants --- */
/* ----------------- */

// Maximum number of files in a group file: (taken from the original "kextract.c")
#define MAX_NB_FILES 4096


/* ------------- */
/* --- Types --- */
/* ------------- */

// Boolean
#ifndef __cplusplus  // Standard C++ has already this "bool"
   typedef enum { false, true } bool;
#endif

// Properties of a grouped file
typedef struct {
   char         Name [13];            // 8.3 format, zero-terminated
   bool         MarkedForExtraction;
   unsigned int Length;
   unsigned int Offset;
} FileProperties_t;


/* -------------- */
/* --- Macros --- */
/* -------------- */

// If the OS don't have the "min" macro, define it
#ifndef min
   // Return the minimum value of 2 values
   #define min(a,b) (((a) < (b)) ? (a) : (b))
#endif


/* ------------------------- */
/* --- Globals variables --- */
/* ------------------------- */

// Properties of the grouped files
static FileProperties_t FilesProperties [MAX_NB_FILES];

// Total number of files in the group file, and number of files to extract
static unsigned int NbFiles, NbFilesToExtract;


/* ----------------- */
/* --- Functions --- */
/* ----------------- */

/* ------ Prototypes ------ */

// Mark the files to extract
static void MarkFiles (const char* FileSpec);


/* ------ Implementations ------ */

/*
====================
main

Main procedure
====================
*/
int main (int ArgC, char* ArgV [])
{
   // Variables
   FILE *GroupFile, *CrtFile;
   unsigned char Buffer [64 * 1024];
   unsigned int Ind, Ind2, Ind3;
   unsigned int DataSize;
   unsigned int CrtOffset;
   const char* FileName;

   // Header
   printf ("\n"
           "KEXTRACT Gold version 0.0 by Mathieu Olivier\n"
           "   Original utility by Kenneth Silverman\n"
           "============================================\n\n"
          );

   // Check arguments
   if (ArgC < 3)
   {
      printf ("Syntax: KEXTRACT [grouped file][@file or filespec...]\n"
              "   This program extracts files from a previously grouped group file.\n"
              "   You can extract files using the ? and * wildcards.\n"
              "   Ex: kextract stuff.dat tiles000.art nukeland.map palette.dat\n"
              "      (stuff.dat is the group file, the rest are the files to extract)\n"
             );
      return -1;
   }

   // Open the group file
   FileName = ArgV[1];
   GroupFile = fopen (FileName, "rb");
   if (GroupFile == NULL)
   {
      printf ("Error: %s could not be opened\n", FileName);
      return -1;
   }

   // Check the group file tag (12 first bytes must be "KenSilverman")
   if (fread (Buffer, 1, 16, GroupFile) != 16 ||
       strncmp ("KenSilverman", (char*)Buffer, 12) != 0)
   {
      fclose (GroupFile);
      printf ("Error: %s not a valid group file\n", FileName);
      return -1;
   }

   // Read the grouped files properties
   NbFiles = Buffer[12] | Buffer[13] << 8 | Buffer[14] << 16 | Buffer[15] << 24;
   CrtOffset = 16 + NbFiles * 16;  // base offset, just after the header
   for (Ind = 0; Ind < NbFiles; Ind++)
   {
      // Extract the name of the file
      fread (FilesProperties[Ind].Name, 1, 12, GroupFile);
      FilesProperties[Ind].Name[12] = '\0';  // "close" the name manually if it was too long (12 chars)

      // Extract the length of the file
      fread (Buffer, 1, 4, GroupFile);
      FilesProperties[Ind].Length = Buffer[0] | Buffer[1] << 8 | Buffer[2] << 16 | Buffer[3] << 24;

      // Assign the current offset to the file
      FilesProperties[Ind].Offset = CrtOffset;

      // Unmark file for extraction
      FilesProperties[Ind].MarkedForExtraction = false;

      // Compute the new current offset
      CrtOffset += FilesProperties[Ind].Length;
   }

   // Mark files for extraction
   NbFilesToExtract = 0;
   for (Ind = 2; Ind < (unsigned int)ArgC; Ind++)
   {
      FileName = ArgV[Ind];

      // If it's the name of a list of files
      if (FileName[0] == '@')
      {
         // Open the list of files
         CrtFile = fopen (&FileName[1], "rt");
         if (CrtFile == NULL)
         {
            printf ("Error: can't open files list %s\n", &FileName[1]);
            continue;
         }

         // Mark the files in the list for extraction
         while (fgets ((char*)Buffer, sizeof (Buffer), CrtFile) != NULL)
         {
            Ind2 = 0;

            // Skip spaces and other strange chars
            while (Buffer[Ind2] != '\0' && Buffer[Ind2] <= ' ')
               Ind2++;

            // Mark all files named in "Buffer"
            while (Buffer[Ind2] != '\0')
            {
               // Get the next file name
               Ind3 = Ind2;
               while (Buffer[Ind3] != '\0' && Buffer[Ind3] > ' ')
                  Ind3++;

               // Mark it
               Buffer[Ind3] = '\0';
               MarkFiles ((char*)&Buffer[Ind2]);
               Ind2 = Ind3 + 1;

               // Skip spaces and other strange chars
               while (Buffer[Ind2] != '\0' && Buffer[Ind2] <= ' ')
                  Ind2++;
            }
         }
         fclose (CrtFile);
      }

      // Else, mark the chosen file
      else
         MarkFiles (FileName);
   }

   // If there's no file to extract, we quit
   if (NbFilesToExtract == 0)
   {
      fclose (GroupFile);
      printf ("No files found in group file with those names\n");
      return 0;
   }

   // Extract the chosen files
   for (Ind = 0; Ind < NbFiles; Ind++)
   {
      // If the file doesn't have to be extracted, skip it
      if (FilesProperties[Ind].MarkedForExtraction == 0)
         continue;

      // Seek the file in the group file
      if (fseek (GroupFile, FilesProperties[Ind].Offset, SEEK_SET) != 0)
      {
         printf ("Error: %s has an invalid offset\n", FilesProperties[Ind].Name);
         continue;
      }

      // Create the file
      CrtFile = fopen (FilesProperties[Ind].Name, "wb");
      if (CrtFile == NULL)
      {
         printf ("Error: Could not create %s\n", FilesProperties[Ind].Name);
         continue;
      }

      // Fill it
      printf ("Extracting %s... ", FilesProperties[Ind].Name);
      fflush (stdout);
      for (Ind2 = 0; Ind2 < FilesProperties[Ind].Length; Ind2 += sizeof (Buffer))
      {
         // Read the data from the group file
         DataSize = min (FilesProperties[Ind].Length - Ind2, sizeof (Buffer));
         if (fread (Buffer, 1, DataSize, GroupFile) != DataSize)
         {
            printf ("error (can't read the whole file)\n");
            break;
         }

         // Write the data to the new created file
         if (fwrite (Buffer, 1, DataSize, CrtFile) != DataSize)
         {
            printf ("error (can't write the whole file)\n");
            fclose (CrtFile);
            fclose (GroupFile);
            return -1;
         }
      }
      printf ("done\n");
      fclose (CrtFile);
   }

   // Finalization
   fclose (GroupFile);
   printf ("\n");
   return 0;
}


/*
====================
MarkFiles

Mark the files to extract
====================
*/
static void MarkFiles (const char* FileSpec)
{
   // Variables
   unsigned int Ind, Ind2, Ind3;
   char LocalFileSpec [12], LocalFileName [12];
   char SpecChar, NameChar;
   bool Matched;

   // Put the FileSpec in LocalFileSpec with a format on 12 characters ("<prefix>_SPACES_.<suffix>")
   Ind2 = 0;
   for (Ind = 0; FileSpec[Ind] != '\0'; Ind++)
   {
      // if we find the suffix, move forward to its place in LocalFileSpec
      // and fill the space between prefic and suffix with spaces
      if (FileSpec[Ind] == '.')
         while (Ind2 < 8)
            LocalFileSpec[Ind2++] = ' ';

      // Copy the current character in LocalFileSpec
      LocalFileSpec[Ind2++] = FileSpec[Ind];
   }

   // Match LocalFileSpec with FileName for each file in the group file
   for (Ind = 0; Ind < NbFiles; Ind++)
   {
      // Put the FileProperties[Ind].Name in LocalFileName with the same format on 12 characters
      Ind2 = 0;
      for (Ind3 = 0; FilesProperties[Ind].Name[Ind3] != '\0'; Ind3++)
      {
         // if we find the suffix, move forward to its place in LocalFileSpec
         // and fill the space between prefic and suffix with spaces
         if (FilesProperties[Ind].Name[Ind3] == '.')
            while (Ind2 < 8)
               LocalFileName[Ind2++] = ' ';

         // Copy the current character in LocalFileName
         LocalFileName[Ind2++] = FilesProperties[Ind].Name[Ind3];
      }

      // Matching...
      Matched = true;
      for (Ind2 = 0; Ind2 < sizeof (LocalFileSpec); Ind2++)
      {
         // Extract the 2 current characters and "upcase" them
         SpecChar = LocalFileSpec[Ind2];
         if (SpecChar >= 'a' && SpecChar <= 'z')
            SpecChar -= 32;
         NameChar = LocalFileName[Ind2];
         if (NameChar >= 'a' && NameChar <= 'z')
            NameChar -= 32;

         // If it's the '*' joker
         if (SpecChar == '*')
         {
            // If we are in the suffix part, that's OK
            if (Ind2 >= 8)
               break;

            // Else, go to the suffix part
            Ind2 = 8;
         }

         // Else, if the 2 characters can't be matched, we skip this file
         else if (SpecChar != '?' && SpecChar != NameChar)
         {
            Matched = false;
            break;
         }
      }

      // If this file must be extracted, mark it
      if (Matched)
      {
         FilesProperties[Ind].MarkedForExtraction = true;
         NbFilesToExtract++;
      }
   }
}

